package com.transsion.palmsdk.http;

import android.os.Bundle;
import android.text.TextUtils;
import android.util.Log;

import com.transsion.gslb.GslbSdk;
import com.transsion.palmsdk.PalmConstants;
import com.transsion.palmsdk.data.PalmTokenInfo;
import com.transsion.palmsdk.exception.PalmServerException;
import com.transsion.palmsdk.util.PalmUtil;
import com.transsion.palmsdk.util.ThreadExecutors;

import org.json.JSONObject;

import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.UnknownHostException;
import java.nio.charset.Charset;
import java.util.Map;

import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;

public class PalmHttpClient {

    private static PalmHttpClient sPalmHttpClient;
    private static javax.net.ssl.SSLSocketFactory sslSocketFactory = null;

    private final int serverMode;

    private static final String METHOD_GET = "GET";
    private static final String METHOD_POST = "POST";

    public final static String PATH_AUTH_CODE = "/oauth/authorize";
    public final static String PATH_AUTH_CODE_CONFIRM = "/oauth/authorize_confirm";
    public final static String PATH_ACCESS_TOKEN = "/client/token";
    public final static String PATH_PALM_CENTER = "/palmid/sdk/#/user-center";
    public final static String PATH_USER_PROFILE = "/palmid/sdk/#/user-profile";
    public final static String PATH_BIND_PHONE = "/palmid/sdk/#/bind-phone";
    public final static String PATH_EXPIRE_TOKEN = "/openapi/user/expire-token";

    //网络错误码 / 业务错误码
    public static final int SUCCESS = 200;
    public static final int ERROR_PARAM = 400;
    public static final int TOKEN_TIMEOUT = 40055;
    public static final int PERMISSION_DENIED = 401400;  // 权限不足
    public static final int AUTH_TOKEN_TIME_OUT = 401401;  // token过期
    public static final int NEED_TO_LOGIN = 400006;
    public static final int CLOUD_SYNC_CONFLICT = 10000003;

    private PalmHttpClient() {
        serverMode = PalmConstants.SERVER_MODE;
    }

    public static PalmHttpClient getInstance() {
        if (sPalmHttpClient == null) {
            synchronized (PalmHttpClient.class) {
                if (sPalmHttpClient == null) {
                    sPalmHttpClient = new PalmHttpClient();
                }
            }
        }
        return sPalmHttpClient;
    }

//    /**
//     * 是否使用palm id公共服务器
//     */
//    public boolean isPalmServer(String redirectUri) {
//        String publicHost = Uri.parse(PalmConstants.PALM_REDIRECT_URI_ONLINE).getHost();
//        return redirectUri != null
//                && publicHost != null && redirectUri.contains(publicHost);
//    }

    /**
     * 返回host
     */
    public String getBaseUrl() {
        String baseUrl = PalmConstants.PALM_ID_HOST_ONLINE;
        if (serverMode == PalmConstants.SERVER_DEV) {
            baseUrl = PalmConstants.PALM_ID_HOST_DEV;
        } else if (serverMode == PalmConstants.SERVER_TEST) {
            baseUrl = PalmConstants.PALM_ID_HOST_TEST;
        } else if (serverMode == PalmConstants.SERVER_PRE) {
            baseUrl = PalmConstants.PALM_ID_HOST_PRE;
        }
        return GslbSdk.getDomain(baseUrl, true);
    }

    /**
     * 返回完整请求接口
     */
    public String getFullUrl(String path, Map<String, Object> params) {
        return getBaseUrl() + path + (params == null ? "" : "?" + PalmUtil.urlEncodeUTF8(params));
    }

    /**
     * 返回palm app版本和应用信息
     */
    public String getPalmAppConfig(Map<String, String> header, Map<String, Object> formData) {
        try {
            return doPost(getBaseUrl() + "/app/cloud/config", header, formData);
        } catch (Exception e) {
            PalmUtil.LOG.e(Log.getStackTraceString(e));
        }
        return null;
    }

    /**
     * 访问开发接口
     */
    public void callOpenApi(String path, Map<String, String> header,
                            Map<String, Object> formData, PalmHttpCallback<String> callback) {
        ThreadExecutors.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    String response;
                    if (formData == null || formData.size() == 0) {
                        response = doGet(getBaseUrl() + path, header, null);
                    } else {
                        response = doPost(getBaseUrl() + path, header, formData);
                    }
                    callback.onSuccess(response);
                } catch (Exception e) {
                    if (e instanceof PalmServerException) {
                        PalmServerException exception = (PalmServerException) e;
                        callback.onFailure(exception.errorCode, exception.errorMessage);
                    } else {
                        callback.onFailure(PalmConstants.ERROR_NETWORK, e.getMessage());
                    }
                }
            }
        });
    }

    /**
     * 获取auth code
     * 异步
     * https://xuanniao-doc.shalltry.com/#/api/sdk-os
     */
    public void getAuthCodeConfirm(Map<String, String> header,
                              Map<String, Object> formData, PalmHttpCallback<String> callback) {
        ThreadExecutors.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    String response = doPost(getBaseUrl() + PATH_AUTH_CODE_CONFIRM, header, formData);
                    callback.onSuccess(response);
                } catch (Exception e) {
                    if (e instanceof PalmServerException) {
                        PalmServerException exception = (PalmServerException) e;
                        callback.onFailure(exception.errorCode, exception.errorMessage);
                    } else {
                        callback.onFailure(PalmConstants.ERROR_NETWORK, e.getMessage());
                    }
                }
            }
        });
    }

    /**
     * 获取auth code
     * 异步
     * https://xuanniao-doc.shalltry.com/#/api/sdk-os
     */
    public void getAuthCode(Map<String, String> header,
                            Map<String, Object> params, PalmHttpCallback<String> callback) {
        ThreadExecutors.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    String response = doGet(getBaseUrl() + PATH_AUTH_CODE, header, params);
                    callback.onSuccess(response);
                } catch (Exception e) {
                    if (e instanceof PalmServerException) {
                        PalmServerException exception = (PalmServerException) e;
                        callback.onFailure(exception.errorCode, exception.errorMessage);
                    } else {
                        callback.onFailure(PalmConstants.ERROR_NETWORK, e.getMessage());
                    }
                }
            }
        });
    }

    /**
     * 获取token
     * 异步
     */
    public void getAccessToken(Map<String, Object> params, PalmHttpCallback<String> callback) {
        ThreadExecutors.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    String response = doGet(getBaseUrl() + PATH_ACCESS_TOKEN, null, params);
                    callback.onSuccess(response);
                } catch (Exception e) {
                    if (e instanceof PalmServerException) {
                        PalmServerException exception = (PalmServerException) e;
                        callback.onFailure(exception.errorCode, exception.errorMessage);
                    } else {
                        callback.onFailure(PalmConstants.ERROR_NETWORK, e.getMessage());
                    }
                }
            }
        });
    }

    /**
     * 刷新token
     */
    public void refreshToken(Map<String, Object> params, PalmHttpCallback<PalmTokenInfo> callback) {
        ThreadExecutors.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    String response = doGet(getBaseUrl() + PATH_ACCESS_TOKEN, null, params);
                    callback.onSuccess(new PalmTokenInfo(response));
                } catch (Exception e) {
                    if (e instanceof PalmServerException) {
                        PalmServerException exception = (PalmServerException) e;
                        callback.onFailure(exception.errorCode, exception.errorMessage);
                    } else {
                        callback.onFailure(PalmConstants.ERROR_NETWORK, e.getMessage());
                    }
                }
            }
        });
    }

    /**
     * 使token过期
     */
    public void expireToken(Map<String, String> header,
                            Map<String, Object> params, PalmHttpCallback<String> callback) {
        ThreadExecutors.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    String response = doGet(getBaseUrl() + PATH_EXPIRE_TOKEN, header, params);
                    if (callback != null) {
                        callback.onSuccess(response);
                    }
                } catch (Exception e) {
                    if (e instanceof PalmServerException) {
                        PalmServerException exception = (PalmServerException) e;
                        callback.onFailure(exception.errorCode, exception.errorMessage);
                    } else {
                        callback.onFailure(-1, e.getMessage());
                    }
                }
            }
        });
    }


    /**
     * GET请求
     */
    private String doGet(String urlString, Map<String, String> header, Map<String, Object> params) throws Exception {
        URL url;
        HttpURLConnection urlConnection = null;
        try {
            if (params != null && params.size() > 0) {
                urlString = urlString + "?" + PalmUtil.urlEncodeUTF8(params);
            }
            PalmUtil.LOG.d("doGet url = " + urlString);
            url = new URL(urlString);
            urlConnection = (HttpURLConnection) url.openConnection();
            if (urlConnection instanceof HttpsURLConnection) {
                if (sslSocketFactory == null) {
                    SSLContext sslContext = SSLContextUtil.getSSLContext();
                    if (sslContext != null) {
                        sslSocketFactory = sslContext.getSocketFactory();
                    }
                }
                if (sslSocketFactory != null) {
                    ((HttpsURLConnection) urlConnection).setSSLSocketFactory(sslSocketFactory);
                    ((HttpsURLConnection) urlConnection).setHostnameVerifier(SSLContextUtil.myHostNameVerifier);
                }
            }
            urlConnection.setRequestProperty("Content-Type", "application/x-www-form-urlencoded");
            urlConnection.setRequestProperty("accept", "*/*");
            urlConnection.setInstanceFollowRedirects(false);

            if (header != null && header.size() > 0) {
                for (Map.Entry<String, String> entry : header.entrySet()) {
                    if (TextUtils.isEmpty(entry.getKey())
                            || TextUtils.isEmpty(entry.getValue())) {
                        continue;
                    }
                    urlConnection.setRequestProperty(entry.getKey(), entry.getValue());
                }
            }

            urlConnection.setRequestMethod(METHOD_GET);

            urlConnection.setReadTimeout(30000);
            urlConnection.setConnectTimeout(30000);

            urlConnection.setDoInput(true);
            urlConnection.setDoOutput(true);

            int responseCode = urlConnection.getResponseCode();
            PalmUtil.LOG.d("doGet responseCode = " + responseCode);
            if (responseCode == HttpURLConnection.HTTP_OK) {
                InputStream is = urlConnection.getInputStream();
                BufferedReader bf = new BufferedReader(new InputStreamReader(is, Charset.forName("UTF-8")));
                StringBuilder buffer = new StringBuilder();
                String line;
                while ((line = bf.readLine()) != null) {
                    buffer.append(line);
                }
                bf.close();
                is.close();
                String response = buffer.toString();
//                PalmUtil.LOG.d("doGet response = " + response);
                return response;
            } else if (responseCode == 301 || responseCode == 302 || responseCode == 303) {
                String response = urlConnection.getHeaderField("Location");
//                PalmUtil.LOG.d("doGet response = " + response);
                Bundle bundle = PalmUtil.urlParse(response);
                if (bundle == null) {
                    throw new PalmServerException(PalmConstants.ERROR_HTTP_REQ, "unknown server error");
                }
                String code = bundle.getString("code", null);
                String state = bundle.getString("state", null);
                if (code != null) {
                    return new JSONObject()
                            .put("code", code).put("state", state).toString();
                } else {
                    String error = bundle.getString("errorCode");
                    int errorCode = PalmConstants.ERROR_HTTP_REQ;
                    if (error.startsWith("400")) {
                        errorCode = Integer.parseInt(error);
                    }
                    String errorMessage = bundle.getString("error_description");
                    throw new PalmServerException(errorCode, errorMessage);
                }
            } else {
                BufferedReader bf = new BufferedReader(new InputStreamReader(
                        urlConnection.getErrorStream(), Charset.forName("UTF-8")));
                StringBuilder buffer = new StringBuilder();
                String line;
                while ((line = bf.readLine()) != null) {
                    buffer.append(line);
                }
                bf.close();
                String response = buffer.toString();
                PalmUtil.LOG.d("doGet response = " + response);
                int errorCode = responseCode;
                String errorMessage = "unknown server error";
                try {
                    JSONObject errorJson = new JSONObject(response);
                    if (errorJson.has("code")) {
                        errorCode = errorJson.optInt("code", errorCode);
                        errorMessage = errorJson.optString("message");
                    } else if (errorJson.has("status")) {
                        errorCode = errorJson.optInt("status", errorCode);
                        errorMessage = errorJson.optString("message");
                    }
                    if (errorCode == PalmConstants.ERROR_TOKEN_TIMEOUT) {
                        errorMessage = "token expired";
                    }
                } catch (Exception e) {
//                    PalmUtil.LOG.e(Log.getStackTraceString(e));
                }
                throw new PalmServerException(errorCode, errorMessage);
            }
        } catch (Exception e) {
            if (e instanceof UnknownHostException) {
                PalmUtil.LOG.e(e.getMessage());
            } else if (!(e instanceof PalmServerException)) {
                PalmUtil.LOG.e(Log.getStackTraceString(e));
            }
            throw e;
        } finally {
            if (urlConnection != null) {
                urlConnection.disconnect();
            }
        }
    }

    /**
     * POST请求
     * @param formData 表单方式发送数据
     */
    private String doPost(String urlString, Map<String, String> header, Map<String, Object> formData) throws Exception {
        URL url;
        HttpURLConnection urlConnection = null;
        try {
            PalmUtil.LOG.d("doPost url = " + urlString);
            url = new URL(urlString);
            urlConnection = (HttpURLConnection) url.openConnection();
            if (urlConnection instanceof HttpsURLConnection) {
                if (sslSocketFactory == null) {
                    SSLContext sslContext = SSLContextUtil.getSSLContext();
                    if (sslContext != null) {
                        sslSocketFactory = sslContext.getSocketFactory();
                    }
                }
                if (sslSocketFactory != null) {
                    ((HttpsURLConnection) urlConnection).setSSLSocketFactory(sslSocketFactory);
                    ((HttpsURLConnection) urlConnection).setHostnameVerifier(SSLContextUtil.myHostNameVerifier);
                }
            }
            urlConnection.setRequestProperty("Content-Type", "application/x-www-form-urlencoded");
            urlConnection.setRequestProperty("accept", "*/*");
            urlConnection.setInstanceFollowRedirects(false);

            if (header != null && header.size() > 0) {
                for (Map.Entry<String, String> entry : header.entrySet()) {
                    if (TextUtils.isEmpty(entry.getKey())
                            || TextUtils.isEmpty(entry.getValue())) {
                        continue;
                    }
                    urlConnection.setRequestProperty(entry.getKey(), entry.getValue());
                }
            }

            urlConnection.setRequestMethod(METHOD_POST);

            urlConnection.setReadTimeout(30000);
            urlConnection.setConnectTimeout(30000);

            urlConnection.setDoInput(true);
            urlConnection.setDoOutput(true);

            StringBuilder builder = new StringBuilder();
            if (formData != null) {
                for (Map.Entry<String, Object> entry : formData.entrySet()) {
                    if (builder.length() == 0) {
                       builder.append(entry.getKey()).append("=").append(entry.getValue());
                    } else {
                        builder.append("&")
                                .append(entry.getKey()).append("=").append(entry.getValue());
                    }
                }
            }
            PalmUtil.LOG.d("doPost formData = " + builder.toString());
            PrintWriter printWriter = new PrintWriter(urlConnection.getOutputStream());
            // 发送请求参数
            printWriter.write(builder.toString());
            // flush输出流的缓冲
            printWriter.flush();
            printWriter.close();

            int responseCode = urlConnection.getResponseCode();
            PalmUtil.LOG.d("doPost responseCode = " + responseCode);
            if (responseCode == HttpURLConnection.HTTP_OK) {
                InputStream is = urlConnection.getInputStream();
                BufferedReader bf = new BufferedReader(new InputStreamReader(is, Charset.forName("UTF-8")));
                StringBuilder buffer = new StringBuilder();
                String line;
                while ((line = bf.readLine()) != null) {
                    buffer.append(line);
                }
                bf.close();
                is.close();
                String response = buffer.toString();
//                PalmUtil.LOG.d("doPost response = " + response);
                return response;
            } else if (responseCode == 301
                    || responseCode == 302 || responseCode == 303) {
                String response = urlConnection.getHeaderField("Location");
//                PalmUtil.LOG.d("doPost response = " + response);
                Bundle bundle = PalmUtil.urlParse(response);
                if (bundle == null) {
                    throw new PalmServerException(PalmConstants.ERROR_HTTP_REQ, "unknown server error");
                }
                String code = bundle.getString("code", null);
                String state = bundle.getString("state", null);
                if (code != null) {
                    return new JSONObject()
                            .put("code", code).put("state", state).toString();
                } else {
                    String error = bundle.getString("errorCode");
                    int errorCode = PalmConstants.ERROR_HTTP_REQ;
                    if (error.startsWith("400")) {
                        errorCode = Integer.parseInt(error);
                    }
                    String errorMessage = bundle.getString("error_description");
                    throw new PalmServerException(errorCode, errorMessage);
                }
            } else {
                BufferedReader bf = new BufferedReader(new InputStreamReader(
                        urlConnection.getErrorStream(), Charset.forName("UTF-8")));
                StringBuilder buffer = new StringBuilder();
                String line;
                while ((line = bf.readLine()) != null) {
                    buffer.append(line);
                }
                bf.close();
                String response = buffer.toString();
                PalmUtil.LOG.d("doPost response = " + response);
                int errorCode = responseCode;
                String errorMessage = "unknown server error";
                try {
                    JSONObject errorJson = new JSONObject(response);
                    if (errorJson.has("code")) {
                        errorCode = errorJson.optInt("code", errorCode);
                        errorMessage = errorJson.optString("message");
                    } else if (errorJson.has("status")) {
                        errorCode = errorJson.optInt("status", errorCode);
                        errorMessage = errorJson.optString("message");
                    }
                    if (errorCode == PalmConstants.ERROR_TOKEN_TIMEOUT) {
                        errorMessage = "token expired";
                    }
                } catch (Exception e) {
//                    PalmUtil.LOG.e(Log.getStackTraceString(e));
                }
                throw new PalmServerException(errorCode, errorMessage);
            }
        } catch (Exception e) {
            if (e instanceof UnknownHostException) {
                PalmUtil.LOG.e(e.getMessage());
            } else if (!(e instanceof PalmServerException)) {
                PalmUtil.LOG.e(Log.getStackTraceString(e));
            }
            throw e;
        } finally {
            if (urlConnection != null) {
                urlConnection.disconnect();
            }
        }
    }
}
