/*
 * Transsion Top Secret
 * Copyright (C) 2021 Transsion Inc.
 * Welife
 * @author OS : chunming.hu (18646581)
 * @version 1.0
 */

package com.smartlife.nebula.device

import android.net.ConnectivityManager
import com.smartlife.nebula.process.*
import kotlinx.coroutines.flow.Flow

/**
 * @author chunming.hu
 * @date 2021/4/25 11:14
 * @Description TODO
 */
interface IDeviceManger : WelifeDevice.ScanResultCallback {

    companion object {
        const val DEFAULT_SCAN_TIME: Long = 90 * 1000
        const val WIFI_STATE_ENABLE = 1
        const val WIFI_STATE_DISABLE = 2
        const val HOT_SPOT_OPEN = 13 //热点打开
        const val HOT_SPOT_CLOSE = 11 //热点关闭
        const val BLUETOOTH_STATE_ENABLE = 1
        const val BLUETOOTH_STATE_DISABLE = 2

        //产品类型
        const val DEVICE_TYPE_WEARABLE = 0x00
        const val DEVICE_TYPE_MEDIA = 0x01
        const val DEVICE_TYPE_HOME = 0x02
        const val DEVICE_TYPE_ILLUMINATION = 0x03
        const val DEVICE_TYPE_ELECTRONIC = 0x04
        const val DEVICE_TYPE_ROUTER = 0x05

        //产品子类型
        const val DEVICE_SUB_TYPE_WATCH = 0x00
        const val DEVICE_SUB_TYPE_BRAND = 0x01
        const val DEVICE_SUB_TYPE_TWS = 0x02
        const val DEVICE_SUB_TYPE_AUDIO = 0x03
        const val DEVICE_SUB_TYPE_LIGHT = 0x04
        const val DEVICE_SUB_TYPE_SOCKET = 0x05
        const val DEVICE_SUB_TYPE_FAN = 0x06
        const val DEVICE_SUB_TYPE_TV = 0x07
        const val DEVICE_SUB_TYPE_MIFI = 0x08
        const val DEVICE_SUB_TYPE_CPE = 0x09
        const val DEVICE_SUB_TYPE_NOTEPAD = 0x0A
        const val DEVICE_SUB_TYPE_ROUTER = 0x0B
        const val DEVICE_SUB_TYPE_GATEWAY = 0x0C

        const val BRAND_ORAIMO = 0x00
        const val BRAND_SYINIX = 0x01
        const val BRAND_ITEL = 0x02
        const val BRAND_TECNO = 0x03
        const val BRAND_INFINIX = 0x04
        const val BRAND_BOOMPLAY = 0x05

        const val MANUFACTURE_TRANSSION = 0x00
        const val MANUFACTURE_SKYWORTH = 0x01
        const val MANUFACTURE_CHANGHONG = 0x02
        const val MANUFACTURE_YOUCHUANGYI = 0x03

        //------通过pid解析出的无线连接类型
        /**
         * 只支持经典蓝牙
         */
        const val WIRELESS_CHANNEL_BT = 0x00

        /**
         * 只支持BLE 手表手环
         */
        const val WIRELESS_CHANNEL_BLE = 0x01

        /**
         * BLE_MESH
         */
        const val WIRELESS_CHANNEL_BLE_MESH = 0x02

        /**
         * 支持wifi  关掉wifi和热点清除扫描到的数据
         */
        const val WIRELESS_CHANNEL_WIFI = 0x10

        /**
         * mifi/pcie  关掉wifi清除扫描的数据
         */
        const val WIRELESS_CHANNEL_WIFI_AP = 0x11

        /**
         * 热点 纯wifi的tv 关掉热点清除扫描到的数据
         */
        const val WIRELESS_CHANNEL_WIFI_STA = 0x12

        /**
         * 蓝牙加wifi 带蓝牙的TV
         */
        const val WIRELESS_CHANNEL_BT_BLE_WIFI = 0x31

        /**
         * 全部类型
         */
        const val WIRELESS_CHANNEL_NONE = 0xFF

        //自定义的扫描类型后续考虑把这些常量删除，与上面的无线连接类型重复
        //蓝牙
        const val WIRELESS_SCAN_CHANNEL_BLE = 110 //17

        //wifi
        const val WIRELESS_SCAN_CHANNEL_WIFI = 120 //18

        //wifi热点
        const val WIRELESS_SCAN_CHANNEL_AP = 130   //19

        //GPS
        const val WIRELESS_SCAN_CHANNEL_GPS = 140   //19

    }

    /**
     *连接设备
     * @param device 设备扫描信息
     */
    fun connectDevice(pid: String, did: String, callback: (Boolean) -> Unit)

    /**
     * 断开设备
     * @param did 设备唯一标识
     */
    fun disConnectDevice(pid: String, did: String)

    /**
     * 添加蓝牙状态改变回调
     * @param callback 蓝牙状态改变回调
     */
    fun addBluetoothStateChangeListener(callback: (Int) -> Unit)

    /**
     * 移除蓝牙状态改变回调
     * @param callback 需要移除的回调
     */
    fun removeBluetoothStateChangeListener(callback: (Int) -> Unit)

    /**
     * 添加WIFI状态改变回调
     * @param callback WIFI 状态改变回调
     */
    fun addWifiStateChangeListener(callback: (Int) -> Unit)

    /**
     * 移除Wifi状态改变回调
     * @param callback 需要移除的回调
     */
    fun removeWifiStateChangeListener(callback: (Int) -> Unit)

    /**
     * 添加AP状态变化回调
     * @param callback AP 状态改变回调
     */
    fun addApStateChangeListener(callback: (Int) -> Unit)

    /**
     * 移除Ap状态改变回调
     * @param callback 需要移除的回调
     */
    fun removeApStateChangeListener(callback: (Int) -> Unit)


    /**
     * 添加网络变化回调
     * @param callback 网络变化监听
     */
    fun addNetworkCallback(callback: ConnectivityManager.NetworkCallback)

    /**
     * 移除网络变化回调
     * @param callback 网络变化监听
     */
    fun removeNetworkCallback(callback: ConnectivityManager.NetworkCallback)

    /**
     * 蓝牙是否已经打开
     * @return 蓝牙开启状态
     */
    fun isBluetoothEnable(): Boolean

    /**
     * WIFI 是否打开
     * @return WIFI 开关状态
     */
    fun isWifiEnable(): Boolean

    /**
     * WIFI 是否连接
     * @return WIFI连接状态
     */
    fun isWifiConnected(): Boolean

    /**
     * 热点是否打开
     * @return 热点开启状态
     */
    fun isApEnable(): Boolean

    /**
     * 获取特定设备类型的扫描结果
     * instead [getScanProcessorResult],[getScanProcessorError],[getScanProcessorChangeState]
     * @param pid 产品PID
     * @return 指定产品类型的扫描结果
     */
    @Deprecated("instead getScanProcessorResult,getScanProcessorError,getScanProcessorChangeState ")
    fun getScanResult(pid: String, callback: WelifeDevice.ScanResultCallback)


    /**
     * 监听全部[onScanResult],如果需要指定设备请使用使用[Flow]的过滤操作符对[ScanProcessorResult]进行过滤
     *  example:
     *  getScanProcessorResult()
     *  .filter{[WelifeDevice.getScanProcessorId]==[ScanProcessorResult.scanProcessorWirelessChannel]}
     *  .collectLatest{得到的指定scanProcessorWirelessChannelde的[ScanProcessorResult]事件}
     *
     * @return
     */
    fun getScanProcessorResult(): Flow<ScanProcessorResult>

    /**
     * 监听全部[onScanError],如果需要指定设备请使用使用[Flow]的过滤操作符对[ScanProcessorErrorState]进行过滤,请参考[getScanProcessorResult]
     *
     * @return
     */
    fun getScanProcessorError(): Flow<ScanProcessorErrorState>

    /**
     * 监听全部[onScanStateChange],如果需要指定设备请使用使用[Flow]的过滤操作符对[ScanProcessorChangeState]进行过滤,请参考[getScanProcessorResult]
     *
     * @return
     */
    fun getScanProcessorChangeState(): Flow<ScanProcessorChangeState>

    /**
     * 停止扫描
     * @param pid 产品PID
     * @return 指定产品类型的扫描结果
     */
    fun stopScan(pid: String)

    /**
     * 获取Device 实例
     * @param pid 产品ID
     * @return 设备实例
     */
    fun getDevice(pid: String): WelifeDevice?


    /**
     * 根据设备类型断开同类型设备
     * @param pid 产品唯一标识
     */
    fun disconnectDeviceByDeviceType(pid: String)

    /**
     * 根据welifeDevice获取对应的[AbsScanProcessor]
     *
     * @param welifeDevice
     * @return
     */
    fun getScanProcess(welifeDevice: WelifeDevice): AbsScanProcessor?


    /**
     * 生成设备的无线扫描类型
     *
     * @param welifeDevice
     * @return
     */
    fun getWirelessType(welifeDevice: WelifeDevice): Int


    /**
     * 生成[AbsScanProcessor.mScanProcessorId]
     * @param welifeDevice
     * @return
     */
    fun getScanProcessorId(welifeDevice: WelifeDevice): Int

    /**
     * 根据预设的[android.bluetooth.BluetoothDevice.getName]映射[WelifeDevice]
     *
     * @return
     */
    fun getWelifeDeviceDisplayNameMap(): HashMap<
            String, WelifeDevice>

    /**
     *
     *由[AbsScanProcessor]调用该方法，然后在具体的实现类中进行事件的统一处理
     * @param wirelessType
     * @param scanProcessorWirelessChannel
     * @param scannedDeviceInfo
     */
    fun onScanResult(
        wirelessType: Int,
        scanProcessorWirelessChannel: Int,
        scannedDeviceInfo: WelifeDevice.IScannedDeviceInfo
    )

    /**
     * 由[AbsScanProcessor]调用该方法，然后在具体的实现类中进行事件的统一处理
     *
     * @param wirelessType
     * @param scanProcessorWirelessChannel
     * @param errorCode
     */
    fun onScanError(wirelessType: Int, scanProcessorWirelessChannel: Int, errorCode: Int)

    /**
     * 由[AbsScanProcessor]调用该方法，然后在具体的实现类中进行事件的统一处理
     *
     * @param wirelessType
     * @param scanProcessorWirelessChannel
     * @param newState
     */
    fun onScanStateChange(
        wirelessType: Int,
        scanProcessorWirelessChannel: Int,
        newState: WelifeDevice.ScanResultCallback.ScanState
    )

    /**
     * 开启welink的ble扫描
     *
     * @param absScanProcessor
     * @param useKolunScan
     * @param productSubTypeList
     * @param minRssi
     * @param needAutoConnect
     * @param scanModeLowLatency
     * @param needAllResult
     */
    fun startBleScanAndConnect(
        absScanProcessor: AbsScanProcessor,
        useKolunScan: Boolean,
        productSubTypeList: ArrayList<Byte>,
        minRssi: Int,
        needAutoConnect: Boolean,
        scanModeLowLatency: Int,
        needAllResult: Boolean
    ): Boolean

    /**
     * 关闭welink的ble扫描
     *
     * @param absScanProcessor
     */
    fun stopBleScan(absScanProcessor: AbsScanProcessor)

    /**
     * 由于抽象接口没有依赖welink aar，又需要保持BleScanTool实例的唯一性，无法直接反BleScanTool使用any替代
     * 后续考虑优化
     *
     * @return
     */
    fun getBleScanTool(): Any

}