package com.smartlife.nebula.device

import android.content.Context
import android.os.Parcelable
import com.smartlife.nebula.device.base.AdapterScanDeviceInfo
import com.smartlife.nebula.lib.IWelifeDevice
import com.smartlife.nebula.process.AbsScanProcessor
import com.smartlife.nebula.process.IScanProcessorFilter
import java.util.*
import kotlin.collections.HashMap

/**
 * @author chunming.hu
 * @date 2021/4/26 20:50
 * @Description TODO
 */
abstract class WelifeDevice(private val context: Context, private val manger: IDeviceManger) :
    IWelifeDevice {

    companion object {
        private var mUpdateListener: ((String, Map<Int, Any>) -> Unit)? = null
    }


    /**
     * 设备连接状态
     */
    enum class ConnectState {
        /**
         * 设备连接状态连接中
         */
        CONNECT_STATE_CONNECTING,

        /**
         * 设备连接状态已连接
         */
        CONNECT_STATE_CONNECTED,

        /**
         * 设备连接状态断开连接
         */
        CONNECT_STATE_DISCONNECTED
    }

    enum class ParamsType {
        PARAMS_TYPE_STATUS,
        PARAMS_TYPE_CARD_CONTENT
    }

    /**
     * 卡片类型
     */
    enum class CardDataType {
        CARD_DATA_TYPE_FIRST_ICON,
        CARD_DATA_TYPE_FIRST_LABEL,
        CARD_DATA_TYPE_FIRST_CONTENT,
        CARD_DATA_TYPE_SECOND_ICON,
        CARD_DATA_TYPE_SECOND_LABEL,
        CARD_DATA_TYPE_SECOND_CONTENT,
        CARD_DATA_TYPE_STATUS_CONTENT,
        CARD_DATA_TYPE_STATUS,
        CARD_DATA_TYPE_BATTERY_VALUE,
        CARD_DATA_TYPE_BATTERY_IS_CHARGING,
        CARD_TYPE_COLLAPSED_TEXT_THIRD,
        CARD_TYPE_COLLAPSED_IMG_FIRST,
        CARD_TYPE_COLLAPSED_IMG_SECOND,
        CARD_TYPE_COLLAPSED_IMG_THIRD,
        CARD_TYPE_EXPENDED_TEXT_FIRST,
        CARD_TYPE_EXPENDED_TEXT_SECOND,
        CARD_TYPE_EXPENDED_TEXT_THIRD,
    }

    /**
     * 图标类型
     */
    enum class IconType {
        ICON_TYPE_RESOURCE,
        ICON_TYPE_BATTERY,
        ICON_TYPE_SIGNAL
    }

    /**
     * 扫描操作结果状态
     */
    enum class ScanOperateState {
        /**
         * anadroid S 附近的设备权限
         *
         */
        SCAN_OPERATE_STATE_NEED_NEARBY_DEVICE_PERMISSION,
        /**
         * 需要开启蓝牙
         */
        SCAN_OPERATE_STATE_NEED_OPEN_BLE,

        /**
         * 需要开启WIFI
         */
        SCAN_OPERATE_STATE_NEED_OPEN_WIFI,

        /**
         * 需要开启热点
         */
        SCAN_OPERATE_STATE_NEED_OPEN_AP,

        /**
         * 需要打开位置开关
         */
        SCAN_OPERATE_STATE_NEED_OPEN_LOCATION,

        /**
         * 需要请求权限
         */
        SCAN_OPERATE_STATE_NEED_REQUEST_PERMISSION,


        /**
         * 操作成功
         */
        SCAN_OPERATE_STATE_SUCCESS,

        /**
         * 无状态（无任何设备开启扫描）
         */
        SCAN_OPERATE_STATE_NONE


    }


    /**
     * 获取设备配置信息
     * @return 设备配置信息
     */
    abstract fun getDeviceConfig(): DeviceConfig



    /**
     * 为了兼容mifi,仍然保留.后续迭代删除该方法
     * @param context Context
     * @param callback 扫描的设备结果回调
     * @return 扫描启动结果状态回调
     */
    @Deprecated("instead startScan")
    open fun startScan(
        context: Context,
        callback: ScanResultCallback
    ): ScanOperateState {
        return ScanOperateState.SCAN_OPERATE_STATE_SUCCESS
    }

    /**
     * 开始扫描
     * @return 扫描启动结果状态回调
     */
    fun startScan(filter: IScanProcessorFilter? =null,reconnect:Boolean = false): ScanOperateState {
        return getScanProcess()?.startScan(filter,reconnect) ?: ScanOperateState.SCAN_OPERATE_STATE_NONE
    }


    /**
     * 为了兼容mifi,仍然保留.后续迭代删除该方法
     * 停止扫描
     */
    @Deprecated("instead stopScanProcessor()")
    open fun stopScan() {

    }

    /**
     * 停止扫描
     */
    fun stopScanProcessor() {
        getScanProcess()?.stopScan()
    }

    /**
     * 如果要使用自定义的[AbsScanProcessor]请复写该方法并return ture
     *
     * @return
     */
    open fun isCustomProccessor(): Boolean {
        return false
    }


    fun getScanProcess(): AbsScanProcessor? {
        return manger.getScanProcess(this)
    }

    /**
     * 生成自定义的[AbsScanProcessor]
     *
     * @param wirelessType
     * @param scanProcessorWirelessChannel
     * @return
     */
    @Deprecated("3.0 replace with ILinkHelper")
    open fun getCustomScanProcess(): AbsScanProcessor? {
        return null
    }

    /**
     * 如果多种[WelifeDevice]公用一个[AbsScanProcessor],请复写该方法
     *
     * @return
     */
    @Deprecated("3.0 replace with ILinkHelper")
    open fun getScanProcessorWirelessChannelPid(): String {
        return getDeviceConfig().pid
    }

    /**
     *
     *与设备绑定的[AbsScanProcessor]的[AbsScanProcessor.mScanProcessorId]
     * 默认的是pid的无线扫描类型
     * @return
     */
    @Deprecated("3.0 replace with ILinkHelper")
    fun getScanProcessorId(): Int {
        return getScanProcess()?.getScanProcessorId()?:-1
    }



    /**
     * 连接设备
     * @param deviceInfo 扫描到的设备信息
     * @param callback 连接状态回调
     */
    abstract fun connect(deviceInfo: IScannedDeviceInfo)

    /**
     * 兼容3.0连接方法
     */
    open fun connect(did: String,extra:Any?){
        connect(AdapterScanDeviceInfo(getDeviceConfig().displayName,did,getDeviceConfig().pid,""))
    }


    /**
     * 断开连接
     * @param did 设备唯一标识
     */
    abstract fun disconnect(did: String)

    /**
     * 移除已添加设备
     * @param did 设备唯一标识
     */
    abstract fun remove(did: String)

    /**
     * 更新设备状态信息
     * @param did 设备唯一标识
     * @param params 需要更新的字段
     */
    fun update(did: String, vararg params: Pair<Int, Any>) {
        // TODO: 2021/9/16/016  UI更新字段
        mUpdateListener?.invoke(
            did, HashMap<Int, Any>().apply {
                params.forEach { param ->
                    put(param.first, param.second)
                }
            }
        )
    }

    fun setUpdateListener(listener: (String, Map<Int, Any>) -> Unit) {
        mUpdateListener = listener
    }

    /**
     * 设备首页卡片是否能够展开
     * @return 卡片能否展开
     */
    @Deprecated("3.0 deprecated")
    abstract fun isCardCanExpend(): Boolean


    /**
     * 获取特定类型开的设备首页卡片填充内容
     * @param isExpended 是否为展内容
     * @param did 设备唯一标识
     * @param cardDataType 数据类型
     * @return 特定卡片填充内容
     */
    abstract fun getCardContent(isExpended: Boolean, did: String, cardDataType: CardDataType): Any?


    /**
     * 获取设备所有的卡片内容
     * @param isExpended 是否为展开内容
     * @param did 设备唯一标识
     * @return 卡片所有填充内容
     */
    abstract fun getCardContents(isExpended: Boolean, did: String): HashMap<Int, Any>

    /**
     * 获取设备的额外信息，比如：获取路由器的psn
     */
    open fun getDeviceExtraInfo(param: String?, callback: IWelifeDeviceParameterCallback) {

    }


    /**
     * 首页卡片显示回调
     * @param did 设备唯一标识
     */
    open fun onCardShow(did: String) {

    }

    /**
     * 首页卡片隐藏回调
     * @param did 设备唯一标识
     */
    open fun onCardHide(did: String) {

    }

    /**
     * 设置配置类
     * @property displayName 设备显示名称
     * @property icon 设备图标名称（放入对应屏幕密度Drawable文件夹下的设备图标文件名称，不包含扩展名）
     * @property controlPagePath 控制页面路径
     * @property pid 产品类型唯一标识
     * @property controlPageClassName 设备详情页类名
     */

    class DeviceConfig private constructor(
        var displayName: String,
        //设备图标资源名称
        var icon: String,
        //设备控制页面路径
        var controlPagePath: String,
        //设备型号ID（平台化后平台分配的唯一标识）
        var pid: String,
        var deviceControlProvide: AbstractDeviceControlProvide?,
        var controlPageClassName: String,
    ) {
        class Builder {
            private var mDisplayName: String = ""
            private var mIcon: String = ""
            private var mControlPagePath: String = ""
            private var mPid: String = UUID.randomUUID().toString()
            private var mDeviceControlProvide: AbstractDeviceControlProvide? = null
            private var mControlPageClassName: String = ""

            /**
             * 设备设备显示名称
             * @param displayName 设备显示名称
             * @return @code this
             */
            fun setDisplayName(displayName: String): Builder {
                mDisplayName = displayName
                return this
            }


            /**
             * 设置设备图标
             * @param icon 设备图片名称
             * @return @code this
             */
            fun setIcon(icon: String): Builder {
                mIcon = icon
                return this
            }

            /**
             * 设置控制页面路径
             * @param controlPagePath 设备控制路径
             * @return @code this
             */
            fun setControlPath(controlPagePath: String): Builder {
                mControlPagePath = controlPagePath
                return this
            }


            /**
             * 设置设备型号ID
             * @param pid 设备型号ID
             * @return @code this
             */
            fun setPid(pid: String): Builder {
                mPid = pid
                return this
            }

            /**
             * 设置DeviceControl Provide
             * @param Device Control配置实现类
             * @return @code this
             */
            fun setDeviceControlProvide(deviceControlProvide: AbstractDeviceControlProvide): Builder {
                mDeviceControlProvide = deviceControlProvide
                return this
            }

            fun setControlPageClassName(controlPageClassName: String): Builder {
                mControlPageClassName = controlPageClassName
                return this
            }


            fun build(): DeviceConfig {
                return DeviceConfig(
                    displayName = mDisplayName,
                    icon = mIcon,
                    controlPagePath = mControlPagePath,
                    pid = mPid,
                    deviceControlProvide = mDeviceControlProvide,
                    controlPageClassName =  mControlPageClassName
                )
            }

        }
    }

    /**
     * 扫描到设备，设备信息回调接口
     */
    @Deprecated("3.0 replace with ILinkHelper")
    interface ScanResultCallback {
        /**
         * 扫描状态
         */
        enum class ScanState {
            /**
             * 扫描中状态
             */
            SCAN_STATE_SCANNING,

            /**
             * 扫描完成状态
             */
            SCAN_STATE_COMPLETE,

            /**
             * 扫描超时
             */
            SCAN_STATE_TIME_OUT,

            /**
             * 扫描错误状态
             */
            SCAN_STATE_ERROR,

            /**
             * 扫描硬件通道关闭
             */
            SCAN_STATE_HW_CLOSED
        }

//        /**
//         * 扫描结果返回
//         * @param channel 扫描结果渠道（WIFI/AP/BLE）
//         * @param scannedDeviceInfo 设备信息
//         */
//        fun onResult(channel: WirelessType, scannedDeviceInfo: IScannedDeviceInfo)

        /**
         * 扫描结果返回
         * @param channel 扫描结果渠道（WIFI/AP/BLE）,
         * 20210917更新，该字段所有方法用WIRELESS_TYPE_NONE入参，业务逻辑不要再使用该字段
         * @param scannedDeviceInfo 设备信息
         */
        @Deprecated("instead  IDeviceManger#onScanResult")
        fun onResult(wirelessChannel: WirelessType, scannedDeviceInfo: IScannedDeviceInfo)

        /**
         * 扫描失败
         * @param wirelessChannel 设备类型
         * @param errorCode 错误码
         */
        @Deprecated("instead  IDeviceManger#onScanError")
        fun onError(wirelessChannel: Int, errorCode: Int)

        /**
         * 状态改变回调
         * @param wirelessType 扫描类型
         * @param newState 新的状态  可使用的状态参见[ScanState]
         */
        @Deprecated("instead  IDeviceManger#onScanStateChange")
        fun onStateChange(wirelessType: Int, newState: ScanState)
    }

    /**
     * @author chunming.hu
     * @Description 抽象设备信息接口
     * @property did 设备唯一标识
     * @property pid 产品唯一标识
     */
    interface IScannedDeviceInfo : Parcelable {
        //设备唯一标识,mac地址
        var did: String

        //产品唯一标识（一个类型中有多个产品，每个产品对应唯一的产品ID）
        var pid: String

        var deviceName: String

        //扩展字段
        var extraString: String

    }

    interface IWelifeDeviceParameterCallback{
        fun onResult(result: String?)
    }


    enum class WirelessType {
        // TODO: 2021/9/17/017 该枚举类不再进行业务逻辑处理,mifi也未使用该字段,
        //  仅保留作为ScanResultCallback中onResult方法的入参,仅传递WIRELESS_TYPE_NONE,后续迭代考虑删除该枚举类
        WIRELESS_TYPE_WIFI,
        WIRELESS_TYPE_AP,
        WIRELESS_TYPE_BLE,
        WIRELESS_TYPE_NONE
    }
}