package com.transsion.iot.communication.udp

import android.os.Handler
import android.util.Log
import com.google.gson.Gson
import com.google.gson.JsonSyntaxException
import com.transsion.iot.communication.packet.resp.BasicInfo
import java.io.IOException
import java.net.DatagramPacket
import java.net.DatagramSocket
import java.net.InetSocketAddress
import java.util.concurrent.atomic.AtomicBoolean

/**
 * @Author fuhuang.fan
 * @Date 2022/7/19/019 15:05
 * @Description
 */
object UDPClient {

    const val BROADCAST_PORT = 6000
    const val MSG_RECEIVE_DATA = 100
    const val MSG_RECEIVE_ERROR = -100

    private lateinit var mReceiveThread: SocketReceiveThread
    private lateinit var mHandler: Handler
    private var mPort = BROADCAST_PORT
    private var mSocket: DatagramSocket? = null

    fun startReceive(port: Int = BROADCAST_PORT) {
        mPort = port
        try {
            if (mSocket != null) {
                mSocket?.close()
            }
            mSocket = DatagramSocket(mPort)
            if (mSocket == null) {
                mSocket = DatagramSocket(null)
                mSocket?.reuseAddress = true
                mSocket?.bind(InetSocketAddress(mPort))
            }
            mReceiveThread = SocketReceiveThread()
            mReceiveThread.start()
        } catch (e: Exception) {
            val msg = mHandler.obtainMessage()
            msg.what = MSG_RECEIVE_ERROR
            msg.obj = e.message
            mHandler.sendMessage(msg)
        }

    }

    fun stopReceive() {
        mReceiveThread.threadExit()
        mHandler.removeMessages(MSG_RECEIVE_ERROR)
        mHandler.removeMessages(MSG_RECEIVE_DATA)
        if (mSocket != null) {
            mSocket?.close()
        }
    }

    fun setReceiveHandler(handler: Handler) {
        mHandler = handler
    }

    class SocketReceiveThread : Thread() {
        private var threadExit = AtomicBoolean(false)
        override fun run() {
            val buffer = ByteArray(1024)
            while (!threadExit.get()) {
                try {
                    val datagramPacket = DatagramPacket(buffer, buffer.size)
                    mSocket?.receive(datagramPacket)
                    val receiveData = String(buffer, 0, datagramPacket.length)
                    Log.d("UDPClient", "receiveData=$receiveData")
                    val basicInfo = Gson().fromJson(receiveData, BasicInfo::class.java)
                    Log.d("UDPClient", "basicInfo=$basicInfo")
                    val msg = mHandler.obtainMessage()
                    msg.what = MSG_RECEIVE_DATA
                    msg.obj = basicInfo
                    mHandler.sendMessage(msg)
                } catch (e: IOException) {
                    e.printStackTrace()
                    val msg = mHandler.obtainMessage()
                    msg.what = MSG_RECEIVE_ERROR
                    msg.obj = e.message
                    mHandler.sendMessage(msg)
                    threadExit()
                } catch (e: JsonSyntaxException) {
                    e.printStackTrace()
                    val msg = mHandler.obtainMessage()
                    msg.what = MSG_RECEIVE_ERROR
                    msg.obj = e.message
                    mHandler.sendMessage(msg)
                    threadExit()
                }
            }
        }

        fun threadExit() {
            threadExit.set(true)
        }
    }
}