package com.transsion.iot.communication

import android.text.TextUtils
import android.util.Log
import com.google.gson.Gson
import com.google.gson.JsonSyntaxException
import com.google.gson.reflect.TypeToken
import com.transsion.iot.communication.encrypt.AESUtils
import com.transsion.iot.communication.packet.RequestData
import com.transsion.iot.communication.packet.ResponseData
import com.transsion.iot.communication.packet.ServiceIdConstants
import com.transsion.iot.communication.packet.req.*
import com.transsion.iot.communication.packet.resp.*
import com.transsion.iot.communication.tcp.TCPClient
import org.json.JSONException
import java.lang.reflect.Type

/**
 * @Author fuhuang.fan
 * @Date 2022/7/8/008 17:12
 * @Description
 */
class WIFIClient : MessageReceiveListener{

    private val TAG = "WIFIClient"

    private var mMeshSearchListener: MeshSearchListener? = null

    fun getConnectState(): Int {
        return TCPClient.getConnectState()
    }

    fun isConnected(): Boolean {
        return TCPClient.isConnected()
    }

    fun setConnectListener(listener: ConnectListener) {
        TCPClient.setOnConnectListener(listener)
    }

    fun setMeshSearchListener(listener: MeshSearchListener?) {
        mMeshSearchListener = listener
    }

    fun connect(ip: String, port: Int) {
        TCPClient.setMessageReceiveListener(this)
        TCPClient.setIP(ip)
        TCPClient.setPort(port)
        TCPClient.connect()
    }

    fun connect() {
        TCPClient.setMessageReceiveListener(this)
        TCPClient.connect()
    }

    fun disconnect() {
        TCPClient.disconnect()
    }

    fun onDestroy() {
        TCPClient.onDestroy()
    }

    private fun sendData(obj: RequestData): String?{
        return if (!isConnected()) {
            Log.d(TAG, "sendData not connected")
            ""
        } else {
            Log.d(TAG, "sendData obj=$obj")
            var jsonStr: String? = null
            try {
                jsonStr = Gson().toJson(obj, RequestData::class.java)
            } catch (e: JSONException) {
                e.printStackTrace()
            }
            Log.d(TAG, "sendData jsonStr=$jsonStr")
            if (TextUtils.isEmpty(jsonStr)) {
                Log.d(TAG, "sendData jsonStr is null")
                ""
            } else {
//                val requestStr = AESUtils.encryptAES(jsonStr!!)
//                Log.d(TAG, "requestStr encryptAES = $requestStr")
                val resultStr = TCPClient.send(obj.serviceId,jsonStr!!)
                Log.d(TAG, "sendData responseStr=$resultStr")
//                val responseStr = AESUtils.decryptAES(resultStr!!)
//                Log.d(TAG, "responseStr decryptAES = $responseStr")
                resultStr
            }
        }
    }

    private fun sendData(serviceId: String): String? {
        val requestData = RequestData(
            id = System.currentTimeMillis().toString(),
            params = BaseParam(),
            serviceId = serviceId
        )
        return sendData(requestData)
    }

    private fun <T> getData(serviceId: String, type: Type): ResponseData<T>? {
        val responseStr = sendData(serviceId)
        var response : ResponseData<T>? = null
        try {
            response = Gson().fromJson(responseStr, type)
        } catch (e: JsonSyntaxException) {
            e.printStackTrace()
        }
        return response
    }

    private fun <T> getData(obj: RequestData, type: Type): ResponseData<T>? {
        val responseStr = sendData(obj)
        var response : ResponseData<T>? = null
        try {
            response = Gson().fromJson(responseStr, type)
        } catch (e: JsonSyntaxException) {
            e.printStackTrace()
        }
        return response
    }

    private fun getData(obj: RequestData): ResponseData<BaseResult>? {
        val responseStr = sendData(obj)
        val type: Type = object : TypeToken<ResponseData<BaseResult>>(){}.type
        var response : ResponseData<BaseResult>? = null
        try {
            response = Gson().fromJson(responseStr, type)
        } catch (e: JsonSyntaxException) {
            e.printStackTrace()
        }
        return response
    }

    private fun getData(serviceId: String): ResponseData<BaseResult>?{
        val responseStr = sendData(serviceId)
        val type: Type = object : TypeToken<ResponseData<BaseResult>>(){}.type
        var response : ResponseData<BaseResult>? = null
        try {
            response = Gson().fromJson(responseStr, type)
        } catch (e: JsonSyntaxException) {
            e.printStackTrace()
        }
        return response
    }

    fun getCurDeviceNetworkSpeedInfo(): ResponseData<NetworkSpeedInfo>? {
        val type: Type = object : TypeToken<ResponseData<NetworkSpeedInfo>>(){}.type
        return getData(ServiceIdConstants.GET_NETWORK_SPEED_INFO, type)
    }

    fun getOnlineDeviceCount(): ResponseData<OnlineDeviceCount>? {
        val type: Type = object : TypeToken<ResponseData<OnlineDeviceCount>>(){}.type
        return getData(ServiceIdConstants.GET_ONLINE_DEVICE_COUNT, type)
    }


    fun getDeviceBasicInfo(): ResponseData<DeviceBasicInfo>? {
        val type: Type = object : TypeToken<ResponseData<DeviceBasicInfo>>(){}.type
        return getData(ServiceIdConstants.GET_DEVICE_BASIC_INFO, type)
    }

    fun getConnectedDeviceList(): ResponseData<ConnectedDeviceList>? {
        val type: Type = object : TypeToken<ResponseData<ConnectedDeviceList>>(){}.type
        return getData(ServiceIdConstants.GET_CONNECTED_DEVICE_LIST, type)
    }

    fun disableDevice(deviceMac: String): ResponseData<BaseResult>? {
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = DisableDeviceParam(deviceMac), serviceId = ServiceIdConstants.DISABLE_DEVICE)
        return getData(requestData)
    }

    fun enableDevice(deviceMac: String): ResponseData<BaseResult>? {
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = DisableDeviceParam(deviceMac), serviceId = ServiceIdConstants.ENABLE_DEVICE)
        return getData(requestData)
    }

    fun getNetworkSettings(): ResponseData<NetworkSettings>? {
        val type: Type = object : TypeToken<ResponseData<NetworkSettings>>(){}.type
        return getData(ServiceIdConstants.GET_NETWORK_SETTINGS, type)
    }

    fun setSSID(frequency: Int, ssid : String): ResponseData<BaseResult>? {
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SetSSIDParam(frequency, ssid), serviceId = ServiceIdConstants.SET_SSID)
        return getData(requestData)
    }

    fun setPassword(frequency: Int, password : String): ResponseData<BaseResult>? {
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SetPasswordParam(frequency, password), serviceId = ServiceIdConstants.SET_PASSWORD)
        return getData(requestData)
    }

    fun setWIFIVisibility(frequency: Int, visible: Int): ResponseData<BaseResult>? {
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SetVisibleParam(frequency, visible), serviceId = ServiceIdConstants.SET_WIFI_VISIBILITY)
        return getData(requestData)
    }

    fun setWIFIState(frequency: Int, workMode: Int): ResponseData<BaseResult>? {
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SetWIFIStateParam(frequency, workMode), serviceId = ServiceIdConstants.SET_WIFI_STATE)
        return getData(requestData)
    }

    fun setWIFICombineState(wifiCombineMode: Int): ResponseData<BaseResult>? {
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SetWIFICombineParam(wifiCombineMode), serviceId = ServiceIdConstants.SET_WIFI_COMBINE_STATE)
        return getData(requestData)
    }

    fun getSupportChannelList(frequency: Int): ResponseData<SupportChannelList>? {
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = GetSupportChannelParam(frequency), serviceId = ServiceIdConstants.GET_SUPPORT_CHANNEL_LIST)
        val type: Type = object : TypeToken<ResponseData<SupportChannelList>>(){}.type
        return getData(requestData, type)
    }

    fun getSupportBandWidthList(frequency: Int): ResponseData<SupportBandwidthList>? {
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = GetSupportBandwidthParam(frequency), serviceId = ServiceIdConstants.GET_SUPPORT_BANDWIDTH_LIST)
        val type: Type = object : TypeToken<ResponseData<SupportBandwidthList>>(){}.type
        return getData(requestData, type)
    }

    fun setWIFI5State(state: Int): ResponseData<BaseResult>? {
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SwitchStateParam(state), serviceId = ServiceIdConstants.SET_WIFI5_STATE)
        return getData(requestData)
    }

    fun setHwNATState(state: Int): ResponseData<BaseResult>? {
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SwitchStateParam(state), serviceId = ServiceIdConstants.SET_HW_NAT_STATE)
        return getData(requestData)
    }

    fun setCurChannel(frequency: Int, channel: Int): ResponseData<BaseResult>? {
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SetChannelParam(frequency, channel), serviceId = ServiceIdConstants.SET_CUR_CHANNEL)
        return getData(requestData)
    }

    fun setCurSignalPower(frequency: Int, signalPower: Int): ResponseData<BaseResult>? {
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SetSignalPowerParam(frequency, signalPower), serviceId = ServiceIdConstants.SET_CUR_SIGNAL_POWER)
        return getData(requestData)
    }

    fun setCurBandwidth(frequency: Int, bandWidth: String): ResponseData<BaseResult>? {
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SetBandWidthParam(frequency, bandWidth), serviceId = ServiceIdConstants.SET_CUR_BANDWIDTH)
        return getData(requestData)
    }

    fun getWIFIMeshInfo(): ResponseData<MeshedNodes>? {
        val type: Type = object : TypeToken<ResponseData<MeshedNodes>>(){}.type
        return getData(ServiceIdConstants.GET_WIFI_MESH_INFO, type)
    }

    fun getSearchedMeshNodes(): ResponseData<SearchedMeshNodes>? {
        val type: Type = object : TypeToken<ResponseData<SearchedMeshNodes>>(){}.type
        return getData(ServiceIdConstants.GET_SEARCHED_MESH_NODES, type)
    }

    fun sendWIFIMesh(ssid: String): ResponseData<BaseResult>? {
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SendWIFIMeshParam(ssid), serviceId = ServiceIdConstants.SEND_WIFI_MESH)
        return getData(requestData)
    }

    fun startOrStopMeshSearch(mode: Int): ResponseData<BaseResult>?{
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = MeshSearchState(mode), serviceId = ServiceIdConstants.SET_SEARCHED_MESH_MODE)
        return getData(requestData)
    }

    fun networkOptimization(): ResponseData<BaseResult>? {
        return getData(ServiceIdConstants.NETWORK_OPTIMIZATION)
    }

    fun resetFactory(): ResponseData<BaseResult>? {
        return getData(ServiceIdConstants.RESET_FACTORY)
    }

    fun rebootDevice(): ResponseData<BaseResult>? {
        return getData(ServiceIdConstants.REBOOT_DEVICE)
    }

    fun passwordAuthentication(password: String): ResponseData<BaseResult>?{
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = AdminPassword(password), serviceId = ServiceIdConstants.PASSWORD_AUTHENTICATION)
        return getData(requestData)
    }

    override fun onMessageReceived(msg: String) {
        val type: Type = object : TypeToken<ResponseData<SearchedMeshNodes>>(){}.type
        var response: ResponseData<SearchedMeshNodes>? = null
        try {
            response = Gson().fromJson(msg, type)
        } catch (e: JsonSyntaxException) {
            e.printStackTrace()
        }
        if (response != null && response.code == 200) {
            mMeshSearchListener?.onMeshInfoResolved(response.data.list)
        }
    }
}