package com.transsion.iot.communication

import android.text.TextUtils
import android.util.Log
import com.google.gson.Gson
import com.google.gson.JsonSyntaxException
import com.google.gson.reflect.TypeToken
import com.transsion.iot.communication.encrypt.AESUtils
import com.transsion.iot.communication.packet.RequestData
import com.transsion.iot.communication.packet.ResponseData
import com.transsion.iot.communication.packet.ServiceIdConstants
import com.transsion.iot.communication.packet.req.*
import com.transsion.iot.communication.packet.resp.*
import com.transsion.iot.communication.tcp.TCPClient
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import org.json.JSONException
import java.lang.reflect.Type

/**
 * @Author fuhuang.fan
 * @Date 2022/7/8/008 17:12
 * @Description
 */
object WIFIClient {

    private val TAG = "WIFIClient"

    private var mScope = GlobalScope

    fun getConnectState(): Int {
        return TCPClient.getConnectState()
    }

    fun isConnected(): Boolean {
        return TCPClient.isConnected()
    }

    fun addConnectListener(listener: ConnectListener) {
        TCPClient.addConnectListener(listener)
    }

    fun removeConnectListener(listener: ConnectListener) {
        TCPClient.removeConnectListener(listener)
    }

    fun connect(ip: String, port: Int) {
        TCPClient.setIP(ip)
        TCPClient.setPort(port)
        TCPClient.connect()
    }

    fun connect() {
        TCPClient.connect()
    }

    fun disconnect() {
        TCPClient.disconnect()
    }

    fun onDestroy() {
        TCPClient.onDestroy()
    }

    private fun <T> sendData(obj: RequestData<T>): String?{
        return if (!isConnected()) {
            Log.d(TAG, "sendData not connected")
            ""
        } else {
            Log.d(TAG, "sendData obj=$obj")
            var jsonStr: String? = null
            try {
                jsonStr = Gson().toJson(obj, RequestData::class.java)
            } catch (e: JSONException) {
                e.printStackTrace()
            }
            Log.d(TAG, "sendData jsonStr=$jsonStr")
            if (TextUtils.isEmpty(jsonStr)) {
                Log.d(TAG, "sendData jsonStr is null")
                ""
            } else {
                val requestStr = AESUtils.encryptAES(jsonStr!!)
                Log.d(TAG, "requestStr encryptAES = $requestStr")
                val resultStr = TCPClient.send(obj.serviceId,requestStr)
                Log.d(TAG, "sendData resultStr=$resultStr")
                resultStr
            }
        }
    }

    private fun sendData(serviceId: String): String? {
        val requestData = RequestData(
            id = System.currentTimeMillis().toString(),
            params = BaseParam(),
            serviceId = serviceId
        )
        return sendData(requestData)
    }

    private fun <T> getData(serviceId: String, type: Type): ResponseData<T>? {
        val responseStr = sendData(serviceId)
        var response : ResponseData<T>? = null
        try {
            response = Gson().fromJson(responseStr, type)
        } catch (e: JsonSyntaxException) {
            e.printStackTrace()
        }
        return response
    }

    private fun <T> getDataAsync(serviceId: String, type: Type, callback: ResponseCallback<T>){
        mScope.launch(Dispatchers.IO) {
            val responseStr = sendData(serviceId)
            var response : ResponseData<T>? = null
            var errMsg: String? = ""
            try {
                response = Gson().fromJson(responseStr, type)
            } catch (e: JsonSyntaxException) {
                e.printStackTrace()
                errMsg = e.message
            }
            withContext(Dispatchers.Main) {
                if (response != null) {
                    if (response.isSuccess()) {
                        callback.onSuccess(response.data)
                    } else {
                        callback.onError(response.code, response.msg)
                    }
                } else {
                    callback.onError(ResponseCode.CODE_FAILED, errMsg)
                }
            }
        }
    }

    private fun <R,T> getData(obj: RequestData<R>, type: Type): ResponseData<T>? {
        val responseStr = sendData(obj)
        var response : ResponseData<T>? = null
        try {
            response = Gson().fromJson(responseStr, type)
        } catch (e: JsonSyntaxException) {
            e.printStackTrace()
        }
        return response
    }

    private fun <R, T> getDataAsync(obj: RequestData<R>, type: Type,  callback: ResponseCallback<T>) {
        mScope.launch(Dispatchers.IO) {
            val responseStr = sendData(obj)
            var response : ResponseData<T>? = null
            var errMsg: String? = ""
            try {
                response = Gson().fromJson(responseStr, type)
            } catch (e: JsonSyntaxException) {
                e.printStackTrace()
                errMsg = e.message
            }
            withContext(Dispatchers.Main) {
                if (response != null) {
                    if (response.isSuccess()) {
                        callback.onSuccess(response.data)
                    } else {
                        callback.onError(response.code, response.msg)
                    }
                } else {
                    callback.onError(ResponseCode.CODE_FAILED, errMsg)
                }
            }
        }
    }

    private fun <T> getData(obj: RequestData<T>): ResponseData<BaseResult>? {
        val responseStr = sendData(obj)
        val type: Type = object : TypeToken<ResponseData<BaseResult>>(){}.type
        var response : ResponseData<BaseResult>? = null
        try {
            response = Gson().fromJson(responseStr, type)
        } catch (e: JsonSyntaxException) {
            e.printStackTrace()
        }
        return response
    }

    private fun <T> getDataAsync(obj: RequestData<T>,  callback: ResponseCallback<BaseResult>) {
        mScope.launch(Dispatchers.IO) {
            val responseStr = sendData(obj)
            val type: Type = object : TypeToken<ResponseData<BaseResult>>(){}.type
            var response : ResponseData<BaseResult>? = null
            var errMsg: String? = ""
            try {
                response = Gson().fromJson(responseStr, type)
            } catch (e: JsonSyntaxException) {
                e.printStackTrace()
                errMsg = e.message
            }
            withContext(Dispatchers.Main) {
                if (response != null) {
                    if (response.isSuccess()) {
                        callback.onSuccess(response.data)
                    } else {
                        callback.onError(response.code, response.msg)
                    }
                } else {
                    callback.onError(ResponseCode.CODE_FAILED, errMsg)
                }
            }
        }
    }

    private fun getData(serviceId: String): ResponseData<BaseResult>?{
        val responseStr = sendData(serviceId)
        val type: Type = object : TypeToken<ResponseData<BaseResult>>(){}.type
        var response : ResponseData<BaseResult>? = null
        try {
            response = Gson().fromJson(responseStr, type)
        } catch (e: JsonSyntaxException) {
            e.printStackTrace()
        }
        return response
    }

    private fun getDataAsync(serviceId: String,  callback: ResponseCallback<BaseResult>) {
        mScope.launch(Dispatchers.IO) {
            val responseStr = sendData(serviceId)
            val type: Type = object : TypeToken<ResponseData<BaseResult>>(){}.type
            var response : ResponseData<BaseResult>? = null
            var errMsg: String? = ""
            try {
                response = Gson().fromJson(responseStr, type)
            } catch (e: JsonSyntaxException) {
                e.printStackTrace()
                errMsg = e.message
            }
            withContext(Dispatchers.Main) {
                if (response != null) {
                    if (response.isSuccess()) {
                        callback.onSuccess(response.data)
                    } else {
                        callback.onError(response.code, response.msg)
                    }
                } else {
                    callback.onError(ResponseCode.CODE_FAILED, errMsg)
                }
            }
        }
    }

    fun <T> serviceWrite(serviceId: String, params: T?): ResponseData<BaseResult>? {
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = params, serviceId = serviceId)
        return getData(requestData)
    }

    fun getCurDeviceNetworkSpeedInfo(): ResponseData<NetworkSpeedInfo>? {
        val type: Type = object : TypeToken<ResponseData<NetworkSpeedInfo>>(){}.type
        return getData(ServiceIdConstants.GET_NETWORK_SPEED_INFO, type)
    }

    fun getCurDeviceNetworkSpeedInfoAsync(callback: ResponseCallback<NetworkSpeedInfo>) {
        val type: Type = object : TypeToken<ResponseData<NetworkSpeedInfo>>(){}.type
        getDataAsync(ServiceIdConstants.GET_NETWORK_SPEED_INFO, type, callback)
    }

    fun getOnlineDeviceCount(): ResponseData<OnlineDeviceCount>? {
        val type: Type = object : TypeToken<ResponseData<OnlineDeviceCount>>(){}.type
        return getData(ServiceIdConstants.GET_ONLINE_DEVICE_COUNT, type)
    }

    fun getOnlineDeviceCountAsync(callback: ResponseCallback<OnlineDeviceCount>){
        val type: Type = object : TypeToken<ResponseData<OnlineDeviceCount>>(){}.type
        getDataAsync(ServiceIdConstants.GET_ONLINE_DEVICE_COUNT, type, callback)
    }


    fun getDeviceBasicInfo(): ResponseData<DeviceBasicInfo>? {
        val type: Type = object : TypeToken<ResponseData<DeviceBasicInfo>>(){}.type
        return getData(ServiceIdConstants.GET_DEVICE_BASIC_INFO, type)
    }

    fun getDeviceBasicInfoAsync(callback: ResponseCallback<DeviceBasicInfo>){
        val type: Type = object : TypeToken<ResponseData<DeviceBasicInfo>>(){}.type
        getDataAsync(ServiceIdConstants.GET_DEVICE_BASIC_INFO, type, callback)
    }

    fun getConnectedDeviceList(): ResponseData<ConnectedDeviceList>? {
        val type: Type = object : TypeToken<ResponseData<ConnectedDeviceList>>(){}.type
        return getData(ServiceIdConstants.GET_CONNECTED_DEVICE_LIST, type)
    }

    fun getConnectedDeviceListAsync(callback: ResponseCallback<ConnectedDeviceList>){
        val type: Type = object : TypeToken<ResponseData<ConnectedDeviceList>>(){}.type
        getDataAsync(ServiceIdConstants.GET_CONNECTED_DEVICE_LIST, type, callback)
    }

    fun disableDevice(deviceMac: String): ResponseData<BaseResult>? {
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = DisableDeviceParam(deviceMac), serviceId = ServiceIdConstants.DISABLE_DEVICE)
        return getData(requestData)
    }

    fun disableDeviceAsync(deviceMac: String, callback: ResponseCallback<BaseResult>){
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = DisableDeviceParam(deviceMac), serviceId = ServiceIdConstants.DISABLE_DEVICE)
        getDataAsync(requestData, callback)
    }

    fun enableDevice(deviceMac: String): ResponseData<BaseResult>? {
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = DisableDeviceParam(deviceMac), serviceId = ServiceIdConstants.ENABLE_DEVICE)
        return getData(requestData)
    }

    fun enableDeviceAsync(deviceMac: String, callback: ResponseCallback<BaseResult>){
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = DisableDeviceParam(deviceMac), serviceId = ServiceIdConstants.ENABLE_DEVICE)
        getDataAsync(requestData, callback)
    }

    fun getNetworkSettings(): ResponseData<NetworkSettings>? {
        val type: Type = object : TypeToken<ResponseData<NetworkSettings>>(){}.type
        return getData(ServiceIdConstants.GET_NETWORK_SETTINGS, type)
    }

    fun getNetworkSettingsAsync(callback: ResponseCallback<NetworkSettings>){
        val type: Type = object : TypeToken<ResponseData<NetworkSettings>>(){}.type
        getDataAsync(ServiceIdConstants.GET_NETWORK_SETTINGS, type, callback)
    }

    fun setSSID(frequency: Int, ssid : String): ResponseData<BaseResult>? {
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SetSSIDParam(frequency, ssid), serviceId = ServiceIdConstants.SET_SSID)
        return getData(requestData)
    }

    fun setSSIDAsync(frequency: Int, ssid : String, callback: ResponseCallback<BaseResult>){
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SetSSIDParam(frequency, ssid), serviceId = ServiceIdConstants.SET_SSID)
        getDataAsync(requestData, callback)
    }

    fun setPassword(frequency: Int, password : String): ResponseData<BaseResult>? {
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SetPasswordParam(frequency, password), serviceId = ServiceIdConstants.SET_PASSWORD)
        return getData(requestData)
    }

    fun setPasswordAsync(frequency: Int, password : String, callback: ResponseCallback<BaseResult>){
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SetPasswordParam(frequency, password), serviceId = ServiceIdConstants.SET_PASSWORD)
        getDataAsync(requestData, callback)
    }

    fun setWIFIVisibility(frequency: Int, visible: Int): ResponseData<BaseResult>? {
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SetVisibleParam(frequency, visible), serviceId = ServiceIdConstants.SET_WIFI_VISIBILITY)
        return getData(requestData)
    }

    fun setWIFIVisibilityAsync(frequency: Int, visible: Int, callback: ResponseCallback<BaseResult>){
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SetVisibleParam(frequency, visible), serviceId = ServiceIdConstants.SET_WIFI_VISIBILITY)
        getDataAsync(requestData, callback)
    }

    //关闭双频合一WIFI之前需要先判断mesh状态，mesh停止才能关
    fun setWIFIState(frequency: Int, workMode: Int): ResponseData<BaseResult>? {
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SetWIFIStateParam(frequency, workMode), serviceId = ServiceIdConstants.SET_WIFI_STATE)
        return getData(requestData)
    }

    fun setWIFIStateAsync(frequency: Int, workMode: Int, callback: ResponseCallback<BaseResult>){
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SetWIFIStateParam(frequency, workMode), serviceId = ServiceIdConstants.SET_WIFI_STATE)
        getDataAsync(requestData, callback)
    }

    fun setWIFICombineState(wifiCombineMode: Int): ResponseData<BaseResult>? {
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SetWIFICombineParam(wifiCombineMode), serviceId = ServiceIdConstants.SET_WIFI_COMBINE_STATE)
        return getData(requestData)
    }

    fun setWIFICombineStateAsync(wifiCombineMode: Int, callback: ResponseCallback<BaseResult>){
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SetWIFICombineParam(wifiCombineMode), serviceId = ServiceIdConstants.SET_WIFI_COMBINE_STATE)
        getDataAsync(requestData, callback)
    }

    fun getSupportChannelList(frequency: Int): ResponseData<SupportChannelList>? {
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = GetSupportChannelParam(frequency), serviceId = ServiceIdConstants.GET_SUPPORT_CHANNEL_LIST)
        val type: Type = object : TypeToken<ResponseData<SupportChannelList>>(){}.type
        return getData(requestData, type)
    }

    fun getSupportChannelListAsync(frequency: Int, callback: ResponseCallback<SupportChannelList>){
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = GetSupportChannelParam(frequency), serviceId = ServiceIdConstants.GET_SUPPORT_CHANNEL_LIST)
        val type: Type = object : TypeToken<ResponseData<SupportChannelList>>(){}.type
        getDataAsync(requestData, type, callback)
    }

    fun getSupportBandWidthList(frequency: Int): ResponseData<SupportBandwidthList>? {
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = GetSupportBandwidthParam(frequency), serviceId = ServiceIdConstants.GET_SUPPORT_BANDWIDTH_LIST)
        val type: Type = object : TypeToken<ResponseData<SupportBandwidthList>>(){}.type
        return getData(requestData, type)
    }

    fun getSupportBandWidthListAsync(frequency: Int, callback: ResponseCallback<SupportBandwidthList>){
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = GetSupportChannelParam(frequency), serviceId = ServiceIdConstants.GET_SUPPORT_BANDWIDTH_LIST)
        val type: Type = object : TypeToken<ResponseData<SupportBandwidthList>>(){}.type
        getDataAsync(requestData, type, callback)
    }

    fun setWIFI5State(state: Int): ResponseData<BaseResult>? {
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SwitchStateParam(state), serviceId = ServiceIdConstants.SET_WIFI5_STATE)
        return getData(requestData)
    }

    fun setWIFI5StateAsync(state: Int, callback: ResponseCallback<BaseResult>){
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SwitchStateParam(state), serviceId = ServiceIdConstants.SET_WIFI5_STATE)
        getDataAsync(requestData, callback)
    }

    fun setHwNATState(state: Int): ResponseData<BaseResult>? {
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SwitchStateParam(state), serviceId = ServiceIdConstants.SET_HW_NAT_STATE)
        return getData(requestData)
    }

    fun setHwNATStateAsync(state: Int, callback: ResponseCallback<BaseResult>){
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SwitchStateParam(state), serviceId = ServiceIdConstants.SET_HW_NAT_STATE)
        getDataAsync(requestData, callback)
    }

    fun setCurChannel(frequency: Int, channel: Int): ResponseData<BaseResult>? {
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SetChannelParam(frequency, channel), serviceId = ServiceIdConstants.SET_CUR_CHANNEL)
        return getData(requestData)
    }

    fun setCurChannelAsync(frequency: Int, channel: Int, callback: ResponseCallback<BaseResult>){
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SetChannelParam(frequency, channel), serviceId = ServiceIdConstants.SET_CUR_CHANNEL)
        getDataAsync(requestData, callback)
    }

    fun setCurSignalPower(signalPower: Int): ResponseData<BaseResult>? {
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SetSignalPowerParam(signalPower), serviceId = ServiceIdConstants.SET_CUR_SIGNAL_POWER)
        return getData(requestData)
    }

    fun setCurSignalPowerAsync(signalPower: Int, callback: ResponseCallback<BaseResult>){
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SetSignalPowerParam(signalPower), serviceId = ServiceIdConstants.SET_CUR_SIGNAL_POWER)
        getDataAsync(requestData, callback)
    }

    fun setCurBandwidth(frequency: Int, bandWidth: String): ResponseData<BaseResult>? {
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SetBandWidthParam(frequency, bandWidth), serviceId = ServiceIdConstants.SET_CUR_BANDWIDTH)
        return getData(requestData)
    }

    fun setCurBandwidthAsync(frequency: Int, bandWidth: String, callback: ResponseCallback<BaseResult>){
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SetBandWidthParam(frequency, bandWidth), serviceId = ServiceIdConstants.SET_CUR_BANDWIDTH)
        getDataAsync(requestData, callback)
    }

    fun getWIFIMeshInfo(): ResponseData<MeshedNodes>? {
        val type: Type = object : TypeToken<ResponseData<MeshedNodes>>(){}.type
        return getData(ServiceIdConstants.GET_WIFI_MESH_INFO, type)
    }

    fun getWIFIMeshInfoAsync(callback: ResponseCallback<MeshedNodes>){
        val type: Type = object : TypeToken<ResponseData<MeshedNodes>>(){}.type
        getDataAsync(ServiceIdConstants.GET_WIFI_MESH_INFO, type, callback)
    }

    fun getSearchedMeshNodes(): ResponseData<SearchedMeshNodes>? {
        val type: Type = object : TypeToken<ResponseData<SearchedMeshNodes>>(){}.type
        return getData(ServiceIdConstants.GET_SEARCHED_MESH_NODES, type)
    }

    fun getSearchedMeshNodesAsync(callback: ResponseCallback<SearchedMeshNodes>){
        val type: Type = object : TypeToken<ResponseData<SearchedMeshNodes>>(){}.type
        getDataAsync(ServiceIdConstants.GET_SEARCHED_MESH_NODES, type, callback)
    }

    fun sendWIFIMesh(ssid: String): ResponseData<BaseResult>? {
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SendWIFIMeshParam(ssid), serviceId = ServiceIdConstants.SEND_WIFI_MESH)
        return getData(requestData)
    }

    fun sendWIFIMeshAsync(ssid: String, callback: ResponseCallback<BaseResult>){
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = SendWIFIMeshParam(ssid), serviceId = ServiceIdConstants.SEND_WIFI_MESH)
        getDataAsync(requestData, callback)
    }

    fun startOrStopMeshSearch(mode: Int): ResponseData<BaseResult>?{
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = MeshSearchState(mode), serviceId = ServiceIdConstants.SET_SEARCHED_MESH_MODE)
        return getData(requestData)
    }

    fun startOrStopMeshSearchAsync(mode: Int, callback: ResponseCallback<BaseResult>){
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = MeshSearchState(mode), serviceId = ServiceIdConstants.SET_SEARCHED_MESH_MODE)
        getDataAsync(requestData, callback)
    }

    fun getWifiMeshMode(): ResponseData<MeshSearchState>? {
        val type: Type = object : TypeToken<ResponseData<MeshSearchState>>(){}.type
        return getData(ServiceIdConstants.GET_WIFI_MESH_MODE, type)
    }

    fun getWifiMeshModeAsync(callback: ResponseCallback<MeshSearchState>){
        val type: Type = object : TypeToken<ResponseData<MeshSearchState>>(){}.type
        getDataAsync(ServiceIdConstants.GET_WIFI_MESH_MODE, type, callback)
    }

    fun networkOptimization(): ResponseData<BaseResult>? {
        return getData(ServiceIdConstants.NETWORK_OPTIMIZATION)
    }

    fun networkOptimizationAsync(callback: ResponseCallback<BaseResult>) {
        getDataAsync(ServiceIdConstants.NETWORK_OPTIMIZATION, callback)
    }

    fun resetFactory(): ResponseData<BaseResult>? {
        return getData(ServiceIdConstants.RESET_FACTORY)
    }

    fun resetFactoryAsync(callback: ResponseCallback<BaseResult>) {
        getDataAsync(ServiceIdConstants.RESET_FACTORY, callback)
    }

    fun rebootDevice(): ResponseData<BaseResult>? {
        return getData(ServiceIdConstants.REBOOT_DEVICE)
    }

    fun rebootDeviceAsync(callback: ResponseCallback<BaseResult>) {
        getDataAsync(ServiceIdConstants.REBOOT_DEVICE, callback)
    }

    fun passwordAuthentication(password: String): ResponseData<BaseResult>?{
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = AdminPassword(password), serviceId = ServiceIdConstants.PASSWORD_AUTHENTICATION)
        return getData(requestData)
    }


    fun passwordAuthenticationAsync(password: String, callback: ResponseCallback<BaseResult>){
        val requestData = RequestData(id = System.currentTimeMillis().toString(), params = AdminPassword(password), serviceId = ServiceIdConstants.PASSWORD_AUTHENTICATION)
        getDataAsync(requestData, callback)
    }

    fun cancelRequest(serviceId: String) {
        TCPClient.consumeChannel(serviceId)
        TCPClient.consumeJob(serviceId)
    }
}