package com.transsion.iot.communication.tcp

import android.os.Handler
import android.os.HandlerThread
import android.os.Message
import android.util.Log
import com.google.gson.Gson
import com.transsion.iot.communication.ConnectConstants
import com.transsion.iot.communication.ConnectListener
import com.transsion.iot.communication.ResponseCode
import com.transsion.iot.communication.encrypt.AESUtils
import com.transsion.iot.communication.packet.ResponseData
import com.transsion.iot.communication.packet.ServiceIdConstants
import com.transsion.iot.communication.packet.resp.BaseResult
import kotlinx.coroutines.*
import kotlinx.coroutines.channels.Channel
import org.json.JSONException
import org.json.JSONObject
import java.io.IOException
import java.io.InputStream
import java.io.OutputStream
import java.net.Socket
import java.util.concurrent.CopyOnWriteArrayList
import java.util.concurrent.atomic.AtomicBoolean

object TCPClient {
    private val TAG = TCPClient::class.java.simpleName
    private var ip = "192.168.188.1"
    private var port = 8889
    const val HEART_BEAT_INTERVAL = 30 * 1000L
    private const val REQUEST_TIME_OUT = 10 * 1000L
//    private const val MSG_WHAT_SEND_DATA = 100
//    private const val SEND_DATA_DELAYED = 100L
    private const val MAX_RECONNECT_COUNT = 3
    private var reconnectCount = 0
    lateinit var mSocket: Socket
    private var mOutStream: OutputStream? = null
    private var mInStream: InputStream? = null
    private var mConnectThread: SocketConnectThread? = null
    private var mReceiveThread: SocketReceiveThread? = null
    private var mHeartBeatThread: HandlerThread? = null
    private var mHeartBeatHandler: Handler? = null
//    private var mSendThread: HandlerThread? = null
//    private var mSendHandler: Handler? = null
    private var mChannelMap = HashMap<String, Channel<String>>()
    private var mJobMap = HashMap<String, Job>()
    private var mConnectState = ConnectConstants.STATE_DISCONNECTED
    private var mConnectListeners : CopyOnWriteArrayList<ConnectListener> = CopyOnWriteArrayList()
    private var mHeartBeatJob: Job? = null

    private var mHeartBeatRunnable = object : Runnable {
        override fun run() {
            sendHeartBeat()
            mHeartBeatHandler?.postDelayed(this, HEART_BEAT_INTERVAL)
        }
    }

    private fun initSendThread() {
//        mSendThread = HandlerThread("send")
//        mSendThread?.start()

        mHeartBeatThread = HandlerThread("heart-beat")
        mHeartBeatThread?.start()
    }

    private fun initReceiveThread() {
        mReceiveThread = SocketReceiveThread()
        mReceiveThread?.start()
    }

    private fun initSendHandler() {
//        mSendThread?.let {
//            mSendHandler = Handler(it.looper) { message ->
//                val data = message.obj.toString()
//                writeToOutputStream(data.toByteArray())
//                true
//            }
//        }
        mHeartBeatThread?.let {
            mHeartBeatHandler = Handler(it.looper)
            mHeartBeatHandler?.post(mHeartBeatRunnable)
        }
    }

    fun setIP(ip: String) {
        this.ip = ip
    }

    fun setPort(port: Int) {
        this.port = port
    }

    fun addConnectListener(connectListener: ConnectListener) {
        if (!mConnectListeners.contains(connectListener)) {
            mConnectListeners.add(connectListener)
        }
    }

    fun removeConnectListener(connectListener: ConnectListener) {
        if (mConnectListeners.contains(connectListener)) {
            mConnectListeners.remove(connectListener)
        }
    }

    private fun onConnectStateChanged(state: Int) {
        for (listener in mConnectListeners) {
            listener.onConnectStateChanged(state)
        }
    }

    private fun onError(errMsg: String) {
        for (listener in mConnectListeners) {
            listener.onError(errMsg)
        }
    }

    fun getConnectState(): Int {
        return mConnectState
    }

    fun isConnected(): Boolean {
        return mConnectState == ConnectConstants.STATE_CONNECTED
    }

    fun connect() {
        Log.i(TAG, "startConnect")
        if (ip.isEmpty() || port <= 0) {
            onError("ip or port is invalid")
            return
        }
        mConnectThread = SocketConnectThread()
        mConnectThread?.start()
        initSendThread()
        mConnectState = ConnectConstants.STATE_CONNECTING
        onConnectStateChanged(ConnectConstants.STATE_CONNECTING)
    }

    fun reConnect() {
        Log.i(TAG, "start reConnect")
        if (reconnectCount >= MAX_RECONNECT_COUNT) {
            Log.i(TAG, "reconnect  failed  $MAX_RECONNECT_COUNT times")
            return
        }
        reconnectCount++
        disconnect()
        connect()
    }

    fun send(serviceId: String, message: String): String? {
        return runBlocking(Dispatchers.IO){
            val channel = Channel<String>(1)
            mChannelMap[serviceId] = channel
            val job = GlobalScope.launch {
                delay(REQUEST_TIME_OUT)
                mChannelMap[serviceId]?.send(Gson().toJson(ResponseData(ResponseCode.CODE_TIME_OUT, BaseResult(), System.currentTimeMillis().toString(), serviceId, "time out")))
                consumeChannel(serviceId)
            }
            mJobMap[serviceId] = job
            writeToOutputStream(message.toByteArray())
//            val msg = Message()
//            msg.what = MSG_WHAT_SEND_DATA
//            msg.obj = message
//            mSendHandler?.sendMessageDelayed(msg, SEND_DATA_DELAYED)
            mChannelMap[serviceId]?.receive()

        }
    }

    fun sendHeartBeat() {
        mHeartBeatJob = GlobalScope.launch {
            delay(HEART_BEAT_INTERVAL * 2)
            disconnect()
        }
        writeToOutputStream(ServiceIdConstants.HEART_BEAT_PING.toByteArray())
//        val msg = Message()
//        msg.what = MSG_WHAT_SEND_DATA
//        msg.obj = ServiceIdConstants.HEART_BEAT_PING
//        mSendHandler?.sendMessageDelayed(msg, SEND_DATA_DELAYED)
    }

    private fun writeToOutputStream(data: ByteArray) {
        if (data.isEmpty()) return
        Log.d(TAG, "writeToOutputStream data=${String(data)}")
        try {
            mOutStream?.write(data)
            mOutStream?.flush()
        } catch (e: Exception) {
            e.printStackTrace()
            onError("send failed:${e.message}")
            reConnect()
        }
    }

    private fun parseToRespData(receiveData: String) {
        if (receiveData == ServiceIdConstants.HEART_BEAT_PONG) {
            mHeartBeatJob?.cancel()
            return
        }
        val responseStr = AESUtils.decryptAES(receiveData)
        Log.d(TAG, "parseToRespData responseStr=$responseStr")
        var serviceId: String = ServiceIdConstants.GET_DEVICE_BASIC_INFO
        try {
            val obj = JSONObject(responseStr)
            serviceId = obj.optString("serviceId")
        } catch (e: JSONException) {
            e.printStackTrace()
        }
        Log.d(TAG, "parseToRespData serviceId=$serviceId")
        Log.d(TAG, "parseToRespData mChannelMap=${mChannelMap.keys}")
//        if (serviceId == ServiceIdConstants.GET_SEARCHED_MESH_NODES) {
//            mMessageReceiveListener?.onMessageReceived(receiveData)
//            return
//        }
        GlobalScope.launch {
            mChannelMap[serviceId]?.send(responseStr)
            consumeChannel(serviceId)
            consumeJob(serviceId)
            //重置心跳
            mHeartBeatHandler?.removeCallbacks(mHeartBeatRunnable)
            mHeartBeatJob?.cancel()
            mHeartBeatHandler?.postDelayed(mHeartBeatRunnable, HEART_BEAT_INTERVAL)
        }
    }

    fun disconnect() {
        if (!isConnected()) {
            return
        }
        mConnectThread?.isInterrupted
        mReceiveThread?.threadExit()
        mConnectState = ConnectConstants.STATE_DISCONNECTED
        onConnectStateChanged(ConnectConstants.STATE_DISCONNECTED)
        mHeartBeatHandler?.removeCallbacks(mHeartBeatRunnable)
        mHeartBeatThread?.quitSafely()
//        mSendHandler?.removeMessages(MSG_WHAT_SEND_DATA)
        clearChannelMap()
        clearJobMap()
        try {
            mOutStream?.close() //关闭输出流
            mInStream?.close() //关闭输入流
            mSocket.close() //关闭socket
        } catch (e: IOException) {
            e.printStackTrace()
        }
    }

    fun onDestroy() {
        disconnect()
        mConnectListeners.clear()
    }

    fun consumeChannel(serviceId: String) {
        mChannelMap.remove(serviceId)
    }

    fun consumeJob(serviceId: String) {
        if (mJobMap[serviceId]?.isCompleted == false) {
            mJobMap[serviceId]?.cancel()
        }
        mJobMap.remove(serviceId)
    }

    private fun clearChannelMap() {
        if (mChannelMap.isNotEmpty()) {
            mChannelMap.clear()
        }
    }

    private fun clearJobMap() {
        if (mJobMap.isNotEmpty()) {
            for (key in mJobMap.keys) {
                if (mJobMap[key]?.isCompleted == false) {
                    mJobMap[key]?.cancel()
                }
            }
            mJobMap.clear()
        }
    }

    class SocketConnectThread : Thread() {
        override fun run() {
            try {
                mSocket = Socket(ip, port)
                mOutStream = mSocket.getOutputStream()
                mInStream = mSocket.getInputStream()
            } catch (e: Exception) {
                e.printStackTrace()
                onError("connect failed:${e.message}")
                mConnectState = ConnectConstants.STATE_DISCONNECTED
                onConnectStateChanged(ConnectConstants.STATE_DISCONNECTED)
                return
            }
            Log.i(TAG, "connect success")
            mConnectState = ConnectConstants.STATE_CONNECTED
            onConnectStateChanged(ConnectConstants.STATE_CONNECTED)
            reconnectCount = 0
            initReceiveThread()
            initSendHandler()
        }
    }

    class SocketReceiveThread : Thread() {
        private var threadExit = AtomicBoolean(false)
        override fun run() {
            val buffer = ByteArray(40 * 1024)
            while (!threadExit.get()) {
                try {
                    //读取数据，返回值表示读到的数据长度。-1表示结束
                    val count = mInStream?.read(buffer)
                    if (count == null || count == -1) {
                        Log.i(TAG, "read count $count")
                        onError("receive failed: -1")
                        break
                    } else {
                        val receiveData = String(buffer, 0, count)
                        Log.i(TAG, "read buffer:$receiveData,count=$count")
                        parseToRespData(receiveData)
                    }
                } catch (e: IOException) {
                    e.printStackTrace()
                    onError("receive failed: ${e.message}")
                    reConnect()
                    threadExit()
                }
            }
        }

        fun threadExit() {
            threadExit.set(true)
        }
    }
}