package com.transsion.iot.communication.udp

import android.util.Log
import com.google.gson.Gson
import com.google.gson.JsonSyntaxException
import com.transsion.iot.communication.packet.resp.BasicInfo
import java.io.IOException
import java.net.DatagramPacket
import java.net.DatagramSocket
import java.net.InetSocketAddress
import java.util.concurrent.CopyOnWriteArrayList
import java.util.concurrent.atomic.AtomicBoolean

/**
 * @Author fuhuang.fan
 * @Date 2022/7/19/019 15:05
 * @Description
 */
object UDPClient {

    const val BROADCAST_PORT = 6000

    private var mReceiveThread: SocketReceiveThread? = null
    private var mPort = BROADCAST_PORT
    private var mSocket: DatagramSocket? = null
    private var mMessageReceivers: CopyOnWriteArrayList<IUDPMessageReceiver> = CopyOnWriteArrayList()

    fun registerMessageReceiver(receiver: IUDPMessageReceiver) {
        if (!mMessageReceivers.contains(receiver)) {
            mMessageReceivers.add(receiver)
        }
    }

    fun unregisterMessageReceiver(receiver: IUDPMessageReceiver) {
        if (mMessageReceivers.contains(receiver)) {
            mMessageReceivers.remove(receiver)
        }
    }

    private fun onReceiveMessage(basicInfo: BasicInfo?) {
        for (receiver in mMessageReceivers) {
            receiver.onMessageReceived(basicInfo)
        }
    }

    private fun onReceiveError(errMsg: String?) {
        for (receiver in mMessageReceivers) {
            receiver.onErrorReceived(errMsg)
        }
    }

    fun startReceive(port: Int = BROADCAST_PORT) {
        mPort = port
        try {
            mSocket = DatagramSocket(null)
            mSocket?.reuseAddress = true
            mSocket?.bind(InetSocketAddress(mPort))
            mReceiveThread = SocketReceiveThread()
            mReceiveThread?.start()
        } catch (e: Exception) {
            onReceiveError(e.message)
        }
    }

    fun stopReceive() {
        mReceiveThread?.threadExit()
        try {
            if (mSocket?.isClosed == false){
                mSocket?.close()
                mSocket = null
            }
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    fun onDestroy() {
        try {
            if (mSocket?.isClosed == false){
                mSocket?.close()
                mSocket = null
            }
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    class SocketReceiveThread : Thread() {
        private var threadExit = AtomicBoolean(false)
        override fun run() {
            val buffer = ByteArray(1024)
            while (!threadExit.get()) {
                try {
                    val datagramPacket = DatagramPacket(buffer, buffer.size)
                    mSocket?.receive(datagramPacket)
                    val receiveData = String(buffer, 0, datagramPacket.length)
                    Log.d("UDPClient", "receiveData=$receiveData")
                    val basicInfo = Gson().fromJson(receiveData, BasicInfo::class.java)
                    Log.d("UDPClient", "basicInfo=$basicInfo")
                    onReceiveMessage(basicInfo)
                } catch (e: IOException) {
                    e.printStackTrace()
                    onReceiveError(e.message)
                    threadExit()
                } catch (e: JsonSyntaxException) {
                    e.printStackTrace()
                    onReceiveError(e.message)
                    threadExit()
                }
            }
        }

        fun threadExit() {
            threadExit.set(true)
        }
    }
}