package com.transsion.iot.communication.http.repository

import android.content.Context
import android.text.TextUtils
import android.util.Log
import com.transsion.iot.communication.http.*
import com.transsion.iot.communication.http.api.DeviceControlService
import com.transsion.iot.communication.http.bean.BaseInfo
import com.transsion.iot.communication.http.bean.DeviceRequestInfo
import com.transsion.iot.communication.http.bean.StateInfo
import com.transsion.iot.communication.packet.ServiceIdConstants
import com.transsion.iot.communication.packet.req.BaseParam
import com.transsion.iot.communication.packet.resp.*
import kotlinx.coroutines.CancellationException
import kotlinx.coroutines.sync.Mutex
import okhttp3.ResponseBody
import org.json.JSONException
import org.json.JSONObject
import java.io.IOException
import java.nio.charset.Charset
import java.util.*

object DeviceControlRepository {

    const val TAG = "DeviceControlRepository"
    private var mClient: DeviceControlService? = null
    private val mMutex = Mutex()

    private suspend fun getDeviceControlService(context: Context, token: String): DeviceControlService {
        mMutex.lock()
        if (mClient == null) {
            val retrofit = ServiceLoader.load(IRetrofitClientSpi::class.java).first().getRetrofit(context, token)
            Log.d(TAG, "getDeviceControlService retrofit=${retrofit}")
            Log.d(TAG, "getDeviceControlService baseUrl=${retrofit.baseUrl()}")
            mClient = retrofit.create(DeviceControlService::class.java)
        }
        mMutex.unlock()
        Log.d(TAG, "getDeviceControlService client=$mClient")
        return mClient!!
    }

    suspend fun getBaseInfo(context: Context, token: String, pid: String, psn: String, result:(result:BaseInfo?)->Unit) {
        Log.d(TAG, "token=$token,pid=$pid, psn=$psn")
        HttpRequest({
            val map = TreeMap<String, Any?>()
            map["id"] = System.currentTimeMillis().toString()
            map["version"] = "1.0"
            map["params"] = arrayOf("baseInf")
            map["method"] = "app.property.get"
            Log.d(TAG, "token=$token,pid=$pid, psn=$psn, body=${map.getRequestBody()}")
            val result = getDeviceControlService(context, token).getBaseInfo(pid, psn, map.getRequestBody())
            Log.d(TAG, "result:${result}")
            result
        },{
            Log.d(TAG, "baseInfo:${it?.baseInf}")
            it?.let {
                result.invoke(it.baseInf)
            }
            if (it == null) {
                result.invoke(null)
            }
        },{
            Log.e(TAG, "error:${it.errMsg}, ${it.code}")
            result.invoke(null)
        })
    }

    suspend fun getWifiInfo(context: Context, token: String, pid: String, psn: String, result:(result:Settings?)->Unit) {
        HttpRequest({
            val map = TreeMap<String, Any?>()
            map["id"] = System.currentTimeMillis().toString()
            map["version"] = "1.0"
            map["params"] = arrayOf("wifiInfo")
            map["method"] = "app.property.get"
            getDeviceControlService(context, token).getWifiInfo(pid, psn, map.getRequestBody())
        },{
            Log.d(TAG, "wifiInfo:${it?.wifiInfo}")
            it?.let {
                result.invoke(it.wifiInfo)
            }
            if (it == null) {
                result.invoke(null)
            }
        },{
            Log.e(TAG, "error:${it.errMsg}, ${it.code}")
            result.invoke(null)
        })
    }

    suspend fun getMeshInfo(context: Context, token: String, pid: String, psn: String, result:(result:List<MeshNode>?)->Unit) {
        try {
            val map = TreeMap<String, Any?>()
            map["id"] = System.currentTimeMillis().toString()
            map["version"] = "1.0"
            map["params"] = arrayOf("meshInf")
            map["method"] = "app.property.get"
            val data = getDeviceControlService(context, token).getMeshInfo(pid, psn, map.getRequestBody())
            if (data == null) {
                result.invoke(null)
            } else {
                val responseInfo = getResponseInfo(data)
                Log.d(TAG, "responseInfo:${responseInfo}")
                val list = parseMeshInfo(responseInfo)
                result.invoke(list)
            }
        } catch (eThrowable: Throwable) {
            if (eThrowable !is CancellationException) {
                Log.e("HttpRequest", "errorCallBack : ${eThrowable.message}")
            }
            result.invoke(null)
        }

    }

    suspend fun getMeshInfoSync(context: Context, token: String, pid: String, psn: String) : List<MeshNode>? {
        try {
            val map = TreeMap<String, Any?>()
            map["id"] = System.currentTimeMillis().toString()
            map["version"] = "1.0"
            map["params"] = arrayOf("meshInf")
            map["method"] = "app.property.get"
            val data = getDeviceControlService(context, token).getMeshInfo(pid, psn, map.getRequestBody())
            return if (data == null) {
                null
            } else {
                val responseInfo = getResponseInfo(data)
                Log.d(TAG, "responseInfo:${responseInfo}")
                val list = parseMeshInfo(responseInfo)
                list
            }
        } catch (eThrowable: Throwable) {
            if (eThrowable !is CancellationException) {
                Log.e("HttpRequest", "errorCallBack : ${eThrowable.message}")
            }
            return null
        }
    }

    suspend fun getTerminalInfo(context: Context, token: String, pid: String, psn: String, result:(result:List<ConnectedDevice>?)->Unit) {
        try {
            val map = TreeMap<String, Any?>()
            map["id"] = System.currentTimeMillis().toString()
            map["version"] = "1.0"
            map["params"] = arrayOf("clientInf")
            map["method"] = "app.property.get"
            val data = getDeviceControlService(context, token).getClientInfo(pid, psn, map.getRequestBody())
            Log.d(TAG, "data:${data}")
            if (data == null) {
                result.invoke(null)
            } else {
                val responseInfo = getResponseInfo(data)
                Log.d(TAG, "responseInfo:${responseInfo}")
                val list = parseTerminalInfo(responseInfo)
                result.invoke(list)
            }
        } catch (eThrowable: Throwable) {
            if (eThrowable !is CancellationException) {
                Log.e("HttpRequest", "errorCallBack : ${eThrowable.message}")
            }
            result.invoke(null)
        }
    }

    suspend fun getTerminalInfoSync(context: Context, token: String, pid: String, psn: String) : List<ConnectedDevice>?{
        try {
            val map = TreeMap<String, Any?>()
            map["id"] = System.currentTimeMillis().toString()
            map["version"] = "1.0"
            map["params"] = arrayOf("clientInf")
            map["method"] = "app.property.get"
            val data = getDeviceControlService(context, token).getClientInfo(pid, psn, map.getRequestBody())
            Log.d(TAG, "data:${data}")
            return if (data == null) {
                null
            } else {
                val responseInfo = getResponseInfo(data)
                Log.d(TAG, "responseInfo:${responseInfo}")
                val list = parseTerminalInfo(responseInfo)
                list
            }
        } catch (eThrowable: Throwable) {
            if (eThrowable !is CancellationException) {
                Log.e("HttpRequest", "errorCallBack : ${eThrowable.message}")
            }
            return null
        }
    }


    suspend fun <T> serviceWrite(context: Context, token: String, pid: String, psn: String,serviceId: String , params: T, result:(result:Boolean)->Unit) {
        HttpRequest({
            val map = TreeMap<String, Any?>()
            map["id"] = System.currentTimeMillis().toString()
            map["version"] = "1.0"
            map["params"] = params
            map["method"] = "service"
            map["serviceId"] = serviceId
            Log.d(TAG, "token=$token,pid=$pid, psn=$psn, body=${map.getRequestBody()}")
            getDeviceControlService(context, token).serviceSet(pid, psn, map.getRequestBody())
        },{
            result.invoke(true)
        },{
            Log.e(TAG, "error:${it.errMsg}, ${it.code}")
            result.invoke(false)
        })
    }

    suspend fun getSearchedMeshNodes(context: Context, token: String, pid: String, psn: String, result:(result:List<SearchedNode>?)->Unit) {
        HttpRequest({
            val map = TreeMap<String, Any?>()
            map["id"] = System.currentTimeMillis().toString()
            map["version"] = "1.0"
            map["params"] = BaseParam()
            map["method"] = "service"
            map["serviceId"] = ServiceIdConstants.GET_SEARCHED_MESH_NODES
            Log.d(TAG, "token=$token,pid=$pid, psn=$psn, body=${map.getRequestBody()}")
            getDeviceControlService(context, token).getSearchedMeshNodes(pid, psn, map.getRequestBody())
        },{
            it?.let {
                result.invoke(it.list)
            }
            if (it == null) {
                result.invoke(null)
            }
        },{
            Log.e(TAG, "error:${it.errMsg}, ${it.code}")
            result.invoke(null)
        })
    }

    suspend fun getWifiMeshMode(context: Context, token: String, pid: String, psn: String, result:(result: Int?)->Unit) {
        HttpRequest({
            val map = TreeMap<String, Any?>()
            map["id"] = System.currentTimeMillis().toString()
            map["version"] = "1.0"
            map["params"] = BaseParam()
            map["method"] = "service"
            map["serviceId"] = ServiceIdConstants.GET_WIFI_MESH_MODE
            Log.d(TAG, "token=$token,pid=$pid, psn=$psn, body=${map.getRequestBody()}")
            getDeviceControlService(context, token).getWifiMeshMode(pid, psn, map.getRequestBody())
        },{
            it?.let {
                result.invoke(it.mode)
            }
            if (it == null) {
                result.invoke(null)
            }
        },{
            Log.e(TAG, "error:${it.errMsg}, ${it.code}")
            result.invoke(null)
        })
    }

    suspend fun getDeviceStateInfo(context: Context, token: String, pid: String, psn: String, result:(result: List<StateInfo>?)->Unit) {
        HttpRequest({
            val map = TreeMap<String, Any?>()
            map["list"] = arrayListOf(DeviceRequestInfo(pid, psn))
            Log.d(TAG, "token=$token,pid=$pid, psn=$psn, body=${map.getRequestBody()}")
            getDeviceControlService(context, token).getDeviceState(map.getRequestBody())
        },{
            it?.let {
                result.invoke(it)
            }
            if (it == null) {
                result.invoke(null)
            }
        },{
            Log.e(TAG, "error:${it.errMsg}, ${it.code}")
            result.invoke(null)
        })
    }

    suspend fun getDeviceStateInfo(context: Context, token: String, routerList: List<DeviceRequestInfo>, result:(result: List<StateInfo>?)->Unit) {
        HttpRequest({
            val map = TreeMap<String, Any?>()
            map["list"] = routerList
            getDeviceControlService(context, token).getDeviceState(map.getRequestBody())
        }, {
            it?.let {
                result.invoke(it)
            }
            if (it == null) {
                result.invoke(null)
            }
        }, {
            Log.e(DeviceControlRepository.TAG, "error:${it.errMsg}, ${it.code}")
            result.invoke(null)
        })
    }

    suspend fun getWifiMeshModeSync(context: Context, token: String, pid: String, psn: String): Int? {
        try {
            val map = TreeMap<String, Any?>()
            map["id"] = System.currentTimeMillis().toString()
            map["version"] = "1.0"
            map["params"] = BaseParam()
            map["method"] = "service"
            map["serviceId"] = ServiceIdConstants.GET_WIFI_MESH_MODE
            Log.d(TAG, "token=$token,pid=$pid, psn=$psn, body=${map.getRequestBody()}")
            val response = getDeviceControlService(context, token).getWifiMeshMode(pid, psn, map.getRequestBody())
            return if (response.isSuccess()) {
                if (response.ret != null) {
                    response.ret.mode
                } else if (response.data != null) {
                    response.data.mode
                } else {
                    null
                }
            } else {
                null
            }
        } catch (eThrowable: Throwable) {
            if (eThrowable !is CancellationException) {
                Log.e("HttpRequest", "errorCallBack : ${eThrowable.message}")
            }
            return null
        }
    }

    suspend fun getCurDeviceNetworkSpeedInfo(context: Context, token: String, pid: String, psn: String, result:(result: NetworkSpeedInfo?)->Unit) {
        HttpRequest({
            val map = TreeMap<String, Any?>()
            map["id"] = System.currentTimeMillis().toString()
            map["version"] = "1.0"
            map["params"] = BaseParam()
            map["method"] = "service"
            map["serviceId"] = ServiceIdConstants.GET_NETWORK_SPEED_INFO
            Log.d(TAG, "token=$token,pid=$pid, psn=$psn, body=${map.getRequestBody()}")
            getDeviceControlService(context, token).getCurDeviceNetworkSpeedInfo(pid, psn, map.getRequestBody())
        },{
            result.invoke(it)
        },{
            Log.e(TAG, "error:${it.errMsg}, ${it.code}")
            result.invoke(null)
        })
    }

    suspend fun getOnlineDeviceCount(context: Context, token: String, pid: String, psn: String, result:(result: String?)->Unit) {
        HttpRequest({
            val map = TreeMap<String, Any?>()
            map["id"] = System.currentTimeMillis().toString()
            map["version"] = "1.0"
            map["params"] = BaseParam()
            map["method"] = "service"
            map["serviceId"] = ServiceIdConstants.GET_ONLINE_DEVICE_COUNT
            Log.d(TAG, "token=$token,pid=$pid, psn=$psn, body=${map.getRequestBody()}")
            getDeviceControlService(context, token).getOnlineDeviceCount(pid, psn, map.getRequestBody())
        },{
            if (it == null) {
                result.invoke(null)
            } else {
                result.invoke(it.count)
            }
        },{
            Log.e(TAG, "error:${it.errMsg}, ${it.code}")
            result.invoke(null)
        })
    }

    /**
     * 打印返回消息
     *
     * @param response 返回的对象
     */
    private fun getResponseInfo(responseBody: ResponseBody?): String {
        var str = ""
        val contentLength = responseBody!!.contentLength()
        val source = responseBody.source()
        try {
            source.request(Long.MAX_VALUE) // Buffer the entire body.
        } catch (e: IOException) {
            e.printStackTrace()
        }
        val buffer = source.buffer()
        val charset = Charset.forName("utf-8")
        if (contentLength != 0L) {
            str = buffer.clone().readString(charset)
        }
        return str
    }

    private fun parseMeshInfo(responseInfo: String): List<MeshNode>?{
        if (!TextUtils.isEmpty(responseInfo)) {
            try {
                val responseJson = JSONObject(responseInfo)
                val code = responseJson.optInt("flag")
                if (code < 1) {
                    return null
                }
                val ret = responseJson.optJSONObject("ret") ?: return null
                val meshInf = ret.optJSONObject("meshInf")
                Log.d(TAG, "meshInf:${meshInf}")
                if (meshInf == null) {
                    return null
                }
                val deviceMap: HashMap<String, MeshNode> = HashMap()
                for (key in meshInf.keys()) {
                    val keyArray = key.split("_")
                    if (keyArray.size != 3) {
                        continue
                    }
                    if (!deviceMap.containsKey(keyArray[1])) {
                        deviceMap[keyArray[1]] = MeshNode("", "", 0)
                    }
                    val device: MeshNode = deviceMap[keyArray[1]]!!
                    when(keyArray[2]) {
                        "MAC" -> device.mac = meshInf.optString(key)
                        "SSID" -> device.ssid = meshInf.optString(key)
                        "Role" -> device.role = meshInf.optInt(key)
                    }
                }
                val list = ArrayList<MeshNode>()
                for (value in deviceMap.values) {
                    if (!TextUtils.isEmpty(value.mac) || !TextUtils.isEmpty(value.ssid)) {
                        list.add(value)
                    }
                }
                return list
            } catch (e: JSONException) {
                e.printStackTrace()
                return null
            }
        } else {
            return null
        }
    }

    private fun parseTerminalInfo(responseInfo: String): List<ConnectedDevice>? {
        if (!TextUtils.isEmpty(responseInfo)) {
            try {
                val responseJson = JSONObject(responseInfo)
                val code = responseJson.optInt("flag")
                if (code < 1) {
                    return null
                }
                val ret = responseJson.optJSONObject("ret") ?: return null
                val clientInf = ret.optJSONObject("clientInf")
                Log.d(TAG, "clientInf:${clientInf}")
                if (clientInf == null) {
                    return null
                }
                val deviceMap: HashMap<String, ConnectedDevice> = HashMap()
                for (key in clientInf.keys()) {
                    val keyArray = key.split("_")
                    if (keyArray.size != 3) {
                        continue
                    }
                    if (!deviceMap.containsKey(keyArray[1])) {
                        deviceMap[keyArray[1]] = ConnectedDevice("", "", 0, 0, 0)
                    }
                    val device: ConnectedDevice = deviceMap[keyArray[1]]!!
                    when(keyArray[2]) {
                        "deviceName" -> device.deviceName = clientInf.optString(key)
                        "deviceMac" -> device.deviceMac = clientInf.optString(key)
                        "deviceType" -> device.deviceType = clientInf.optInt(key)
                        "deviceState" -> device.deviceState = clientInf.optInt(key)
                        "clientType" -> device.clientType = clientInf.optInt(key)
                    }
                }
                val list = ArrayList<ConnectedDevice>()
                for (value in deviceMap.values) {
                    if(!TextUtils.isEmpty(value.deviceMac) || !TextUtils.isEmpty(value.deviceName)) {
                        list.add(value)
                    }
                }
                return list
            } catch (e: JSONException) {
                e.printStackTrace()
                return null
            }
        } else {
            return null
        }
    }

}