package com.transsion.baseui.dialog

import android.app.Activity
import android.os.Bundle
import android.text.TextUtils
import android.view.Gravity
import android.view.View
import android.view.WindowManager
import android.widget.CheckBox
import android.widget.CompoundButton
import android.widget.ImageView
import android.widget.TextView
import androidx.annotation.DrawableRes
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.fragment.app.Fragment
import com.blankj.utilcode.util.SizeUtils
import com.transsion.baseui.R

/**
 * @des: 通用dialog
 * @author: huangyeling
 * @date: 2021/9/10 13:45
 * @see {@link }
 */
class TRDialog : TRBaseDialog() {

    companion object {

        const val DIALOG_TITLE = "title"        // 标题
        const val DIALOG_MESSAGE = "message"    // 内容
        const val DIALOG_LEFT_TEXT = "leftBtn"  // 左边按钮文本
        const val DIALOG_RIGHT = "rightBtn"     // 右边按钮文本
        const val DIALOG_CB_TIP = "cb_tip"      // checkBox 提示

        const val CLICK_LISTENER = "click_listener" // left button, right button 点击事件
        const val CLOSE_LISTENER = "close_listener" // close button 点击事件
        const val IS_SHOW_CLOSE = "is_show_close"
    }

    private var titleText: TextView? = null
    private var messageText: TextView? = null
    private var leftBtn: TextView? = null
    private var rightBtn: TextView? = null
    private var ivClose: ImageView? = null
    private var cb: CheckBox? = null

    private var isShowClose = false

    private var listener: TRDialogListener? = null
    private var closeListener: TRDialogCloseListener? = null

    private var onCheckedChangeListener: CompoundButton.OnCheckedChangeListener? = null

    private var leftBtnResId = 0        // 左边按钮背景
    private var rightBtnResId = 0       // 右边按钮背景

    private var leftBtnTextColor = 0        // 左边按钮字体颜色
    private var rightBtnTextColor = 0       // 右边按钮字体颜色

    private var leftBtnWidth = 0        //  特殊情况需要调整按钮宽高 px
    private var leftBtnHeight = 0

    private var rightBtnWidth = 0
    private var rightBtnHeight = 0

    private var rightBtnGravity = Gravity.CENTER
    private var leftBtnGravity = Gravity.CENTER
    private var messageGravity = Gravity.CENTER

    override fun getLayoutId(): Int {
        return R.layout.libui_dialog_confirm_layout
    }

    override fun initView(view: View) {
        titleText = view.findViewById(R.id.tv_title)
        messageText = view.findViewById(R.id.tv_msg)
        leftBtn = view.findViewById(R.id.tv_left)
        rightBtn = view.findViewById(R.id.tv_right)
        ivClose = view.findViewById(R.id.iv_close)
        cb = view.findViewById(R.id.cb)

        setDialog(
            arguments?.getString(DIALOG_TITLE),
            arguments?.getString(DIALOG_MESSAGE),
            arguments?.getString(DIALOG_LEFT_TEXT),
            arguments?.getString(DIALOG_RIGHT),
            arguments?.getString(DIALOG_CB_TIP),
        )

    }

    override fun onResume() {
        // 设置dialog宽度
        context?.let {
            dialog?.window?.setLayout(SizeUtils.dp2px(280f),
                WindowManager.LayoutParams.WRAP_CONTENT)
        }
        super.onResume()
    }


    private fun setDialog(
        title: String?,
        message: String?,
        leftText: String?,
        rightText: String?,
        cbTip: String?,
    ): TRBaseDialog {
        context?.let {
            if (TextUtils.isEmpty(title)) {
                titleText?.visibility = View.GONE
                val lp = messageText?.layoutParams
                if (lp is ConstraintLayout.LayoutParams) {
                    lp.topMargin = SizeUtils.dp2px(28f)
                    messageText?.layoutParams = lp
                }
            } else {
                titleText?.text = title
                titleText?.visibility = View.VISIBLE
            }

            if (TextUtils.isEmpty(leftText)) {
                leftBtn?.visibility = View.GONE

                // 仅有RightButton, 居中显示，并调整下宽度
                val lp = rightBtn?.layoutParams
                if (lp is ConstraintLayout.LayoutParams) {
                    lp.startToStart = ConstraintLayout.LayoutParams.PARENT_ID
                    lp.endToEnd = ConstraintLayout.LayoutParams.PARENT_ID
                    lp.marginEnd = 0
                    lp.width = SizeUtils.dp2px(150f)
                    rightBtn?.layoutParams = lp
                }

            } else {
                leftBtn?.text = leftText
                if (leftBtnResId > 0) {
                    leftBtn?.setBackgroundResource(leftBtnResId)
                }
                if (leftBtnTextColor != 0) {
                    leftBtn?.setTextColor(leftBtnTextColor)
                }
                val lp = leftBtn?.layoutParams as ConstraintLayout.LayoutParams
                if (rightBtnWidth > 0) {
                    lp.width = rightBtnWidth
                }

                if (rightBtnHeight > 0) {
                    lp.height = rightBtnHeight
                }
                leftBtn?.layoutParams = lp
            }

            if (TextUtils.isEmpty(rightText)) {
                rightBtn?.visibility = View.GONE

                // 仅有LeftButton, 居中显示，并调整下宽度
                val lp = leftBtn?.layoutParams
                if (lp is ConstraintLayout.LayoutParams) {
                    lp.startToStart = ConstraintLayout.LayoutParams.PARENT_ID
                    lp.endToEnd = ConstraintLayout.LayoutParams.PARENT_ID
                    lp.marginStart = 0
                    lp.width = SizeUtils.dp2px(150f)
                    leftBtn?.layoutParams = lp
                }
            } else {
                rightBtn?.text = rightText
                if (rightBtnResId > 0) {
                    rightBtn?.setBackgroundResource(rightBtnResId)
                }
                if (rightBtnTextColor != 0) {
                    rightBtn?.setTextColor(rightBtnTextColor)
                }

                val lp = rightBtn?.layoutParams as ConstraintLayout.LayoutParams

                if (rightBtnWidth > 0) {
                    lp.width = rightBtnWidth
                }

                if (rightBtnHeight > 0) {
                    lp.height = rightBtnHeight
                }

                rightBtn?.layoutParams = lp
            }

            if (TextUtils.isEmpty(title)) {
                messageText?.textSize = 16f
            } else {
                messageText?.textSize = 14f
            }

            messageText?.text = message

            if (TextUtils.isEmpty(cbTip)) {
                cb?.visibility = View.GONE
            } else {
                cb?.visibility = View.VISIBLE
                cb?.text = cbTip

                cb?.setOnCheckedChangeListener { buttonView, isChecked ->
                    onCheckedChangeListener?.onCheckedChanged(buttonView, isChecked)
                }
            }

            leftBtn?.gravity = leftBtnGravity
            rightBtn?.gravity = rightBtnGravity
            messageText?.gravity = messageGravity

            setClickListener()
            setCloseListener()
        }
        return this
    }

    private fun setClickListener() {
        if (listener == null) return
        when (listener) {
            is Activity -> {
                leftBtn?.setOnClickListener {
                    (activity as TRDialogListener).onLeftButtonClick(this)
                    dismissDialog()
                }
                rightBtn?.setOnClickListener {
                    (activity as TRDialogListener).onRightButtonClick(this)
                    dismissDialog()
                }
            }
            is Fragment -> {
                leftBtn?.setOnClickListener {
                    (parentFragment as TRDialogListener).onLeftButtonClick(this)
                    dismissDialog()
                }
                rightBtn?.setOnClickListener {
                    (parentFragment as TRDialogListener).onRightButtonClick(this)
                    dismissDialog()
                }
            }
            else -> {
//                throw Exception("TNDialogListener must be implement by Activity or Fragment !!!")
                leftBtn?.setOnClickListener {
                    listener?.onLeftButtonClick(this)
                }
                rightBtn?.setOnClickListener {
                    listener?.onRightButtonClick(this)
                }
            }
        }
    }

    private fun setCloseListener() {
        isShowClose = arguments?.getBoolean(IS_SHOW_CLOSE) ?: false
        if (isShowClose) {
            ivClose?.visibility = View.VISIBLE
            ivClose?.setOnClickListener {
                if (closeListener != null) {
                    when (closeListener) {
                        is Activity -> {
                            (activity as TRDialogCloseListener).onCloseClick(this)
                        }
                        is Fragment -> {
                            (parentFragment as TRDialogCloseListener).onCloseClick(this)
                        }
                        else -> {
//                            throw Exception("TNDialogCloseListener must be implement by Activity or Fragment !!!")
                            closeListener?.onCloseClick(this)
                        }
                    }
                }
                dismissDialog()
            }
        } else {
            ivClose?.visibility = View.GONE
        }
    }

    /**
     * 内部配置类
     */
    class Builder constructor() {
        private var titleText: String? = null
        private var messageText: String? = null
        private var leftBtn: String? = null
        private var rightBtn: String? = null
        private var listener: TRDialogListener? = null
        private var closeListener: TRDialogCloseListener? = null
        private var tag: String? = null
        private var isAnim: Boolean = false
        private var dimAmount: Float = 0.5f
        private var cbTip: String? = null
        private var onCheckedChangeListener: CompoundButton.OnCheckedChangeListener? = null
        private var cbVisibility: Int = View.GONE
        private var isShowClose = false
        private var leftBtnResId = 0        // 左边按钮背景
        private var rightBtnResId = 0       // 右边按钮背景
        private var leftBtnTextColor = 0        // 左边按钮字体颜色
        private var rightBtnTextColor = 0       // 右边按钮字体颜色

        private var leftBtnWidth = 0        //  特殊情况需要调整按钮宽高 px
        private var leftBtnHeight = 0

        private var rightBtnWidth = 0
        private var rightBtnHeight = 0
        private var rightBtnGravity = Gravity.CENTER
        private var leftBtnGravity = Gravity.CENTER
        private var messageGravity = Gravity.CENTER

        fun setTitle(title: String): Builder {
            this.titleText = title
            return this
        }

        fun setMessage(message: String): Builder {
            this.messageText = message
            return this
        }

        fun setLeftText(leftText: String): Builder {
            this.leftBtn = leftText
            return this
        }

        fun setRightText(rightText: String): Builder {
            this.rightBtn = rightText
            return this
        }

        fun setCbTip(cbTip: String): Builder {
            this.cbTip = cbTip
            return this
        }

        fun setDimAmount(dimAmount: Float): Builder {
            this.dimAmount = dimAmount
            return this
        }

        /**
         * 设置回调
         */
        fun setListener(listener: TRDialogListener): Builder {
            this.listener = listener
            return this
        }

        fun setCloseListener(listener: TRDialogCloseListener): Builder {
            closeListener = listener
            return this
        }

        fun setOnCheckedChangeListener(onCheckedChangeListener: CompoundButton.OnCheckedChangeListener): Builder {
            this.onCheckedChangeListener = onCheckedChangeListener
            return this
        }

        /**
         * 是否显示动画
         */
        fun setShowAnimation(isAnim: Boolean): Builder {
            this.isAnim = isAnim
            return this
        }

        fun isShowAnimation(): Boolean {
            return this.isAnim
        }

        fun setTag(tag: String): Builder {
            this.tag = tag
            return this
        }

        fun setCbVisibility(visibility: Int): Builder {
            this.cbVisibility = visibility
            return this
        }

        fun setShowClose(isShowClose: Boolean): Builder {
            this.isShowClose = isShowClose
            return this
        }

        fun setLeftBtnResId(@DrawableRes leftBtnResId: Int): Builder {
            this.leftBtnResId = leftBtnResId
            return this
        }

        fun setRightBtnResId(@DrawableRes rightBtnResId: Int): Builder {
            this.rightBtnResId = rightBtnResId
            return this
        }

        fun setLeftBtnTextColor(color: Int): Builder {
            this.leftBtnTextColor = color
            return this
        }

        fun setRightBtnTextColor(color: Int): Builder {
            this.rightBtnTextColor = color
            return this
        }

        fun setLeftBtnWH(width: Int, height: Int): Builder {
            this.leftBtnWidth = width
            this.leftBtnHeight = height
            return this
        }

        fun setRightBtnWH(width: Int, height: Int): Builder {
            this.rightBtnWidth = width
            this.rightBtnHeight = height
            return this
        }

        fun setRightBtnGravity(gravity: Int): Builder = apply {
            this.rightBtnGravity = gravity
        }

        fun setLeftBtnGravity(gravity: Int): Builder = apply {
            this.leftBtnGravity = gravity
        }

        fun setMessageGravity(gravity: Int): Builder = apply {
            this.messageGravity = gravity
        }

        fun build(): TRBaseDialog {
            val dialog = TRDialog()

            val bundle = Bundle()
            bundle.putString(DIALOG_TITLE, titleText)
            bundle.putString(DIALOG_MESSAGE, messageText)
            bundle.putString(DIALOG_LEFT_TEXT, leftBtn)
            bundle.putString(DIALOG_RIGHT, rightBtn)
            bundle.putString(DIALOG_CB_TIP, cbTip)
            bundle.putBoolean(IS_SHOW_CLOSE, isShowClose)
            dialog.arguments = bundle

            dialog.setDimAmount(dimAmount)

            isShowAnimation().let {
                dialog.setShowAnimation(it)
            }

            dialog.listener = listener
            dialog.closeListener = closeListener
            dialog.onCheckedChangeListener = onCheckedChangeListener
            dialog.leftBtnResId = leftBtnResId
            dialog.rightBtnResId = rightBtnResId
            dialog.leftBtnTextColor = leftBtnTextColor
            dialog.rightBtnTextColor = rightBtnTextColor
            dialog.leftBtnWidth = leftBtnWidth
            dialog.leftBtnHeight = leftBtnHeight
            dialog.rightBtnWidth = rightBtnWidth
            dialog.rightBtnHeight = rightBtnHeight
            dialog.rightBtnGravity = rightBtnGravity
            dialog.leftBtnGravity = leftBtnGravity
            dialog.messageGravity = messageGravity

            return dialog
        }
    }
}

fun TRDialog.Builder.doOnClick(leftClickListener: () -> Unit = {},
                               rightClickListener: () -> Unit = {}) = apply {
    setListener(object : TRDialogListener {
        override fun onLeftButtonClick(dialog: TRDialog) {
            leftClickListener()
        }

        override fun onRightButtonClick(dialog: TRDialog) {
            rightClickListener()
        }
    })
}

