package com.transsion.mb.config.download

import android.net.Network
import android.net.NetworkCapabilities
import android.os.Build
import com.tn.lib.logger.MBLogger
import com.tn.lib.net.base.BaseObserver
import com.tn.lib.net.base.BaseScheduler
import com.tn.lib.net.env.Host
import com.tn.lib.net.manager.NetServiceGenerator
import com.tn.lib.util.networkinfo.NetworkUtil
import com.transsion.mb.config.manager.ConfigMMKV
import com.transsion.mb.config.manager.ConfigManager
import com.transsion.mb.config.manager.ConfigParamsProvider
import java.util.ServiceLoader

/**
 *   Transsion MI
 *   Created By Liupeng
 *   On 2023/9/18 15:25
 *
 *   Desc: 请求配置
 */
object RequestConfig {
    private const val TAG = "RequestConfig"
    /**
     *     是否请求成功
     */
    private var isRequestConfigSuccess = false
    private var configLoadListener: IConfigLoadListener? = null

    private var requestConfigDisposable: Boolean = false
    private val configApi by lazy { NetServiceGenerator.instance.getService(ConfigApi::class.java) }

    private var requestKeys = ""

    fun setIConfigLoadListener(configLoadListener: IConfigLoadListener) {
        this.configLoadListener = configLoadListener
    }

    fun requestConfig(host: String = Host.getHost(), path: String = "wefeed-mobile-bff") {
        val disposable = requestConfigDisposable
        if (disposable) {
            MBLogger.d(TAG, "requestConfig ing...")
            return
        }
        requestConfigDisposable  = true
        isRequestConfigSuccess = false

        val configVersion = ConfigMMKV.mmkv.getString(ConfigMMKV.CONFIG_VERSION, "") ?: ""
        configApi.appConfig(path, host, getCombinedRequestParam(), configVersion)
            .compose(BaseScheduler.composeIO())
            .retry { t1, _ ->
                MBLogger.d(TAG, "retry ing.. t1:$t1")
                return@retry isNetConnect == true && t1 <= 1
            }
            .subscribe(object : BaseObserver<AppStartConfig>() {
                override fun onFailure(code: String?, message: String?) {
                    MBLogger.d(TAG, "onFailure code: $code  message: $message")
                    requestConfigDisposable = false
                    configLoadListener?.onConfigLoadFailed()
                }

                override fun onSuccess(data: AppStartConfig?) {
                    super.onSuccess(data)
                    isRequestConfigSuccess = true
                    requestConfigDisposable = false
                    if (data == null) {
                        return
                    }
                    MBLogger.d(TAG, "onSuccess:$data")
                    // 广告配置都在这里进行处理
                    kotlin.runCatching {
                        // 保存服务端下发的广告位置  json类型数据取值
                        data.items?.let {
                            ConfigManager.instance.updateConfig(it)
                        }

                        //保存版本
                        data.version?.let {
                            ConfigMMKV.mmkv.putString(ConfigMMKV.CONFIG_VERSION, it)
                        }

                        configLoadListener?.onConfigLoadSuccess()
                    }
                }
            })
    }

    private fun getCombinedRequestParam(): String {
        if (requestKeys.isNotEmpty()) {
            return requestKeys
        }

        //低版本有概率报service错误，改为全量请求
        if (Build.VERSION.SDK_INT <= Build.VERSION_CODES.M) {
            requestKeys = "all"
            return requestKeys
        }

        try {
            val services = ServiceLoader.load(ConfigParamsProvider::class.java)
            val finalSet = mutableSetOf<String>()

            for (service in services) {
                finalSet.addAll(service.providerParams())
            }

            requestKeys = finalSet.joinToString(",")

        } catch (e: Exception) {
            MBLogger.e("CombinedRequestParam", e.message ?: "")
            e.printStackTrace()
            requestKeys = "all"
        }

        return requestKeys
    }

    fun checkConfigRequest() {
        if (!isRequestConfigSuccess) {
            requestConfig()
        }
    }

    private var isNetConnect: Boolean? = null
        get() {
            if (field == null)
                field = NetworkUtil.hasCapabilityAsync()
            return field
        }

    fun onDisconnected() {
        isNetConnect = false
    }

    fun onConnected(network: Network, networkCapabilities: NetworkCapabilities) {
        isNetConnect = true
        checkConfigRequest()
    }
}