package com.transsion.mb.config.manager

import android.text.TextUtils
import android.util.Log
import com.blankj.utilcode.util.GsonUtils
import com.google.firebase.remoteconfig.FirebaseRemoteConfig
import com.google.gson.JsonArray
import com.google.gson.JsonObject
import com.transsion.base.report.athena.Report
import java.text.SimpleDateFormat
import java.util.Calendar
import java.util.Locale

/**
 *   Transsion MI
 *   Created By Liupeng
 *   On 2023/8/14 18:01
 *
 *   Desc:远程配置类
 *   方案细节参考
 *   https://transsioner.feishu.cn/wiki/TidrwqogsiZs8aknOPHclcOunsc
 */
class ConfigManager {
    companion object {
        val instance by lazy { ConfigManager() }
        private const val CONFIG_AB_RESULT = "config_ab_result"
    }

    private val mCacheMap = hashMapOf<String, ConfigBean?>()
    private val abResultSet = hashSetOf<String>()

    /**
     * 更新simba配置
     * @param items JsonArray
     */
    fun updateConfig(items: JsonArray) {
        items.forEach {
            val jsonObject: JsonObject = it.getAsJsonObject()
            val key = jsonObject.get("key").asString

            //simba和firebase key相同时，优先使用firebase，不覆写
            val beforeValue = ConfigMMKV.mmkv.getString(key, null)
            if (beforeValue.isNullOrEmpty() || !beforeValue.contains("\"isFirebase\":true")) {
                ConfigMMKV.mmkv.putString(key, jsonObject.toString())
            }

            // 在这里输出一下 非标广告的配置
            if (key == "ad_scene_config") {
                Log.w("co_non", "ConfigManager --> updateConfig --> ad_scene_config = $jsonObject")
            }
            // 商业化下载拦截配置
            if (key == "business_guide_config") {
                Log.w("co_non", "ConfigManager --> updateConfig --> business_guide_config = $jsonObject")
            }
        }

        mCacheMap.forEach {
            it.value?.hasUpdate = true
        }
    }

    /**
     * firebase config保存，和simba共用一套
     */
    fun saveFirebaseConfig(firebaseRemoteConfig: FirebaseRemoteConfig) {
        val keys = firebaseRemoteConfig.getKeysByPrefix("")
        for (key in keys) {
            val bean = ConfigBean()
            bean.key = key
            bean.value = firebaseRemoteConfig.getString(key)
            bean.isFirebase = true
            ConfigMMKV.mmkv.putString(key, GsonUtils.toJson(bean))
        }
    }

    /**
     * 获取配置对象
     * @param key String
     * @return ConfigBean?
     */
    fun getConfig(key: String, needNewest: Boolean = true): ConfigBean? {
        var configBean: ConfigBean? = null
        if (mCacheMap.contains(key)) {
            configBean = mCacheMap[key]

            //如果需要最新结果，并且bean已经被更新过，则置为空从mmkv中重新获取
            if (needNewest && configBean?.hasUpdate == true) {
                configBean = null
            }
        }

        if (configBean == null) {
            val json = ConfigMMKV.mmkv.getString(key, null)
            if (!TextUtils.isEmpty(json)) {
                try {
                    configBean = GsonUtils.fromJson(json, ConfigBean::class.java)
                    mCacheMap[key] = configBean
                } catch (e: Exception) {
                    e.printStackTrace()
                }
            }
        }

        val reportKey = key + getCurrentDate()
        if (configBean != null && !abResultSet.contains(reportKey)) {
            reportAbResult(configBean)
            abResultSet.add(reportKey)
        }
        return configBean
    }

    private fun getCurrentDate(): String {
        val currentDate = Calendar.getInstance().time
        val formatter = SimpleDateFormat("yyyy-MM-dd", Locale.getDefault())
        return formatter.format(currentDate)
    }

    /**
     * 上报ab结果 隔天获取要上报
     * @param configBean ConfigBean
     */
    private fun reportAbResult(configBean: ConfigBean) {
        if (!TextUtils.isEmpty(configBean.ops)) {
            val map = hashMapOf<String, String?>()
            map["key"] = configBean.key
            map["ops"] = configBean.ops
            map["isFirebase"] = configBean.isFirebase.toString()
            Report.reportEvent(CONFIG_AB_RESULT, map)
        }
    }
}