package com.transsion.base.image

import android.app.ActivityManager
import android.content.Context
import com.blankj.utilcode.util.Utils
import com.bumptech.glide.Glide
import com.bumptech.glide.GlideBuilder
import com.bumptech.glide.Registry
import com.bumptech.glide.annotation.GlideModule
import com.bumptech.glide.integration.okhttp3.OkHttpUrlLoader
import com.bumptech.glide.load.DecodeFormat
import com.bumptech.glide.load.engine.cache.MemorySizeCalculator
import com.bumptech.glide.load.model.GlideUrl
import com.bumptech.glide.module.AppGlideModule
import com.bumptech.glide.request.RequestOptions
import com.google.net.cronet.okhttptransport.CronetInterceptor
import com.tn.lib.logger.MBLogger
import com.transsion.base.image.net.HttpsTrustManager
import com.transsion.base.image.net.TrustAllHostnameVerifier
import com.transsion.baseui.image.glide.GlideSafeBitmapPool
import okhttp3.Interceptor
import okhttp3.OkHttpClient
import org.chromium.net.CronetEngine
import java.io.InputStream
import java.util.concurrent.TimeUnit


/**
 *   Transsion MI
 *   Created By Liupeng
 *   On 2023/3/27 15:16
 *
 *   Desc:
 */
@GlideModule
class CustomGlideModule : AppGlideModule() {
    private val requestOptions = RequestOptions()


    override fun registerComponents(context: Context, glide: Glide, registry: Registry) {
        //如果检测当前没有cronet支持 图片加载交给okhttp
        if (!isCronetSupported()) {
            MBLogger.d("CustomGlideModule", "glide cronet disabled,use okhttp")
            newGlideOkhttpClient()?.let {
                registry.replace(
                    GlideUrl::class.java,
                    InputStream::class.java,
                    OkHttpUrlLoader.Factory(it)
                )
            }
        }
    }

    private fun isCronetSupported(): Boolean {
        return try {
            val builder = CronetEngine.Builder(Utils.getApp())
            val cronetEngine = builder.build()
            cronetEngine.shutdown()
            true
        } catch (e: Throwable) {
            false
        }
    }

    override fun applyOptions(context: Context, builder: GlideBuilder) {
        val activityManager = context.getSystemService(Context.ACTIVITY_SERVICE) as ActivityManager
        val memoryInfo = ActivityManager.MemoryInfo()
        activityManager.getMemoryInfo(memoryInfo)
        val totalMemory = memoryInfo.totalMem
        val totalMemoryGB = totalMemory / (1024f * 1024 * 1024)

        if (totalMemoryGB < 4) {
            builder.setDefaultRequestOptions(this.requestOptions.format(DecodeFormat.PREFER_RGB_565))
        }

        // 用 Glide 的 MemorySizeCalculator 获取默认大小
        val calculator = MemorySizeCalculator.Builder(context).build()
        val defaultBitmapPoolSize = calculator.bitmapPoolSize.toLong()
        // 使用安全的 Pool 替代默认实现
        builder.setBitmapPool(GlideSafeBitmapPool(defaultBitmapPoolSize))
    }

    /**
     * okHttpclide for glide
     * @return OkHttpClient
     */
    private var okHttpClientForGlide: OkHttpClient? = null

    private fun newGlideOkhttpClient(): OkHttpClient? {
        if (okHttpClientForGlide != null) {
            return okHttpClientForGlide
        }
        val builder = OkHttpClient.Builder()
        //避免时间错误带来的ssl异常
        builder.sslSocketFactory(
            HttpsTrustManager.createSSLSocketFactory(),
            HttpsTrustManager()
        ).hostnameVerifier(TrustAllHostnameVerifier())//配置ssl以支持抓包
        val instance =
            builder.connectTimeout(10 * 4L, TimeUnit.SECONDS)
                .writeTimeout(10 * 2L, TimeUnit.SECONDS)
                .readTimeout(10 * 2L, TimeUnit.SECONDS)
        val cronetInterceptor = getCronetInterceptorForGlide()
        if (cronetInterceptor != null) {
            instance.addInterceptor(cronetInterceptor)//cronet拦截器 必须为最后一个
        }
        okHttpClientForGlide = instance.build()
        return okHttpClientForGlide
    }

    private fun getCronetInterceptorForGlide(): Interceptor? {
        try {
            val builder = CronetEngine.Builder(Utils.getApp())
            builder.enableQuic(true)
                .enableBrotli(true)
                .addQuicHint("pacdn.aoneroom.com", 443, 443)
            val cronetEngine: CronetEngine = builder.build()
            return CronetInterceptor.newBuilder(cronetEngine).build()
        } catch (e: Throwable) {
            return null
        }
    }
}