package com.transsion.base.image

import android.annotation.SuppressLint
import android.content.Context
import android.graphics.drawable.Drawable
import android.text.TextUtils
import android.widget.ImageView
import androidx.annotation.DrawableRes
import androidx.fragment.app.FragmentActivity
import com.blankj.utilcode.util.ScreenUtils
import com.blankj.utilcode.util.Utils
import com.bumptech.glide.Glide
import com.bumptech.glide.Priority
import com.bumptech.glide.integration.cronet.CronetEngineSingleton
import com.bumptech.glide.load.DataSource
import com.bumptech.glide.load.MultiTransformation
import com.bumptech.glide.load.engine.DiskCacheStrategy
import com.bumptech.glide.load.engine.GlideException
import com.bumptech.glide.load.resource.bitmap.CenterCrop
import com.bumptech.glide.load.resource.bitmap.RoundedCorners
import com.bumptech.glide.load.resource.drawable.DrawableTransitionOptions
import com.bumptech.glide.request.RequestListener
import com.bumptech.glide.request.RequestOptions
import com.bumptech.glide.request.target.Target
import com.bumptech.glide.request.transition.DrawableCrossFadeFactory
import com.tn.lib.logger.MBLogger
import com.transsion.base.image.blurhash.BlurHash
import com.transsion.base.image.blurhash.blurPlaceHolder
import org.chromium.net.CronetEngine

class ImageHelper {
    companion object {
        const val TAG = "ImageHelper"
        var defaultGridWith = ScreenUtils.getScreenWidth() / 3
        private var defaultWidth = ScreenUtils.getAppScreenWidth() / 2
        private var defaultHeight = ScreenUtils.getScreenHeight()

        // CustomGlideModule的OkHttpClient使用了Cronet
//        init {
//            // glide 集成库自带的CronetEngine关闭了对quic的支持，利用反射重新赋值
//            try {
//                val engine = CronetEngine.Builder(Utils.getApp())
//                    .enableHttpCache(CronetEngine.Builder.HTTP_CACHE_DISABLED, 0)
//                    .enableHttp2(true)
//                    .enableQuic(true)
//                    .addQuicHint("test-acdn.aoneroom.com", 443, 443)
//                    .addQuicHint("acdn.aoneroom.com", 443, 443)
//                    .build()
//
//                for (f in CronetEngineSingleton::class.java.declaredFields) {
//                    if (f.type == CronetEngine::class.java) {
//                        f.isAccessible = true
//                        f.set(null, engine)
//                        MBLogger.d(TAG, "init CronetEngine")
//                        break
//                    }
//                }
//            } catch (ignore: Throwable) {
//                MBLogger.d(TAG, "init CronetEngine failed", ignore)
//            }
//        }

        fun with(context: Context): ImageBuilder {
            return ImageBuilder(context)
        }

        /**
         * https://help.aliyun.com/document_detail/144582.html
         * 利用阿里云服务 支持图片裁剪和转码 避免下载原图
         *
         * 加载阿里云网络图片必用方法
         * 不超过 屏幕(宽|高)/2 的 按宽适配拉去一份 屏宽1/2的图，超过的需传入要加载的尺寸
         * 超过的单独拉取一份
         * @param imageView View
         * @param url String
         * @param isWebP Boolean
         * @param width Int  不超过 屏幕(宽|高)/2 的 按宽适配拉去一份 屏宽1/2的图，超过的需传入要加载的尺寸
         * @param height Int
         * @param overrideWH glide使用传入的宽高(可针对缩放场景，小imageview加载再放大会模糊，glide内部使用的小imageview尺寸做了处理)
         */
        private fun loadNet(
            context: Context,
            imageView: ImageView,
            url: String,
            @DrawableRes placeHolder: Int = R.color.image_default,
            width: Int = defaultWidth,
            height: Int = defaultHeight,
            round: Int = 0,
            isWebP: Boolean = true,
            thumbnail: String = "",
            isQuality:Boolean = true,
            thumbScaleType:Boolean = true,
            overrideWH: Boolean = false,
            supportH: Boolean = false,
            priority: Priority = Priority.NORMAL,
            blurValue: Int,
            downloadCallback: ((loadCoverSuccess: Boolean, cache: Boolean, duration: Long) -> Unit)? = null
        ) {
            if (context is FragmentActivity && context.isDestroyed) {
                return
            }

            if (TextUtils.isEmpty(url)) {
                //兼容url为null 使用占位图展示
                Glide.with(context)
                    .load("")
                    .transition(fadeTransi)
                    .apply(requestOptions(placeHolder, round, if (overrideWH) height else -1, if (overrideWH) width else -1))
                    .into(imageView)
                return
            }

            var needW = width
            // 没传则用真实的
            if (width == defaultWidth && imageView.layoutParams.width > 0) {
                needW = imageView.layoutParams.width
            }

            //Fatal Exception: com.bumptech.glide.load.engine.CallbackException
            //Unexpected exception thrown by non-Glide code
            try {
                intoWithThumbnail(
                    context,
                    imageView,
                    url,
                    placeHolder,
                    round,
                    needW,
                    height,
                    isWebP,
                    thumbnail,
                    isQuality,
                    thumbScaleType,
                    overrideWH,
                    supportH,
                    priority,
                    blurValue,
                    downloadCallback
                )
            } catch (e: Exception) {
                e.printStackTrace()
            }
        }

        @SuppressLint("StaticFieldLeak")
        private val blurHash = BlurHash(Utils.getApp(), lruSize = 5, punch = .6F)
        private val fadeTransi = DrawableTransitionOptions.with(DrawableCrossFadeFactory.Builder(100).setCrossFadeEnabled(true).build())
        private fun intoWithThumbnail(
            context: Context, imageView: ImageView, url: String,
            @DrawableRes placeHolder: Int = R.color.image_default,
            round: Int = 0,
            width: Int = -1, height: Int = -1,
            isWebP: Boolean = true,
            thumbnail: String = "",
            isQuality: Boolean = true,
            thumbScaleType: Boolean = true,
            overrideWH: Boolean = false,
            supportH: Boolean = false,
            priority: Priority = Priority.NORMAL,
            blurValue: Int = 0,
            downloadCallback: ((loadCoverSuccess: Boolean, cache: Boolean, duration: Long) -> Unit?)? = null
        ) {
            if (context is FragmentActivity && context.isDestroyed) {
                return
            }

            val cachedSize = Resizer.getCachedSize(url)
            val smallUrl = if (cachedSize in 1 until width+1) {
                // thumbnail: 复用已缓存的尺寸
                Resizer.buildResizeUrl(url, cachedSize, isWebP)
            } else if (thumbnail.isEmpty() && width > Resizer.NeedThumbnailMinSize && cachedSize <= width) {
                // thumbnail: 固定尺寸网络小图 如果没有则请求一张小图
                Resizer.buildResizeUrl(url, Resizer.ThumbnailSize, isWebP)
            } else {
                // thumbnail: blur hash or base64
                thumbnail
            }

            val bigUrl = Resizer.getReSizeUrl(url, width, isWebP, isQuality, height, supportH, false, blurValue)
            val reqStartTime = System.currentTimeMillis()

            if (smallUrl.startsWith("http") || smallUrl.startsWith("data:image")) {
                // base64 or 网络小图
                // thumbnail CenterCrop会应用到大图上，glide默认仅支持单个scaleType
                val thumbnailRequest = if (thumbScaleType) {
                    Glide.with(context).load(smallUrl).optionalCenterCrop()
                } else {
                    Glide.with(context).load(smallUrl)
                }
                Glide.with(context)
                    .load(bigUrl)
                    .thumbnail(thumbnailRequest)
                    .listener(object : RequestListener<Drawable>{
                        override fun onLoadFailed(
                            e: GlideException?,
                            model: Any?,
                            target: Target<Drawable>,
                            isFirstResource: Boolean
                        ): Boolean {
                            val duration = System.currentTimeMillis() - reqStartTime
                            downloadCallback?.invoke(false, false, duration)
                            return false
                        }

                        override fun onResourceReady(
                            resource: Drawable,
                            model: Any,
                            target: Target<Drawable>?,
                            dataSource: DataSource,
                            isFirstResource: Boolean
                        ): Boolean {
                            val duration = System.currentTimeMillis() - reqStartTime
                            downloadCallback?.invoke(true, dataSource != DataSource.REMOTE, duration)
                            return false
                        }
                    })
                    .transition(fadeTransi)
                    .apply(requestOptions(placeHolder, round, if (overrideWH) height else -1, if (overrideWH) width else -1, priority))
                    .into(imageView)

            } else if (smallUrl.length >= 6) {
                // blur hash 加载方式
                val loader = Glide.with(context).load(bigUrl)
                loader.blurPlaceHolder(smallUrl, 20, 20, blurHash) {
                    it.into(imageView)
                }
                loader.transition(fadeTransi)
                    .listener(object: RequestListener<Drawable>{
                        override fun onLoadFailed(
                            e: GlideException?,
                            model: Any?,
                            target: Target<Drawable>,
                            isFirstResource: Boolean
                        ): Boolean {
                            val duration = System.currentTimeMillis() - reqStartTime
                            downloadCallback?.invoke(false, false, duration)
                            return false
                        }

                        override fun onResourceReady(
                            resource: Drawable,
                            model: Any,
                            target: Target<Drawable>?,
                            dataSource: DataSource,
                            isFirstResource: Boolean
                        ): Boolean {
                            val duration = System.currentTimeMillis() - reqStartTime
                            downloadCallback?.invoke(true, dataSource != DataSource.REMOTE, duration)
                            return false
                        }
                    })
                    .apply(requestOptions(0, round, if (overrideWH) height else -1, if (overrideWH) width else -1, priority))
                    .into(imageView)
            } else {
                // 不使用缩略图
                Glide.with(context)
                    .load(bigUrl)
                    .transition(fadeTransi)
                    .listener(object : RequestListener<Drawable>{
                        override fun onLoadFailed(
                            e: GlideException?,
                            model: Any?,
                            target: Target<Drawable>,
                            isFirstResource: Boolean
                        ): Boolean {
                            val duration = System.currentTimeMillis() - reqStartTime
                            downloadCallback?.invoke(false, false, duration)
                            return false
                        }

                        override fun onResourceReady(
                            resource: Drawable,
                            model: Any,
                            target: Target<Drawable>?,
                            dataSource: DataSource,
                            isFirstResource: Boolean
                        ): Boolean {
                            val duration = System.currentTimeMillis() - reqStartTime
                            downloadCallback?.invoke(true, dataSource != DataSource.REMOTE, duration)
                            return false
                        }
                    })
                    .apply(requestOptions(placeHolder, round, if (overrideWH) height else -1, if (overrideWH) width else -1, priority))
                    .into(imageView)
            }

            MBLogger.d(TAG, "load $width use $bigUrl with small $smallUrl")
        }

        private fun requestOptions(
            placeHolder: Int,
            round: Int,
            height: Int,
            width: Int,
            priority: Priority = Priority.NORMAL
        ): RequestOptions {
            var options = RequestOptions()
                .priority(priority)
                .diskCacheStrategy(DiskCacheStrategy.DATA)
            if (placeHolder != 0) {
                options = options.placeholder(placeHolder)
            }

            if (round > 0) {
                options = options.transform(MultiTransformation(CenterCrop(), RoundedCorners(round)))
            } else if (round < 0) {
                options = RequestOptions.circleCropTransform()
            }

            if (height != -1 || width != -1) {
                options = options.override(width, height)
            }
            return options
        }

        /**
         * 预加载图片
         * @param url 图片url
         * @param originUrl 原始url
         * @param width 图片宽度
         * @param callback 预加载回调
         */
        fun preload(url: String, originUrl: String, width: Int, callback: ((result: Boolean) -> Unit)? = null) {
            Glide.with(Utils.getApp()).load(url).diskCacheStrategy(DiskCacheStrategy.DATA).addListener(object :
                RequestListener<Drawable> {
                override fun onLoadFailed(
                    e: GlideException?,
                    model: Any?,
                    target: Target<Drawable>,
                    isFirstResource: Boolean
                ): Boolean {
                    //预加载失败
                    callback?.invoke(false)
                    return true
                }

                override fun onResourceReady(
                    resource: Drawable,
                    model: Any,
                    target: Target<Drawable>?,
                    dataSource: DataSource,
                    isFirstResource: Boolean
                ): Boolean {
                    Resizer.putCachedSize(originUrl, width = width)
                    // 预加载成功
                    callback?.invoke(true)
                    return true
                }
            }).preload()
        }

        fun getReSizeUrl(url: String, width: Int = 0, isWebP: Boolean = true, preloadImg:Boolean = false): String {
            return Resizer.getReSizeUrl(url, width, isWebP, preloadImg = preloadImg)
        }
    }


    class ImageBuilder internal constructor(private val context: Context) {
        private val defaultWidth = ScreenUtils.getAppScreenWidth() / 2
        private val defaultHeight = ScreenUtils.getScreenHeight()

        private var width = defaultWidth
        private var height = defaultHeight
        private var url: String = ""
        private var thumbnail: String = ""
        private var placeholder: Int = R.color.image_default
        private var round = 0
        private var isWebP = true
        private var isQuality = true
        private var thumbScaleType = true
        private var overrideWH = false
        private var supportH = false
        private var priority = Priority.NORMAL
        private var downloadCallback: ((loadCoverSuccess: Boolean, cache: Boolean, duration: Long) -> Unit)? = null
        private var blurValue: Int = 0

        fun width(width: Int): ImageBuilder {
            this.width = width
            return this
        }

        fun height(height: Int): ImageBuilder {
            this.height = height
            supportH = true
            return this
        }

        fun load(url: String?): ImageBuilder {
            this.url = (url ?: "")
            return this
        }

        fun placeholder(@DrawableRes res: Int): ImageBuilder {
            this.placeholder = res
            return this
        }

        fun round(round: Int): ImageBuilder {
            this.round = round
            return this
        }

        fun isWebP(isWebP: Boolean): ImageBuilder {
            this.isWebP = isWebP
            return this
        }

        fun thumbnail(thumbnail: String): ImageBuilder {
            this.thumbnail = thumbnail
            return this
        }

        fun isQuality(isQuality: Boolean): ImageBuilder {
            this.isQuality = isQuality
            return this
        }

        fun thumbScaleType(thumbScaleType: Boolean): ImageBuilder {
            this.thumbScaleType = thumbScaleType
            return this
        }

        fun overrideWH(overrideWH: Boolean): ImageBuilder {
            this.overrideWH = overrideWH
            return this
        }

        fun priority(priority: Priority): ImageBuilder {
            this.priority = priority
            return this
        }

        fun downloadCallback(downloadCallback: ((loadCoverSuccess: Boolean, cache: Boolean, duration: Long) -> Unit)?): ImageBuilder {
            this.downloadCallback = downloadCallback
            return this
        }

        //1-50 值越大越模糊
        fun blurValue(blurValue: Int): ImageBuilder {
            this.blurValue = blurValue
            return this
        }

        fun into(imageView: ImageView) {
            loadNet(
                context = context,
                imageView = imageView,
                url = url,
                placeHolder = placeholder,
                width = width,
                height = height,
                round = round,
                isWebP = isWebP,
                thumbnail = thumbnail,
                isQuality = isQuality,
                thumbScaleType = thumbScaleType,
                overrideWH = overrideWH,
                supportH = supportH,
                priority = priority,
                blurValue = blurValue,
                downloadCallback = downloadCallback)
        }
    }
}
