package com.tn.lib.thread

import com.tn.lib.thread.callback.AsyncCallback
import com.tn.lib.thread.callback.RunnableCallback
import com.tn.lib.thread.config.Builder
import com.tn.lib.thread.config.LocalConfigs
import com.tn.lib.thread.dispatcher.DelayDispatcher
import com.tn.lib.thread.wrapper.CallableWrapper
import com.tn.lib.thread.wrapper.RunnableWrapper
import java.util.concurrent.*

/**
 * ThreadEngine
 * 定制化线程池，支持普通Runnable任务、Callabe任务、异步回调任务、延迟任务
 * 可配置线程名称，优先级，线程切换，回调通知
 * @author:huangyeling
 * @date: 2021/7/28
 */
class ThreadEngine(
    threadOption: Int,
    threadSize: Int,
    priority: Int,
    threadName: String,
    runnableCallback: RunnableCallback?,
    deliver: Executor?,
    threadPool: ExecutorService?
) : Executor{

    /**
     * 配置线程池
     */
    private var threadPool: ExecutorService

    /**
     * 配置线程名称
     */
    private var threadName: String = threadName

    /**
     * 配置线程回调
     */
    private var runnableCallback: RunnableCallback? = runnableCallback

    /**
     * 默认配置更新UI线程deliver
     */
    private var deliver: Executor? = deliver

    private var local: ThreadLocal<LocalConfigs>? = null


    private val localLocalConfigs: LocalConfigs
        get() {
        var localConfigs: LocalConfigs? = local?.get()
        if(localConfigs == null){
            localConfigs = LocalConfigs()
            localConfigs.threadName = threadName
            localConfigs.runnableCallback = runnableCallback
            localConfigs.deliver = deliver
            local?.set(localConfigs)
        }
        return localConfigs
    }

    /**
     * 执行Runnable任务
     */
    override fun execute(command: Runnable) {
        var runnable: Runnable = command
        val config: LocalConfigs = localLocalConfigs
        runnable = RunnableWrapper(config).setRunnable(runnable)
        DelayDispatcher.instance.postDelay(config.delay,threadPool,runnable)
        resetLocalConfigs()
    }

    /**
     * 执行异步回调任务
     */
    fun <T> async(callable: Callable<T>, callback: AsyncCallback){
        val localConfigs: LocalConfigs = localLocalConfigs
        localConfigs.asyncCallback = callback
        val runnable: Runnable = RunnableWrapper(localConfigs).setCallable(callable)
        DelayDispatcher.instance.postDelay(localConfigs.delay,threadPool,runnable)
    }

    /**
     * 执行callable任务
     */
   fun <T> submit(callable: Callable<T>): Future<T> {
        val result: Future<T>
        val callable = CallableWrapper(localLocalConfigs,callable)
        result = threadPool.submit(callable)
        resetLocalConfigs()
        return result
   }

    /**
     * 配置线程名称
     */
    fun setThreadName(name: String): ThreadEngine {
        localLocalConfigs.threadName = name
        return this
    }

    /**
     * 设置线程回调
     */
    fun setCallBack(runnableCallback: RunnableCallback): ThreadEngine {
        localLocalConfigs.runnableCallback = runnableCallback
        return this
    }

    /**
     * 设置延期时间
     */
    fun setDelay(time: Long,unit: TimeUnit): ThreadEngine {
        val delay = unit.toMillis(time)
        localLocalConfigs.delay = Math.max(0, delay);
        return this
    }

    /**
     * 关闭线程池 线程状态shutDown 队列里的任务执行完才停止
     */
    fun shutDown(){
        threadPool.shutdown()
    }

    /**
     *  关闭线程池 线程状态stop 尝试interrupt中断
     */
    fun shutDownNow(){
        threadPool.shutdownNow()
    }

    fun resetLocalConfigs(){
        local?.set(null)
    }


    companion object{

        const val THREAD_FIXED = 1
        const val THREAD_CACHEABLE = 2
        const val THREAD_SCHEDULED = 3
        const val THREAD_SINGLE = 4

        fun create(pool: ExecutorService): Builder {
            return Builder(
                1,
                THREAD_SINGLE,
                pool
            )
        }


        /**
         * 创建一个可缓存线程池，如果线程池长度超过处理需要，
         * 可灵活回收空闲线程，若无可回收，则新建线程。
         */
        fun createCacheable(): Builder {
            return Builder(
                0,
                THREAD_CACHEABLE,
                null
            )
        }


        /**
         * 创建一个定长线程池，可控制线程最大并发数，
         * 超出的线程会在队列中等待。
         */
        fun createFixed(size: Int): Builder {
            return Builder(
                size,
                THREAD_FIXED,
                null
            )
        }


        /**
         *  创建一个定长线程池，支持定时及周期性任务执行。
         */
        fun createScheduled(size: Int): Builder {
            return Builder(
                size,
                THREAD_SCHEDULED,
                null
            )
        }


        /**
         *  创建一个单线程化的线程池，它只会用唯一的工作线程来执行任务，
         *  保证所有任务按照指定顺序(FIFO, LIFO, 优先级)执行。
         */
        fun createSingle(): Builder {
            return Builder(
                0,
                THREAD_SINGLE,
                null
            )
        }

    }

    private fun createPool(
        type: Int,
        size: Int,
        priority: Int
    ): ExecutorService {
        return when (type) {
            THREAD_CACHEABLE -> Executors.newCachedThreadPool(
                DefaultFactory(priority)
            )
            THREAD_FIXED -> Executors.newFixedThreadPool(
                size,
                DefaultFactory(priority)
            )
            THREAD_SCHEDULED -> Executors.newScheduledThreadPool(
                size,
                DefaultFactory(priority)
            )
            THREAD_SINGLE -> Executors.newSingleThreadExecutor(
                DefaultFactory(priority)
            )
            else -> Executors.newSingleThreadExecutor(
                DefaultFactory(
                    priority
                )
            )
        }
    }

    private class DefaultFactory constructor(private val priority: Int) :
        ThreadFactory {
        override fun newThread(runnable: Runnable): Thread {
            val thread = Thread(runnable)
            thread.priority = priority
            return thread
        }

    }


    init {
        var pool = threadPool
        if (pool == null) {
            pool = createPool(threadOption, threadSize, priority)
        }
        this.threadPool = pool
        this.threadName = threadName
        this.runnableCallback = runnableCallback
        this.deliver = deliver
        this.local = ThreadLocal()

    }





}