package com.tn.lib.util.networkinfo

import android.content.Context
import android.net.ConnectivityManager
import android.net.NetworkCapabilities
import android.net.NetworkInfo
import android.os.Build
import android.telephony.TelephonyManager
import com.blankj.utilcode.util.Utils


/**
 * @author xinbing.zhang
 * @date :2023/1/30 14:42
 * @description: 获取网络状态的工具类
 *
 * <uses-permission android:name="android.permission.INTERNET"/>
 * <uses-permission android:name="android.permission.ACCESS_NETWORK_STATE"/>
 * <uses-permission android:name="android.permission.ACCESS_WIFI_STATE"/>
 * <uses-permission android:name="android.permission.CHANGE_NETWORK_STATE"/>
 *
 * 当前工具类 提供三种能力
 * 1. 网络状态的变化回调
 * 2. 获取当前网络状态
 * 3. 获取当前网络类型
 */
object NetworkUtil {

    /**
     * 异步获取网络信息初始化
     */
    fun initNetwork() {
        NetWorkCallback.INSTANCE.registerNetworkCallback()
    }

    /**
     * 网络工具释放资源
     */
    fun destroy() {
        NetWorkCallback.INSTANCE.unregisterNetworkCallback()
    }

    /**
     * 网络状态监听
     */
    fun registerNetworkStatusChangedListener(listener: OnNetworkStatusChangedListener?) {
        NetWorkCallback.INSTANCE.registerNetworkStatusChangedListener(listener)
    }

    /**
     * 移除网络监听
     */
    fun unregisterNetworkStatusChangedListener(listener: OnNetworkStatusChangedListener?) {
        NetWorkCallback.INSTANCE.unregisterNetworkStatusChangedListener(listener)
    }


    // =============================================================================================


    /**
     * 同步获取网络是否可用
     */
    fun isNetworkCapability(context: Context): Boolean {
        val mConnectivityManager =
            context.getSystemService(Context.CONNECTIVITY_SERVICE) as? ConnectivityManager
                ?: return false
        return if (Build.VERSION.SDK_INT < Build.VERSION_CODES.M) {
            // NetworkInfo --> This class was deprecated in API level 29.
            // getActiveNetworkInfo() --> This method was deprecated in API level 29.
            // 用于ConnectivityManager#getActiveNetworkInfo()获取代表当前网络连接的实例。
            val mNetworkInfo = mConnectivityManager.activeNetworkInfo
            // isConnected() --> This method was deprecated in API level 28.
            // isConnected表示网络是否已经连接（能够传输数据），能够连接的网络一定是可用的
            // （这个可用是指客户端可以和这个热点通信，但是不一定能接入互联网，比如一台手机接入到一个没有插网线的路由器）。
            // isAvailable表示网络是否可用（与当前有没有连接没关系）。
            null != mNetworkInfo && mNetworkInfo.isConnected
        } else {
            // getActiveNetwork 这个API是 23才新增的
            val activeNetwork = mConnectivityManager.activeNetwork
            val networkCapabilities = mConnectivityManager.getNetworkCapabilities(activeNetwork)
            // NET_CAPABILITY_VALIDATED --> 表示能够和互联网通信（这个为true表示能够上网）
            // NET_CAPABILITY_INTERNET --> 表示是否连接上了互联网（不关心是否可以上网）
            null != networkCapabilities && networkCapabilities.hasCapability(NetworkCapabilities.NET_CAPABILITY_VALIDATED)
        }
    }

    /**
     * 同步判断是否是WiFi连接
     */
    fun isWifiConnected(context: Context): Boolean {
        val mConnectivityManager =
            context.getSystemService(Context.CONNECTIVITY_SERVICE) as? ConnectivityManager
                ?: return false
        return if (Build.VERSION.SDK_INT < Build.VERSION_CODES.M) {
            val mWiFiNetworkInfo =
                mConnectivityManager.getNetworkInfo(ConnectivityManager.TYPE_WIFI)
            null != mWiFiNetworkInfo && mWiFiNetworkInfo.isConnected
        } else {
            val activeNetwork = mConnectivityManager.activeNetwork
            val networkCapabilities = mConnectivityManager.getNetworkCapabilities(activeNetwork)
            // 这里并不知道 当前是否可以上网
            null != networkCapabilities && networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_WIFI)
        }
    }

    /**
     * 同步判断是否是数据网络连接
     */
    fun isMobileConnected(context: Context): Boolean {
        val mConnectivityManager =
            context.getSystemService(Context.CONNECTIVITY_SERVICE) as? ConnectivityManager
                ?: return false
        return if (Build.VERSION.SDK_INT < Build.VERSION_CODES.M) {
            val mWiFiNetworkInfo =
                mConnectivityManager.getNetworkInfo(ConnectivityManager.TYPE_MOBILE)
            null != mWiFiNetworkInfo && mWiFiNetworkInfo.isConnected
        } else {
            val activeNetwork = mConnectivityManager.activeNetwork
            val networkCapabilities = mConnectivityManager.getNetworkCapabilities(activeNetwork)
            null != networkCapabilities && networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR)
        }
    }

    /**
     * 同步获取当前网络连接状态
     */
    fun isNetworkConnected(context: Context): Boolean {
        val mConnectivityManager =
            context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.M) {
            val mNetworkInfo = mConnectivityManager.activeNetworkInfo
            if (mNetworkInfo != null) {
                return mNetworkInfo.isAvailable
            }
        } else {
            val network = mConnectivityManager.activeNetwork ?: return false
            val status = mConnectivityManager.getNetworkCapabilities(network) ?: return false
            // 仅判断网络连接 if (status.hasCapability(NetworkCapabilities.NET_CAPABILITY_VALIDATED)) {
            if (status.hasCapability(NetworkCapabilities.NET_CAPABILITY_INTERNET)) {
                return true
            }
        }
        return false
    }

    /**
     * 异步获取网络是否可用
     */
    fun hasCapabilityAsync(): Boolean {
        return NetWorkCallback.INSTANCE.isConnected()
    }

    /**
     * 获取当前网络状态
     */
    fun getNetState(): Int {
        return NetWorkCallback.INSTANCE.getNetState()
    }

    /**
     * 获取当前网络类型
     */
    fun getNetType(): NetworkType? {
        return NetWorkCallback.INSTANCE.getNetworkType()
    }


    // =============================================================================================


    /***获取系统的服务*/
    private fun getActiveNetworkInfo(): NetworkInfo? {
        val cm =
            Utils.getApp().getSystemService(Context.CONNECTIVITY_SERVICE) as? ConnectivityManager
        return cm?.activeNetworkInfo
    }

    /** Return whether using ethernet.
     * <p>Must hold
     * {@code <uses-permission android:name="android.permission.ACCESS_NETWORK_STATE" />}</p>
     *
     * @return {@code true}: yes<br>{@code false}: no*/
    private fun isEthernet(): Boolean {
        val cm =
            Utils.getApp().getSystemService(Context.CONNECTIVITY_SERVICE) as? ConnectivityManager
                ?: return false
        val info = cm.getNetworkInfo(ConnectivityManager.TYPE_ETHERNET) ?: return false
        val state = info.state ?: return false
        return state == NetworkInfo.State.CONNECTED || state == NetworkInfo.State.CONNECTING
    }

    /*** 获取当前网络类型*/
    fun getNetworkType(): NetworkType {
        // 判断当前是否有网络
        if (isEthernet()) {
            return NetworkType.NETWORK_ETHERNET
        }

        // NetworkInfo 过时了 需要使用新的api getDataNetworkType()
        // https://developer.android.google.cn/reference/kotlin/android/net/NetworkInfo?hl=en
        // https://developer.android.google.cn/reference/kotlin/android/telephony/TelephonyManager.html#getDataNetworkType()
//        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
//            val telMgr = Utils.getApp().getSystemService(TELEPHONY_SERVICE) as? TelephonyManager
//            telMgr?.dataNetworkType
//            return NetworkType.NETWORK_NO
//        } else {
        val info = getActiveNetworkInfo()
        return if (info != null) {
            when (info.type) {
                ConnectivityManager.TYPE_WIFI -> {
                    NetworkType.NETWORK_WIFI
                }

                ConnectivityManager.TYPE_MOBILE -> {
                    when (info.subtype) {
                        TelephonyManager.NETWORK_TYPE_GSM, TelephonyManager.NETWORK_TYPE_GPRS, TelephonyManager.NETWORK_TYPE_CDMA, TelephonyManager.NETWORK_TYPE_EDGE, TelephonyManager.NETWORK_TYPE_1xRTT, TelephonyManager.NETWORK_TYPE_IDEN -> NetworkType.NETWORK_2G
                        TelephonyManager.NETWORK_TYPE_TD_SCDMA, TelephonyManager.NETWORK_TYPE_EVDO_A, TelephonyManager.NETWORK_TYPE_UMTS, TelephonyManager.NETWORK_TYPE_EVDO_0, TelephonyManager.NETWORK_TYPE_HSDPA, TelephonyManager.NETWORK_TYPE_HSUPA, TelephonyManager.NETWORK_TYPE_HSPA, TelephonyManager.NETWORK_TYPE_EVDO_B, TelephonyManager.NETWORK_TYPE_EHRPD, TelephonyManager.NETWORK_TYPE_HSPAP -> NetworkType.NETWORK_3G
                        TelephonyManager.NETWORK_TYPE_IWLAN, TelephonyManager.NETWORK_TYPE_LTE -> NetworkType.NETWORK_4G
                        TelephonyManager.NETWORK_TYPE_NR -> NetworkType.NETWORK_5G
                        else -> {
                            val subtypeName = info.subtypeName
                            if (subtypeName.equals(
                                    "TD-SCDMA", ignoreCase = true
                                ) || subtypeName.equals(
                                    "WCDMA", ignoreCase = true
                                ) || subtypeName.equals("CDMA2000", ignoreCase = true)
                            ) {
                                NetworkType.NETWORK_3G
                            } else {
                                NetworkType.NETWORK_UNKNOWN
                            }
                        }
                    }
                }

                else -> {
                    NetworkType.NETWORK_UNKNOWN
                }
            }
        } else NetworkType.NETWORK_NO
//        }
    }

}