package com.tn.lib.logger.impl

import android.util.Log
import com.google.gson.Gson
import com.tn.lib.logger.ILogger
import com.tn.lib.logger.LoggerApplication

class LogcatLoggerImpl : ILogger {

    companion object {
        val instance: LogcatLoggerImpl by lazy(mode = LazyThreadSafetyMode.SYNCHRONIZED) {
            LogcatLoggerImpl()
        }
        const val DEFAULT_TAG = "DefaultTag"
    }

    private var mGson: Gson = Gson()

    private fun isLoggable(tag: String): Boolean {
        return LoggerApplication.iSDebug() || Log.isLoggable(tag, Log.DEBUG)
    }

    override fun i(tag: String, vararg msg: String) {
        if (isLoggable(tag)) {
            Log.i(tag, msg.joinToString(separator = " , "))
        }
    }

    override fun i(msg: String) {
        if (isLoggable(DEFAULT_TAG)) {
            Log.i(DEFAULT_TAG, msg)
        }
    }


    override fun v(tag: String, vararg msg: String) {
        if (isLoggable(tag)) {
            Log.v(tag, msg.joinToString(separator = " , "))
        }
    }

    override fun v(msg: String) {
        if (isLoggable(DEFAULT_TAG)) {
            Log.v(DEFAULT_TAG, msg)
        }
    }

    override fun d(tag: String, vararg msg: String) {
        if (isLoggable(tag)) {
            Log.d(tag, msg.joinToString(separator = " , "))
        }
    }

    override fun d(msg: String) {
        if (isLoggable(DEFAULT_TAG)) {
            Log.d(DEFAULT_TAG, msg)
        }
    }

    override fun w(tag: String, vararg msg: String) {
        if (isLoggable(tag)) {
            Log.w(tag, msg.joinToString(separator = " , "))
        }
    }

    override fun w(msg: String) {
        if (isLoggable(DEFAULT_TAG)) {
            Log.w(DEFAULT_TAG, msg)
        }
    }

    override fun e(tag: String, vararg msg: String) {
        if (isLoggable(tag)) {
            Log.e(tag, msg.joinToString(separator = " , "))
        }
    }

    override fun e(msg: String) {
        if (isLoggable(DEFAULT_TAG)) {
            Log.e(DEFAULT_TAG, msg)
        }
    }

    override fun json(tag: String, obj: Any?) {
        if (isLoggable(tag)) {
            if (obj == null) {
                return
            }
            try {
                if (obj is String || obj is Int || obj is Float
                    || obj is Double || obj is Long || obj is Byte || obj is Short
                    || obj is Boolean || obj is Char
                ) {
                    Log.d(tag, obj.toString())
                } else {
                    Log.d(tag, mGson.toJson(obj))
                }
            } catch (e: Exception) {
                Log.e(tag, e.stackTrace.toString())
                e.printStackTrace()
            }
        }
    }

    override fun json(obj: Any?) {
        if (isLoggable(DEFAULT_TAG)) {
            if (obj == null) {
                return
            }
            try {
                if (obj is String || obj is Int || obj is Float
                    || obj is Double || obj is Long || obj is Byte || obj is Short
                    || obj is Boolean || obj is Char
                ) {
                    Log.d(DEFAULT_TAG, obj.toString())
                } else {
                    Log.d(DEFAULT_TAG, mGson.toJson(obj))
                }
            } catch (e: Exception) {
                Log.e(DEFAULT_TAG, e.stackTrace.toString())
                e.printStackTrace()
            }
        }
    }

    override fun xml(tag: String, xmlStr: String) {
        if (isLoggable(tag)) {
            Log.d(tag, xmlStr)
        }
    }

    override fun xml(xmlStr: String) {
        if (isLoggable(DEFAULT_TAG)) {
            Log.d(DEFAULT_TAG, xmlStr)
        }
    }
}