package com.tn.lib.util.device

import android.annotation.SuppressLint
import android.app.ActivityManager
import android.app.Application
import android.content.Context
import android.content.Intent
import android.os.Build
import android.provider.Settings
import android.telephony.TelephonyManager
import android.text.TextUtils
import com.blankj.utilcode.util.PhoneUtils
import com.tencent.mmkv.MMKV
import com.tn.lib.logger.Logger
import com.tn.lib.util.UtilApplication
import com.tn.lib.util.device.oaid.OAIDService
import com.tn.lib.util.md5.MD5
import com.tn.lib.util.mmkv.MKCons
import java.io.BufferedReader
import java.io.FileNotFoundException
import java.io.FileReader
import java.io.IOException
import java.util.*

object TNDeviceHelper {


    val netMmkv by lazy { MMKV.mmkvWithID(MKCons.KV_ID_VSHOW) }
    private var kvAndroidId: String? = null
    private var kvGmsAdId: String? = null
    private var kvDeviceId: String? = null
    private const val GID_ACTION = "com.google.android.gms.ads.identifier.service.START"
    private const val GMS_PACKAGE = "com.google.android.gms"
    private const val CPU_INFO = "/proc/cpuinfo"
    private const val STABLE_ANDROID_ID = "9774d56d682e549c"



    fun doGetGid(context: Context) {
        if (context !is Application) {
            Logger.d(
                "DeviceHelper",
                "please pass in application to avoid Service not registered exception "
            )
            return
        }
        val gid = netMmkv?.getString(MKCons.KV_KEY_GMS_AD_ID, "")
        if (!TextUtils.isEmpty(gid)) {
            Logger.d("DeviceHelper", "gid has get gid $gid")
            return
        }
        val intent =
            Intent(GID_ACTION)
        intent.setPackage(GMS_PACKAGE)
        OAIDService.bind(context, intent)
    }

    fun saveGaid(gid: String) {
        netMmkv?.putString(MKCons.KV_KEY_GMS_AD_ID, gid)
    }

    fun saveAndroidId(androidId: String) {
        netMmkv?.putString(MKCons.KV_KEY_ANDROID_ID, androidId)
    }

    fun getAndroidId(): String {
        if (TextUtils.isEmpty(kvAndroidId)) {
            kvAndroidId = netMmkv?.getString(MKCons.KV_KEY_ANDROID_ID, "")
        }
        var androidId = kvAndroidId
        if (TextUtils.isEmpty(androidId)) {
            androidId = Settings.System.getString(
                UtilApplication.application?.contentResolver,
                Settings.Secure.ANDROID_ID
            )
        }
        return androidId ?: ""
    }

    fun saveDeviceId(deviceId: String) {
        netMmkv?.putString(MKCons.KV_KEY_DEVICE_ID, deviceId)
    }

    fun getGaid(): String {
        if (TextUtils.isEmpty(kvGmsAdId)) {
            kvGmsAdId = netMmkv?.getString(MKCons.KV_KEY_GMS_AD_ID, "")
        }
        val gaid = kvGmsAdId.toString()
        if (!TextUtils.isEmpty(gaid)) {
            return gaid
        }
        val android = getAndroidId()
        if (!TextUtils.isEmpty(android)) {
            return android
        }

        return UUID.randomUUID().toString()
    }


    @SuppressLint("MissingPermission")
    fun getImei(): String {
        return try {
            PhoneUtils.getIMEI()
        } catch (e: Exception) {
            e.printStackTrace()
            ""
        }
    }

    @SuppressLint("MissingPermission")
    fun getImsi(): String {
        return try {
            PhoneUtils.getIMSI()
        } catch (e: Exception) {
            e.printStackTrace()
            ""
        }
    }

    fun getBrand(): String {
        return Build.BRAND
    }


    /**
     * 手机型号
     */
    fun getModel(): String {
        return Build.MODEL
    }

    /**
     * 手机系统版本
     */
    fun getOsVersion(): String {
        return Build.VERSION.RELEASE
    }

    /**
     * sim
     */
    fun getSimOperator(): String {
        val tm =
            UtilApplication.application?.getSystemService(Context.TELEPHONY_SERVICE) as TelephonyManager
        return tm.simOperator
    }

    fun getHardWare(): String? {
        try {
            val fr = FileReader(CPU_INFO)
            val br = BufferedReader(fr)
            var text: String
            var last = ""
            while (br.readLine().also { text = it } != null) {
                last = text
                //一般机型的cpu型号都会在cpuinfo文件的最后一行
                if (last.contains("Hardware")) {
                    val hardWare = last.split(":\\s+".toRegex(), 2).toTypedArray()
                    return hardWare[1]
                }
            }
        } catch (e: FileNotFoundException) {
            e.printStackTrace()
        } catch (e: IOException) {
            e.printStackTrace()
        }
        return Build.HARDWARE
    }

    /**
     * 获取设备唯一Id
     * 先从sp取设备id,没有向下执行
     * 先取sp中gid，sp没有取google gid 没有向下执行
     * gid没有取sp中androidId,sp没有取设备androidId，没有向下执行
     * androidId没有取sp中uuid，sp没有取java生成的uuid
     */
    fun getDeviceId(): String {
        if (TextUtils.isEmpty(kvDeviceId)) {
            kvDeviceId = netMmkv?.getString(MKCons.KV_KEY_DEVICE_ID, "")
        }
        var deviceId = kvDeviceId.toString()
        if (TextUtils.isEmpty(kvAndroidId)) {
            kvAndroidId = netMmkv?.getString(MKCons.KV_KEY_ANDROID_ID, "")
        }
        var androidId = kvAndroidId.toString()
        if (!TextUtils.isEmpty(deviceId)) {
            return deviceId
        }
        if (!TextUtils.isEmpty(androidId)) {
            val decodeAndroidId = MD5.getStringMD5(androidId)
            saveDeviceId(decodeAndroidId)
            return decodeAndroidId
        }

        //不同app，签名不同，androidId也会不同，所以这里生成id和vskit生成的不同，vskit中拿不到gaid，应该是bug，后期为了考虑版本兼容，不适用gid，仍然使用androidId
        androidId = Settings.System.getString(
            UtilApplication.application?.contentResolver,
            Settings.Secure.ANDROID_ID
        )
        if (!TextUtils.isEmpty(androidId) && androidId != STABLE_ANDROID_ID) {
            deviceId = MD5.getStringMD5(androidId)
            saveDeviceId(deviceId)
            saveAndroidId(androidId)
        } else {
            val uuid = UUID.randomUUID().toString()
            deviceId = MD5.getStringMD5(uuid)
            saveDeviceId(deviceId)
        }

        return deviceId
    }

    fun getMemoryInfo(context: Context): ActivityManager.MemoryInfo {
        val am = context.getSystemService(Context.ACTIVITY_SERVICE) as ActivityManager
        val mi = ActivityManager.MemoryInfo()
        am.getMemoryInfo(mi)
        return mi
    }




    /**
     * 获取系统语言
     */
    fun getLanguage(): String {
        val currentLocale = Locale.getDefault()
        val currentLanguage = currentLocale.language
        return currentLanguage
    }

    /**
     * 获取国家
     */
    fun getCountry(): String {
        val currentLocale = Locale.getDefault()
        return currentLocale.country
    }

    /**
     * arm64-v8a,armeabi-v7a,armeabi
     */
    fun getCPU(): String {
        return Build.SUPPORTED_ABIS.contentDeepToString().replace("[", "").replace("]", "")
    }

    fun getCountryCode(): String? {
        val telephonyManager = UtilApplication.application?.getSystemService(Context.TELEPHONY_SERVICE) as? TelephonyManager
        if (telephonyManager?.simCountryIso.isNullOrEmpty()) {
            return telephonyManager?.networkCountryIso
        }
        return telephonyManager?.simCountryIso
    }
}