package com.transsion.baselib.report

import android.app.Application
import android.os.Bundle
import android.os.SystemClock
import android.text.TextUtils
import android.util.Log
import com.blankj.utilcode.util.Utils
import com.tn.lib.logger.Logger
import com.tn.lib.thread.ThreadSingleExecutor
import com.tn.lib.util.device.TNDeviceHelper
import com.tn.lib.util.mmkv.MKCons
import com.tn.lib.util.mmkv.MMKVHelper
import com.transsion.athenacust.AthenaCust
import com.transsion.ga.AthenaAnalytics
import com.transsion.gslb.GslbSdk
import org.json.JSONObject
import java.util.*


/**
 *   Transsion MI
 *   Created By Liupeng
 *   On 2022/8/11 13:51
 *
 *   Desc 上报工具类
 *   业务埋点文档 https://transsioner.feishu.cn/sheets/shtcnhA93Vk1ia9XON3Ln0PWm9g?sheet=zQ7Tfx
 *   埋点sdk文档 https://transsioner.feishu.cn/wiki/wikcnOYd5HX7w30uhwNSk71Q6wd#
 *
 */
object Report {

    val TAG = "Report"

    private const val EXT = "ext"


    /**
     * 上一个页面的名称
     */
    private var mLastPageName: String = ""

    var api: ReportApi? = null

    fun initSDK(
        context: Application, debug: Boolean, api: ReportApi?, spcode:String? = null, pkgIn: Boolean = false
    ) {
        val deviceId = TNDeviceHelper.getDeviceId()
        var mcc = ""
        if ((spcode?.length ?: 0) >= 3) {
            mcc = spcode?.substring(0, 3).toString()
        }
        if (pkgIn) {
            GslbSdk.init(
                context,
                "404",
                deviceId,
                initDomains(),
                object : GslbSdk.InitListener {
                    override fun onInitSuccess(map: Map<String, String>) {
                        Logger.d(TAG, "gslbSdk success did 11 $deviceId map $map")
                    }

                    override fun onInitFail() {
                    }
                })
        } else if (TextUtils.equals("250", mcc)) {
            //检测到为ee 故意切换到非ee服务
            GslbSdk.init(
                context,
                "621",
                deviceId,
                initDomains(),
                object : GslbSdk.InitListener {
                    override fun onInitSuccess(map: Map<String, String>) {
                        Logger.d(TAG, "gslbSdk success did 22 $deviceId map $map")
                    }

                    override fun onInitFail() {
                    }
                })
        }
        AthenaAnalytics.setWorkMode(AthenaAnalytics.MODE_NORMAL)
        /**
         * @param context
         * @param channel
         * @param appId
         * @param isDebug 是否打印日志
         * @param isHost 是否宿主
         */
        AthenaAnalytics.init(context, "Oneroom", 2570, true, true)
        // 数据上报到正式 or 测试环境
        AthenaAnalytics.setTest(false)
        // 是否启用Athena
        AthenaAnalytics.enable(true)
        AthenaAnalytics.setDelayTime(2000)
        AthenaAnalytics.submitAll()
        this.api = api
        Logger.d(TAG, "initSDK debug $debug")
    }

    private fun initDomains(): Array<String> {
        val domains = arrayOf(
            "dsu-a.shalltry.com",
        )
        return domains
    }

    /**
     * 页面曝光事件
     * @param logConfig ILogViewConfig 页面信息
     */
    fun reportPV(logConfig: LogViewConfig) {
        logConfig.paramsMap["page_from"] = logConfig.fromPageName ?: mLastPageName
        logConfig.ops?.let {
            logConfig.paramsMap["ops"] = it
        }

        reportUIEvent(
            "reportPV", logConfig.pageName, ReportConstants.PV, logConfig.paramsMap
        )
    }

    /**
     * 页面曝光事件
     * @param logConfig ILogViewConfig 页面信息
     */
    fun reportPT(logConfig: LogViewConfig) {
        logConfig.paramsMap["page_from"] = logConfig.fromPageName ?: mLastPageName
        logConfig.paramsMap["duration"] =
            (SystemClock.elapsedRealtime() - logConfig.resumeTime).toString()
        logConfig.paramsMap["is_load_success"] = logConfig.isLoadSuccess.toString()
        logConfig.ops?.let {
            logConfig.paramsMap["ops"] = it
        }

        reportUIEvent(
            "reportPT", logConfig.pageName, ReportConstants.PT, logConfig.paramsMap
        )
        if (!TextUtils.equals(mLastPageName, logConfig.pageName) && logConfig.needAddToPageFrom) {
            mLastPageName = logConfig.pageName
        }
    }


    /**
     * 曝光/展现类事件
     * @param pageName String 页面名称
     * @param event String  事件名称
     * @param map Map<String, String> 其他参数
     */
    fun reportShow(
        pageName: String,
        event: String,
        map: Map<String, String?>,
    ) {
        reportUIEvent("reportShow", pageName, event, map)
    }


    /**
     * 点击/交互 类事件
     * @param pageName String 页面名称
     * @param event String  事件名称
     * @param map Map<String, String> 其他参数
     */
    fun reportClick(
        pageName: String,
        event: String,
        map: Map<String, String?>,
    ) {
        reportUIEvent("reportClick", pageName, event, map)
    }

    /**
     * 时长 类事件
     * @param pageName String 页面名称
     * @param event String  事件名称
     * @param map Map<String, String> 其他参数
     */
    fun reportUseTime(
        pageName: String,
        event: String,
        useTime: Long,
        map: MutableMap<String, String>,
    ) {
        map["duration"] = useTime.toString()
        reportUIEvent("reportUseTime", pageName, event, map)
    }

    private fun reportUIEvent(
        category: String,
        pageName: String,
        event: String,
        map: Map<String, String?>,
    ) {
        ThreadSingleExecutor.INSTANCE.execute {
            Logger.d(
                TAG, "$category event=$event  pageName=$pageName map=$map"
            )
            val bundle = Bundle()
            bundle.putString("page_name", pageName)
            //bundle.putString("event", event)
            val map2Jsonstr = map2Jsonstr(map)
            bundle.putString(EXT, map2Jsonstr)
            track(event, bundle)
            addToRecentEvent(pageName, event, map)
        }
    }


    private fun addToRecentEvent(pageName: String, event: String, map: Map<String, String?>) {
        api?.addToRecentEvent(pageName, event, map)
    }

    /**
     *  启动等无页面相关的事件
     * @param event String  事件名称
     * @param map Map<String, String> 其他参数
     */
    fun reportEvent(event: String, map: Map<String, String?>) {
        ThreadSingleExecutor.INSTANCE.execute {
            Logger.d(TAG, "reportEvent() --> event = $event -- map = $map")
            val bundle = Bundle()
            //bundle.putString("event", event)
            val map2Jsonstr = map2Jsonstr(map)
            bundle.putString(EXT, map2Jsonstr)
//            if (TextUtils.isEmpty(pageName).not()) {
//                bundle.putString("page_name", pageName)
//            }
            track(event, bundle)
        }
    }

    /**
     *  启动等无页面相关的事件
     * @param event String  事件名称
     * @param map Map<String, String> 其他参数
     */
    fun reportEventV2(event: String, map: Map<String, Any?>, pageName: String = "") {
        ThreadSingleExecutor.INSTANCE.execute {
            val bundle = Bundle()
            //val map2JsonStr = map2Jsonstr(map)
            val map2JsonStr = JSONObject(map).toString()
            bundle.putString(EXT, map2JsonStr)
            if (TextUtils.isEmpty(pageName).not()) {
                bundle.putString("page_name", pageName)
            }
            Logger.d(TAG + "_ad", "reportEvent() --> event = $event --> bundle = $bundle")
            track(event, bundle)
        }
    }

    /**
     * 登录激活
     */
    fun loginActive(userType: Int, userid: String?) {
        ThreadSingleExecutor.INSTANCE.execute {
            Logger.d(TAG, "loginActive:userType$userType,userid:$userid")
            val event = "login_active"
            val map = HashMap<String, String>()
            map["userType"] = userType.toString()
            map["userId"] = userid ?: "0"
            val json = map2Jsonstr(map)
            val bundle = Bundle()
            //bundle.putString("pageName", pageName)
            bundle.putString("event", event)
            bundle.putString(EXT, json)

            track(event, bundle)
        }
    }

    /**
     * 最底层都会调用这个方法.
     */
    private fun track(event: String, bundle: Bundle) {
        try {
            bundle.putString("vaid", AthenaAnalytics.getAppVAID(Utils.getApp()))
            val commonBundle = api?.addCommonParams(event, bundle) ?: bundle
            // 更新 Athena 用户数据
            api?.getAccount().apply {
                if (this != null) {
                    AthenaAnalytics.setAccount(this.first, this.second)
                }
            }
            AthenaCust(event).trackCommon(commonBundle, null).submit()
        } catch (ignore: Exception) {
            Logger.e("report", "event $event exception $ignore", true)
        }
    }

    fun getLastPageName(): String {
        return mLastPageName
    }

    fun setLastPageName(lastPageName: String) {
        mLastPageName = lastPageName
    }

    fun map2Jsonstr(map: Map<String, String?>): String {
        return JSONObject(map).toString()
    }
}
