package com.transsion.lib_web.zip

import android.text.TextUtils
import com.blankj.utilcode.util.Utils
import com.tencent.mmkv.MMKV
import com.tn.lib.net.manager.NetServiceGenerator
import com.transsion.lib_web.WebLogger
import com.transsion.lib_web.zip.db.WebResDatabase
import com.transsion.lib_web.zip.db.WebResRemoteBean
import com.transsion.lib_web.zip.db.WebViewCacheConfigData
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch

/**
 * @author: zhangxinbing
 * @date : 2024/12/10 20:10
 * @description: 远程资源增量更新
 */
object WebResManager {

    private fun getClassTag(): String = javaClass.simpleName

    private val iWebResApi by lazy { NetServiceGenerator.instance.getService(IWebResApi::class.java) }

    private const val MMKV_KEY_LAST_UPDATE_TIME = "lastUpdateTime"

    val mmkv: MMKV by lazy { MMKV.mmkvWithID("mb_web_res") }


    // =============================================================================================


    /**
     * 初始化
     */
    fun init(){
        ObserveNetworkState.setNetworkStateListener()
    }

    /**
     * 获取离线资源
     */
    fun getWebResConfig() {
        CoroutineScope(Dispatchers.IO).launch {
            kotlin.runCatching {
                val curLastUpdateTime = mmkv.getString(MMKV_KEY_LAST_UPDATE_TIME, "")
                WebLogger.logD("${getClassTag()} --> loadWebRes() --> start api ..... curLastUpdateTime = $curLastUpdateTime")

                val webResConfig = iWebResApi.getWebResConfig(lastUpdateTime = curLastUpdateTime)
                webResConfig.data?.let {
                    handleData(it, curLastUpdateTime)
                }
            }.getOrElse {
                WebLogger.logE("${getClassTag()} --> getWebResConfig() --> it = $it")
            }
        }
    }


    // =============================================================================================


    /**
     * 数据处理
     */
    private suspend fun handleData(response: WebResRemoteBean, curLastUpdateTime: String?) {

        // 和服务端约定，response.lastUpdateTime == 0 --> 服务端的数据都下架了
        if (TextUtils.equals("0", response.lastUpdateTime)) {
            WebLogger.logD("${getClassTag()} --> 服务端的数据都下架了 --> curLastUpdateTime = $curLastUpdateTime --> lastUpdateTime = ${response.lastUpdateTime}")
            clearCacheFile()
            return
        }

        // 已经是最新数据
        if (TextUtils.equals(curLastUpdateTime, response.lastUpdateTime)) {
            WebLogger.logD("${getClassTag()} --> 已经是最新数据 --> curLastUpdateTime = $curLastUpdateTime --> lastUpdateTime = ${response.lastUpdateTime}")
            // 检查资源是否下载完成
            checkDownloadZip()
            return
        }

        // 增量更新
        WebResDatabase.getInstance(Utils.getApp()).webResDao().apply {
            WebLogger.logD("${getClassTag()} --> handleData() --> success --> curLastUpdateTime = $curLastUpdateTime --> response = $response")

            // 解析数据
            val remoteList = response.list ?: emptyList()
            updateDb(remoteList)

            // 找出需要删除的本地数据
            deleteExpiredData(remoteList)

            // 检查资源是否下载完成
            checkDownloadZip()

            // 保存版本号到本地
            mmkv.putString(MMKV_KEY_LAST_UPDATE_TIME, response.lastUpdateTime)
            WebLogger.logD("${getClassTag()} --> handleData() --> 数据库增量更新完成 --> 删除过期资源")
        }
    }

    /**
     * 更新本地数据库数据
     */
    private suspend fun updateDb(remoteList: List<WebViewCacheConfigData>) {
        WebResDatabase.getInstance(Utils.getApp()).webResDao().apply {
            remoteList.forEach { remoteListItem ->
                val item = getItem(remoteListItem.scene)
                // 当前场景已经存在数据 --> 并且zipMd5的地址变化了
                if (item != null && TextUtils.equals(
                        remoteListItem.zipMd5, item.zipMd5
                    ).not()
                ) {
                    // --> 删除将过期文件 --> 重新下载
                    WebViewCacheDownloader.clearCacheFile(Utils.getApp(), item)
                }
                // --> 直接保存 覆盖过期数据
                insert(remoteListItem)
            }
        }
    }

    /**
     * 删除过期数据
     */
    private suspend fun deleteExpiredData(remoteList: List<WebViewCacheConfigData>) {
        WebResDatabase.getInstance(Utils.getApp()).webResDao().apply {
            val itemsToDelete = getAllItem().filter { localItem ->
                remoteList.none { remoteItem ->
                    remoteItem.zipMd5 == localItem.zipMd5
                }
            }
            if (itemsToDelete.isNotEmpty()) {
                // 删除本地多余的数据
                deleteList(itemsToDelete)
                // 删除本地资源
                itemsToDelete.forEach {
                    WebViewCacheDownloader.clearCacheFile(Utils.getApp(), it)
                }
            }
        }
    }

    /**
     * 清空本地文件
     */
    private suspend fun clearCacheFile() {
        WebResDatabase.getInstance(Utils.getApp()).webResDao().apply {
            getAllItem().forEach { item ->
                // 删除本地多余的数据
                deleteItem(item)
                // 删除本地资源
                WebViewCacheDownloader.clearCacheFile(Utils.getApp(), item)
            }
        }
    }

    /**
     * 检查文件下载
     */
    private suspend fun checkDownloadZip() {
        WebResDatabase.getInstance(Utils.getApp()).webResDao().apply {
            getAllItem().forEach { dbItem ->
                // 下载任务
                WebViewCacheDownloader.checkDownloadZip(Utils.getApp(), dbItem)
            }
        }
    }

}