package com.transsion.lib_web.zip.loader

import android.content.Context
import android.net.Uri
import android.webkit.WebResourceRequest
import android.webkit.WebResourceResponse
import com.blankj.utilcode.util.EncryptUtils
import com.tn.lib.net.interceptor.HttpLoggingInterceptor.Logger
import com.transsion.lib_web.WebLogger
import com.transsion.lib_web.zip.WebViewCacheDownloader
import com.transsion.lib_web.zip.utils.FileUtils
import com.transsion.lib_web.zip.utils.MimeTypeMapUtils
import java.io.File
import java.io.FileInputStream
import java.io.FileNotFoundException
import java.io.InputStream

/**
 * @author: zhangxinbing
 * @date : 2024/12/11 13:59
 * @description:
 */
class WebViewFileCacheLoader(val context: Context, private val originUrl: String) : WebViewLoader {

    private val cacheFileList: HashSet<String> = hashSetOf()

    var userCache = false
    var zipMD5: String? = null

    private fun getClassTag(): String = javaClass.simpleName


    // =============================================================================================


    override fun interceptRequest(request: WebResourceRequest?): WebResourceResponse? {
        return interceptRequest(request?.url?.toString(), request?.requestHeaders ?: hashMapOf())
    }

    override fun interceptRequest(
        url: String?, headers: MutableMap<String, String>
    ): WebResourceResponse? {
        if (null == url || url == "yy://__QUEUE_MESSAGE__") {
            return null
        }

        var isIndex = false
        //检测url是否匹配 开始查找index.html文件，将目录对应的全部文件 存进cacheFileList中去
        //
        val newPath = if (isIndex(url)) {
            val path = "index.html".let { fileName ->
                try {
                    // 这里所有的缓存文件 -->
                    WebViewCacheDownloader.getWebViewCacheFile(context).listFiles()?.filter {
                        // 因为H5缓存资源是分页面保存的 --> 所以这里需要获取到 --> 对应的文件夹下面所有的文件
                        it.name.startsWith(EncryptUtils.encryptMD5ToString(originUrl.getBaseUrl()))
                    }?.maxByOrNull {
                        // 如果文件名相同 --> 返回最近修改的那一个文件
                        it.lastModified()
                    }?.let { parentFile ->
                        // 获取所有缓存文件
                        updateConfig()
                        // 获取当前URL对应的资源
                        scanFile(parentFile, fileName)
                    }
                } catch (ignore: Exception) {
                    null
                }
            }?.absolutePath
            isIndex = path.isNullOrEmpty().not()
            path
        } else {
            File(url).name.let { fileName ->
                try {
                    cacheFileList.firstOrNull { path -> File(path + fileName).exists() }?.let {
                        it + fileName
                    }
                } catch (e: Exception) {
                    null
                }
            }
        }
        WebLogger.logD("${getClassTag()} --> interceptRequest() --> newPath = $newPath url = $url")
        if (null == newPath) {
            WebLogger.logE("${getClassTag()} --> interceptRequest() --> service url = $url")
            return null
        }
        try {
            val mimeType: String? = MimeTypeMapUtils.getMimeTypeFromUrl(newPath)
            val inputStream: InputStream = FileInputStream(File(newPath))
            WebLogger.logD("${getClassTag()} --> interceptRequest() --> local file  -->  \nurl = $url  \nnewPath = $newPath  \nmimeType = $mimeType")
            val response = WebResourceResponse(
                mimeType, "", 200, "OK", WebViewLoader.responseHeaders, inputStream
            )
            userCache = true
            return response
        } catch (e: FileNotFoundException) {
            e.printStackTrace()
        }
        //Logger.logD("${getClassTag()} --> interceptRequest() --> service url = $url")
        return null

    }

    override fun getCachePath(): File = WebViewCacheDownloader.getWebViewCacheFile(context)

    override fun clearCache() {
        FileUtils.deleteFile(WebViewCacheDownloader.getWebViewCacheFile(context))
    }


    // =============================================================================================


    private fun isIndex(url: String?): Boolean {
        try {
            if (null == url) {
                return false
            }
            return originUrl == url || (url.contains("?") && url.split("?")[0] == originUrl)
        } catch (e: Exception) {
            return false
        }
    }

    private fun updateConfig() {
        getDir(WebViewCacheDownloader.getWebViewCacheFile(context), cacheFileList)
    }

    private fun getDir(file: File, cacheFileList: HashSet<String>) {
        file.listFiles()?.filter { it.isDirectory }?.forEach {
            cacheFileList.add(it.absolutePath + File.separator)
            getDir(it, cacheFileList)
        }
    }

    private fun scanFile(file: File, name: String?): File? {
        if (file.exists().not() || name.isNullOrEmpty()) {
            return null
        }
        file.listFiles()?.forEach {
            val findFile: File? = if (it.isDirectory) {
                scanFile(it, name)
            } else {
                if (it.name == name) {
                    it
                } else {
                    null
                }
            }
            if (null != findFile) {
                return findFile
            }
        }

        return null
    }

}

fun String.getBaseUrl():String?{
    val uri = Uri.parse(this)
    val scheme = uri.scheme ?: return null
    val host = uri.host ?: return null
    val path = uri.path ?: return null
    return "$scheme://$host$path"
}