package com.transsion.lib_web.zip.loader

import android.content.Context
import android.text.TextUtils
import android.webkit.WebResourceRequest
import android.webkit.WebResourceResponse
import com.blankj.utilcode.util.NetworkUtils
import com.tn.lib.logger.Logger
import com.transsion.lib_web.zip.utils.MimeTypeMapUtils
import com.transsion.lib_web.zip.utils.NetUtils
import okhttp3.Cache
import okhttp3.CacheControl
import okhttp3.OkHttpClient
import okhttp3.Request
import java.io.File
import java.io.IOException
import java.util.concurrent.TimeUnit

/**
 * @author: zhangxinbing
 * @date : 2024/12/11 14:01
 * @description:
 */
class WebViewOkhttpLoader(val context: Context) : WebViewLoader {

    private val mCacheFile: File = File(context.cacheDir, "web_okhttp")
    private val mCacheSize: Long = 1024 * 1024 * 100
    private val mConnectTimeout: Long = 30
    private val mReadTimeout: Long = 30
    private val mHttpClient: OkHttpClient


    var mCacheExtensionConfig: CacheExtensionConfig = CacheExtensionConfig()

    init {
        mCacheExtensionConfig.addExtension("json").removeExtension("swf")

        val cache = Cache(mCacheFile, mCacheSize)
        val builder: OkHttpClient.Builder =
            OkHttpClient.Builder().cache(cache).connectTimeout(mConnectTimeout, TimeUnit.SECONDS)
                .readTimeout(mReadTimeout, TimeUnit.SECONDS)
                .addNetworkInterceptor(HttpCacheInterceptor())
        mHttpClient = builder.build()
    }


    override fun interceptRequest(request: WebResourceRequest?): WebResourceResponse? {
        return request?.let {
            interceptRequest(it.url.toString(), it.requestHeaders)
        }
    }

    override fun getCachePath(): File? {
        return null
    }

    override fun clearCache() {

    }


    override fun interceptRequest(
        url: String?, headers: MutableMap<String, String>
    ): WebResourceResponse? {
        if (null == url) {
            return null
        }
        if (!checkUrl(url)) {
            return null
        }
        try {
            val reqBuilder: Request.Builder = Request.Builder().url(url)
            val extension: String = MimeTypeMapUtils.getFileExtensionFromUrl(url)
            if (mCacheExtensionConfig.isHtml(extension)) {
                headers[HttpCacheInterceptor.KEY_CACHE] = "ordinal"
            }
            addHeader(reqBuilder, headers)

            if (!NetworkUtils.isConnected()) {
                reqBuilder.cacheControl(CacheControl.FORCE_CACHE)
            }
            val request: Request = reqBuilder.build()
            val response = mHttpClient.newCall(request).execute()
            val cacheRes = response.cacheResponse
            if (cacheRes != null) {
                Logger.d(String.format("from cache: %s", url))
            } else {
                Logger.d(String.format("from server: %s", url))
            }
            val mimeType: String? = MimeTypeMapUtils.getMimeTypeFromUrl(url)
            val webResourceResponse =
                WebResourceResponse(mimeType, "", response.body!!.byteStream())
            if (response.code == 504 && !NetworkUtils.isConnected()) {
                return null
            }
            var message = response.message
            if (TextUtils.isEmpty(message)) {
                message = "OK"
            }
            try {
                webResourceResponse.setStatusCodeAndReasonPhrase(response.code, message)
            } catch (e: Exception) {
                return null
            }
            webResourceResponse.responseHeaders =
                NetUtils.multimapToSingle(response.headers.toMultimap())
            return webResourceResponse
        } catch (e: IOException) {
            e.printStackTrace()
        }
        return null
    }

    private fun checkUrl(url: String): Boolean {
        if (TextUtils.isEmpty(url)) {
            return false
        }
        //okhttp only deal with http[s]
        if (!url.startsWith("http")) {
            return false
        }

//        if (TextUtils.equals(
//                url,
//                "https://ali-test-share.mylichking.com/vskit/share?videoid=02226c3e-bd74-11e8-bd50-02d3fbab0488&type=2&userid=958544a5-ee78-4e3d-b5c0-88ef84b487f9"
//            )
//        ) {
//            return true
//        }
        val extension: String = MimeTypeMapUtils.getFileExtensionFromUrl(url)
        if (TextUtils.isEmpty(extension)) {
            return false
        }
        if (mCacheExtensionConfig.isMedia(extension)) {
            return false
        }
        return mCacheExtensionConfig.canCache(extension)
    }

    private fun addHeader(reqBuilder: Request.Builder, headers: Map<String, String>?) {
        headers?.forEach { entry ->
            if (null != entry.key && null != entry.value) reqBuilder.addHeader(
                entry.key!!, entry.value!!
            )
        }
    }


}