/*
 * Copyright (C) 2010-2013 Alibaba Group Holding Limited
 */
package com.transsion.lib_web.zip.utils

import org.apache.tools.zip.ZipEntry
import org.apache.tools.zip.ZipFile
import org.apache.tools.zip.ZipOutputStream
import java.io.BufferedInputStream
import java.io.BufferedOutputStream
import java.io.File
import java.io.FileInputStream
import java.io.FileOutputStream
import java.io.IOException
import java.io.InputStream
import java.io.OutputStream

object ZipUtils {
    /**
     * 使用GBK编码可以避免压缩中文文件名乱码
     */
    private const val CHINESE_CHARSET = "GBK"

    /**
     * 文件读取缓冲区大小
     */
    private const val CACHE_SIZE = 1024

    /**
     *
     *
     * 压缩文件
     *
     *
     * @param sourceFolder 压缩文件夹
     * @param zipFilePath  压缩文件输出路径
     * @throws Exception
     */
    @Throws(Exception::class)
    fun zip(sourceFolder: String?, zipFilePath: String?) {
        val out: OutputStream = FileOutputStream(zipFilePath)
        val bos = BufferedOutputStream(out)
        val zos = ZipOutputStream(bos)
        // 解决中文文件名乱码
        zos.encoding = CHINESE_CHARSET
        val file = File(sourceFolder)
        var basePath: String? = null
        basePath = if (file.isDirectory) {
            file.path
        } else {
            file.parent
        }
        zipFile(file, basePath, zos)
        zos.closeEntry()
        zos.close()
        bos.close()
        out.close()
    }

    @Throws(Exception::class)
    fun zipFile(inputFile: String?, basePath: String?, outputFile: String?): Int {
        val out: OutputStream = FileOutputStream(outputFile)
        val bos = BufferedOutputStream(out)
        val zos = ZipOutputStream(bos)
        // 解决中文文件名乱码
//        zos.setEncoding(CHINESE_CHARSET);
        val file = File(inputFile)
        zipFile(file, basePath, zos)
        zos.closeEntry()
        zos.close()
        bos.close()
        out.close()
        return 0
    }

    /**
     *
     *
     * 递归压缩文件
     *
     *
     * @param parentFile
     * @param basePath
     * @param zos
     * @throws Exception
     */
    @Throws(Exception::class)
    private fun zipFile(parentFile: File?, basePath: String?, zos: ZipOutputStream) {
        var files = arrayOfNulls<File>(0)
        if (parentFile!!.isDirectory) {
            files = parentFile.listFiles()
        } else {
            files = arrayOfNulls(1)
            files[0] = parentFile
        }
        var pathName: String
        var `is`: InputStream
        var bis: BufferedInputStream
        val cache = ByteArray(CACHE_SIZE)
        for (file in files) {
            if (file!!.isDirectory) {
                pathName = file.path.substring(basePath!!.length + 1) + "/"
                zos.putNextEntry(ZipEntry(pathName))
                zipFile(file, basePath, zos)
            } else {
                pathName = file.path.substring(basePath!!.length + 1)
                `is` = FileInputStream(file)
                bis = BufferedInputStream(`is`)
                zos.putNextEntry(ZipEntry(pathName))
                var nRead = 0
                while (bis.read(cache, 0, CACHE_SIZE).also { nRead = it } != -1) {
                    zos.write(cache, 0, nRead)
                }
                bis.close()
                `is`.close()
            }
        }
    }

    /**
     *
     *
     * 解压压缩包
     *
     *
     * @param zipFilePath 压缩文件路径
     * @param destDir     压缩包释放目录
     * @throws Exception
     */
    @Throws(Exception::class)
    fun unZip(zipFilePath: String?, destDir: String) {
        val zipFile = ZipFile(zipFilePath, CHINESE_CHARSET)
        val emu = zipFile.entries
        var bis: BufferedInputStream
        var fos: FileOutputStream
        var bos: BufferedOutputStream
        var file: File
        var parentFile: File?
        var entry: ZipEntry
        var entryName: String? = null
        val cache = ByteArray(CACHE_SIZE)
        val sb = StringBuffer()
        while (emu.hasMoreElements()) {
            entry = emu.nextElement() as ZipEntry
            entryName = entry.name.substring(entry.name.lastIndexOf("/"), entry.name.length)
            if (entry.isDirectory) {
                File(destDir + entryName).mkdirs()
                continue
            }
            bis = BufferedInputStream(zipFile.getInputStream(entry))
            sb.delete(0, sb.length)
            sb.append(destDir).append(File.separator).append(entryName)
            file = File(sb.toString())
            parentFile = file.parentFile
            if (parentFile != null && !parentFile.exists()) {
                parentFile.mkdirs()
            }
            fos = FileOutputStream(file)
            bos = BufferedOutputStream(fos, CACHE_SIZE)
            var nRead = 0
            while (bis.read(cache, 0, CACHE_SIZE).also { nRead = it } != -1) {
                fos.write(cache, 0, nRead)
            }
            bos.flush()
            bos.close()
            fos.close()
            bis.close()
        }
        zipFile.close()
    }

    @Throws(Exception::class)
    fun ensureZipPathSafety(outputFile: File, destDirectory: String?) {
        val destDirCanonicalPath = File(destDirectory).canonicalPath
        val outputFilecanonicalPath = outputFile.canonicalPath
        if (!outputFilecanonicalPath.startsWith(destDirCanonicalPath)) {
            throw Exception(String.format("Found Zip Path Traversal Vulnerability with %s", outputFilecanonicalPath))
        }
    }

    /**
     * 解压zip
     *
     * @param zipPath      zip文件路径 必传
     * @param saveFilePath 如果为空那么解压到zipPath的当前目录,不为空解压到指定目录
     */
    @Throws(Exception::class)
    fun UnZipFolder(zipPath: String?, saveFilePath: String) {
        var count = -1
        var file: File? = null
        var `is`: InputStream? = null
        var fos: FileOutputStream? = null
        var bos: BufferedOutputStream? = null
        var zipFile: ZipFile? = null
        try {
            //解决中文乱码问题  格式有GBK  UTF8
            zipFile = ZipFile(zipPath, "GBK")
            val entries = zipFile.entries
            while (entries.hasMoreElements()) {
                val buf = ByteArray(1024)
                val entry = entries.nextElement() as ZipEntry
                var filename = entry.name
                var ismkdir = false
                //检查此文件是否带有文件夹
                if (filename.lastIndexOf("/") != -1) {
                    ismkdir = true
                }
                filename = saveFilePath + File.separator + filename
                //如果是文件夹先创建
                if (entry.isDirectory) {
                    file = File(filename)
                    file.mkdirs()
                    continue
                }
                file = File(filename)
                if (!file.exists()) {
                    //如果是目录先创建
                    if (ismkdir) {
                        //目录先创建
                        File(filename.substring(0, filename.lastIndexOf("/"))).mkdirs()
                    }
                }
                //创建文件
                file.createNewFile()
                `is` = zipFile.getInputStream(entry)
                fos = FileOutputStream(file)
                bos = BufferedOutputStream(fos, 1024)
                while (`is`.read(buf).also { count = it } > -1) {
                    bos.write(buf, 0, count)
                }
                bos.flush()
                bos.close()
                fos.close()
                `is`.close()
            }
            zipFile.close()
        } catch (ioe: IOException) {
            throw ioe
        } finally {
            try {
                bos?.close()
                fos?.close()
                `is`?.close()
                zipFile?.close()
            } catch (e: Exception) {
                e.printStackTrace()
            }
        }
    }
}