package com.transsion.lib_web

import android.annotation.SuppressLint
import android.content.pm.ActivityInfo
import android.graphics.Bitmap
import android.graphics.Color
import android.net.Network
import android.net.NetworkCapabilities
import android.os.Build
import android.os.Bundle
import android.text.TextUtils
import android.view.LayoutInflater
import android.view.MotionEvent
import android.view.View
import android.view.ViewConfiguration
import android.view.ViewGroup
import android.webkit.JavascriptInterface
import android.webkit.RenderProcessGoneDetail
import android.webkit.WebChromeClient
import android.webkit.WebChromeClient.CustomViewCallback
import android.webkit.WebResourceError
import android.webkit.WebResourceRequest
import android.webkit.WebResourceResponse
import android.webkit.WebSettings
import android.webkit.WebView
import android.webkit.WebViewClient
import android.widget.FrameLayout
import androidx.activity.OnBackPressedCallback
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentActivity
import androidx.viewpager2.widget.ViewPager2
import com.blankj.utilcode.util.BarUtils
import com.blankj.utilcode.util.GsonUtils
import com.blankj.utilcode.util.SizeUtils
import com.github.lzyzsd.jsbridge.BridgeWebView
import com.gyf.immersionbar.BarHide
import com.gyf.immersionbar.ImmersionBar
import com.tn.lib.util.networkinfo.NetworkUtil
import com.tn.lib.util.networkinfo.OnNetworkStatusChangedListener
import com.transsion.lib_web.databinding.LibWebWebFragmentLayoutBinding
import com.transsion.lib_web.domain.DomPerformance
import com.transsion.lib_web.domain.LoadInfoStats
import com.transsion.lib_web.domain.LoadStatus
import com.transsion.lib_web.cache.TWebViewCacheManager
import com.transsion.lib_web.zip.loader.WebViewFileCacheLoader
import com.transsion.lib_web.zip.loader.WebViewLoaderManager
import org.json.JSONObject

/**
 * @author: zhangxinbing
 * @date : 2025/4/23 11:45
 * @description: 仅封装底层能力，不处理业务
 */
abstract class BaseLibWebFragment : Fragment(), OnNetworkStatusChangedListener {

    private var fullscreenContainer: FrameLayout? = null
    lateinit var binding: LibWebWebFragmentLayoutBinding
    var mWebViewLoaderManager: WebViewLoaderManager? = null // 静态资源加载
    var strTitle: String? = ""
    var currentState = LoadStatus.INIT


    fun getClassTag(): String = javaClass.simpleName

    /**
     * Web View支持视频全屏
     */
    private var webCustomView: View? = null
    private var originalSystemUiVisibility = 0
    private var customViewCallback: CustomViewCallback? = null
    private var originalOrientation: Int = ActivityInfo.SCREEN_ORIENTATION_UNSPECIFIED // 默认值


    /**
     * Fragment onCreate() --> WebView 初始化
     */
    lateinit var mWebView: BridgeWebView

    fun getWebView(): BridgeWebView = mWebView


    // =================================== Fragment 生命周期 =========================================


    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        // 初始化 Web View
        mWebView = TWebViewCacheManager.getWebView(getUrl(), context)
    }

    override fun onCreateView(
        inflater: LayoutInflater, container: ViewGroup?, savedInstanceState: Bundle?,
    ): View? {
        binding = LibWebWebFragmentLayoutBinding.inflate(inflater)
        return binding.root
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        binding.webContainer.addView(mWebView)
        // 初始化标题
        initLayout()
        initListener()

        // 初始化 Web View相关
        initWebView()

        // 监听网络变化
        NetworkUtil.registerNetworkStatusChangedListener(this)
    }

    override fun onDestroyView() {
        super.onDestroyView()
        //getWebView().destroy()
        // 资源回收
        TWebViewCacheManager.destroy(getWebView())
    }

    override fun onDestroy() {
        super.onDestroy()
        NetworkUtil.unregisterNetworkStatusChangedListener(this)
        WebLogger.logE("${getClassTag()} --> onDestroy()")
    }


    // ================================= 网络变化 ====================================================


    override fun onConnected(network: Network, networkCapabilities: NetworkCapabilities) {}

    override fun onDisconnected() {}


    // =============================== 初始化 容器 ===================================================


    private fun initListener() {
        binding.ivBack.setOnClickListener {
            activity?.onBackPressedDispatcher?.onBackPressed()
        }
        binding.ivClose.setOnClickListener {
            activity?.finish()
        }
        binding.ivRight.setOnClickListener {
            loadInfoStats.reload = true
            getWebView().reload()
        }
    }

    @SuppressLint("ClickableViewAccessibility")
    private fun initLayout() {
        // 用于客户端控制是否展示状态栏
        val statusBarHide =
            arguments?.getBoolean(WebConstant.FIELD_STATUS_BAR_HIDDEN, false) ?: false
        val isFieldToolBarHidden =
            arguments?.getBoolean(WebConstant.FIELD_TOOL_BAR_HIDDEN, false) ?: false
        val bottomMargin = arguments?.getInt(WebConstant.BOTTOM_MARGIN, 0) ?: 0
        val isNestedScrollIntercept =
            arguments?.getBoolean(WebConstant.NESTED_SCROLL_INTERCEPT, false) ?: false

        binding.apply {
            root.fitsSystemWindows = !statusBarHide
            if (bottomMargin > 0) {
                //getWebView().setPadding(0, 0, 0, bottomMargin)
                // 动态设置 bottomMargin
                val layoutParams = webContainer.layoutParams as ConstraintLayout.LayoutParams
                layoutParams.bottomMargin = bottomMargin
                webContainer.layoutParams = layoutParams
            }

            if (isNestedScrollIntercept) {//父类横滑跟webview竖滑 处理滑动冲突
                getWebView().setOnTouchListener(object : View.OnTouchListener {
                    var startX = 0f
                    var startY = 0f
                    var isDragging = false
                    private val touchSlop =
                        ViewConfiguration.get(getWebView().context).scaledTouchSlop

                    override fun onTouch(v: View, event: MotionEvent): Boolean {
                        when (event.action) {
                            MotionEvent.ACTION_DOWN -> {
                                startX = event.x
                                startY = event.y
                                // 禁止父容器拦截事件（让WebView优先处理）
                                v.parent.requestDisallowInterceptTouchEvent(true)
                                isDragging = false
                            }

                            MotionEvent.ACTION_MOVE -> {
                                val deltaX = kotlin.math.abs(event.x - startX)
                                val deltaY = kotlin.math.abs(event.y - startY)
                                if (!isDragging && (deltaX > touchSlop || deltaY > touchSlop)) {
                                    isDragging = true
                                    if (deltaX > deltaY * 1.2f) { // 水平滑动
                                        v.parent.requestDisallowInterceptTouchEvent(false)
                                    } else { // 垂直滑动
                                        v.parent.requestDisallowInterceptTouchEvent(true)
                                    }
                                }
                            }

                            MotionEvent.ACTION_UP, MotionEvent.ACTION_CANCEL -> {
                                v.parent.requestDisallowInterceptTouchEvent(false)
                                isDragging = false
                            }
                        }
                        return false
                    }
                })
            }
        }
        //默认展示toolbar 如果外部需要隐藏 OR  URL字段判断是否需要隐藏
        val hideNavigationBar = H5BarUtil.hideNavigationBar(getUrl())
        hideToolBar(isFieldToolBarHidden || hideNavigationBar)
        WebLogger.logD("${getClassTag()} --> initLayout() --> statusBarHide = $statusBarHide --> isFieldToolBarHidden = $isFieldToolBarHidden -- bottomMargin = $bottomMargin --> hideNavigationBar = $hideNavigationBar")

        val isShowHomeHeader = arguments?.getBoolean(WebConstant.NEED_HEADER, false) ?: false
        if (isShowHomeHeader) {
            binding.subWebHeaderBg.apply {
                layoutParams.height = SizeUtils.dp2px(80f)

                val backgroundRes = arguments?.getInt(WebConstant.HEADER_BG) ?: 0
                if (backgroundRes != 0) {
                    setBackgroundResource(backgroundRes)
                }
                visibility = View.VISIBLE
            }
        }
    }

    private fun hideToolBar(isHideToolBar: Boolean) {
        binding.apply {
            if (isHideToolBar) {
                binding.llToolBar.visibility = View.GONE
            } else {
                binding.llToolBar.visibility = View.VISIBLE
            }
        }
    }

    // =============================== 初始化 WebView ================================================


    /**
     * 初始化 Web View相关
     */
    private fun initWebView() {

        // 下面两个方法在WebView Provider 设置
        //initWebOptions(getWebView())
        //initSettings(getWebView())

        initWebContentMonitor(getWebView())
        initWebChromeClient(getWebView())
        initWebViewClient(getWebView())
        setDownloadListener(getWebView())
        addJsInterface(mWebView)
        addCallback(getWebView())
        initWebViewFileCacheLoader()
        // 互动广告
        getWebView().addJavascriptInterface(object : OkSpinJsBridge(getWebView()) {
            @JavascriptInterface
            override fun close() {
                super.close()
                // 关闭当前页面
                activity?.finish()
            }
        }, "MbOkSpinJsBridge")

        // 时效性，需要重新load
        getWebView().loadUrl(getUrl())
    }


    /**
     * 初始化 Web View配置
     */
    private fun initWebOptions(webView: BridgeWebView?) {
        webView?.apply {
            setGson(GsonUtils.getGson())
            setLayerType(View.LAYER_TYPE_HARDWARE, null)
            overScrollMode = WebView.OVER_SCROLL_NEVER
        }
    }

    @SuppressLint("SetJavaScriptEnabled")
    private fun initSettings(webView: BridgeWebView?) {
        webView?.settings?.apply {
            javaScriptEnabled = true
            setSupportZoom(true)
            builtInZoomControls = false
            //savePassword = false
            cacheMode = if (NetworkUtil.isNetworkConnected(webView.context)) {
                //根据cache-control获取数据。
                WebSettings.LOAD_DEFAULT
            } else {
                //没网，则从本地获取，即离线加载
                WebSettings.LOAD_CACHE_ELSE_NETWORK
            }
            //适配5.0不允许http和https混合使用情况
            mixedContentMode = WebSettings.MIXED_CONTENT_ALWAYS_ALLOW
            textZoom = 100
            databaseEnabled = true
            loadsImagesAutomatically = true
            setSupportMultipleWindows(false)
            // 是否阻塞加载网络图片  协议http or https
            blockNetworkImage = false
            // 允许加载本地文件html  file协议
            allowFileAccess = true
            // TODO 通过 file url 加载的 Javascript 读取其他的本地文件 .建议关闭
            //allowFileAccessFromFileURLs = false
            // TODO 允许通过 file url 加载的 Javascript 可以访问其他的源，包括其他的文件和 http，https 等其他的源
            //allowUniversalAccessFromFileURLs = false
            javaScriptCanOpenWindowsAutomatically = true
            layoutAlgorithm = WebSettings.LayoutAlgorithm.SINGLE_COLUMN
            loadWithOverviewMode = false
            useWideViewPort = false
            domStorageEnabled = true
            setNeedInitialFocus(true)
            defaultTextEncodingName = "utf-8" //设置编码格式

            defaultFontSize = 16
            minimumFontSize = 12 //设置 WebView 支持的最小字体大小，默认为 8

            setGeolocationEnabled(true)
        }
    }

    /**
     * 初始化 WebChromeClient
     */
    private fun initWebChromeClient(webView: BridgeWebView?) {
        webView?.webChromeClient = object : WebChromeClient() {
            override fun onProgressChanged(view: WebView, newProgress: Int) {
                super.onProgressChanged(view, newProgress)
                onWebChromeClientProgressChanged(view, newProgress)
            }

            override fun onReceivedTitle(view: WebView, title: String) {
                super.onReceivedTitle(view, title)
                strTitle = title
                onWebChromeClientReceivedTitle(view, title)
            }

            override fun onShowCustomView(view: View?, callback: CustomViewCallback?) {
                super.onShowCustomView(view, callback)
                onWebChromeClientShowCustomView(view, callback)
            }

            override fun onHideCustomView() {
                super.onHideCustomView()
                onWebChromeClientHideCustomView()
            }
        }
    }

    /**
     * 初始化 WebViewClient
     */

    private fun initWebViewClient(webView: BridgeWebView?) {

        webView?.webViewClient = object : WebViewClient() {

            override fun onPageStarted(view: WebView, url: String, favicon: Bitmap?) {
                super.onPageStarted(view, url, favicon)
                onWebViewClientPageStarted(view, url, favicon)
            }

            override fun onPageFinished(view: WebView, url: String) {
                super.onPageFinished(view, url)
                onWebViewClientPageFinished(view, url)
            }

            override fun onRenderProcessGone(
                view: WebView?, detail: RenderProcessGoneDetail?,
            ): Boolean {
                // 当 WebView 的渲染进程意外终止时（例如由于内存不足或其他原因），这个方法会被调用。
                //
                // 返回 true 表示你已经处理了渲染进程的消失，WebView 将不会再尝试重启渲染进程。
                // 返回 false 表示你没有处理这个情况，WebView 会尝试重启渲染进程。
                //
                // 在这个方法中，如果你返回 false，那么系统会尝试重新启动渲染进程，通常这是推荐的行为，因为它可以恢复 WebView 的正常工作。
                // 但是，如果你的应用在这种情况下不需要重新加载或重新启动渲染进程，你可以选择返回 true。
                return true
            }

            override fun onReceivedHttpError(
                view: WebView, request: WebResourceRequest, errorResponse: WebResourceResponse,
            ) {
                super.onReceivedHttpError(view, request, errorResponse)
                onWebViewClientReceivedHttpError(view, request, errorResponse)
            }

            override fun onReceivedError(
                view: WebView?, errorCode: Int, description: String?, failingUrl: String?,
            ) {
                super.onReceivedError(view, errorCode, description, failingUrl)
                onWebViewClientReceivedError(view, errorCode, description, failingUrl)
            }

            override fun onReceivedError(
                view: WebView, request: WebResourceRequest, error: WebResourceError,
            ) {
                super.onReceivedError(view, request, error)
                onWebViewClientReceivedError(view, request, error)
            }

            override fun shouldOverrideUrlLoading(view: WebView, url: String): Boolean {
                return if (ParseDeeplinkManager.deepLink(url, context)) {
                    true
                } else {
                    super.shouldOverrideUrlLoading(view, url)
                }
            }

            override fun shouldOverrideUrlLoading(
                view: WebView?, request: WebResourceRequest?,
            ): Boolean {
                val url = request?.url?.toString() ?: ""
                return if (ParseDeeplinkManager.deepLink(url, context)) {
                    true
                } else {
                    super.shouldOverrideUrlLoading(view, request)
                }
            }

            override fun shouldInterceptRequest(
                view: WebView?, url: String?,
            ): WebResourceResponse? {
                return mWebViewLoaderManager?.interceptRequest(url) ?: super.shouldInterceptRequest(
                    view, url
                )
            }

            override fun shouldInterceptRequest(
                view: WebView, request: WebResourceRequest,
            ): WebResourceResponse? {
                return mWebViewLoaderManager?.interceptRequest(request)
                    ?: super.shouldInterceptRequest(
                        view, request
                    )
            }


        }
    }

    /**
     * 下载行为拦截
     */
    private fun setDownloadListener(webView: BridgeWebView?) {
        // 下载监听
        webView?.setDownloadListener { url, userAgent, contentDisposition, mimetype, contentLength ->
            if (loadUrlOnly()) {//只加载url，不做嗅探下载
                return@setDownloadListener
            }

            // 当url以.apk结尾 或者 mimetype = application/vnd.android.package-archive 拦截不下载
            if (url.endsWith(".apk") || url.endsWith(".APK") || TextUtils.equals(
                    mimetype, "application/vnd.android.package-archive"
                )
            ) {
                WebLogger.logW("拦截APK下载 --> mimetype = $mimetype --> url = $url --> contentLength = $contentLength --> contentDisposition = $contentDisposition --> userAgent = $userAgent")
                return@setDownloadListener
            }

            addDownloadWithDialogForWeb(
                requireActivity(), "web_load", url, strTitle, contentLength, getUrl() ?: ""
            )
        }
    }

    /**
     * Activity onBackPressed 处理
     */
    private fun addCallback(webView: BridgeWebView?) {
        requireActivity().onBackPressedDispatcher.addCallback(
            this, object : OnBackPressedCallback(true) {
                override fun handleOnBackPressed() {
                    // 在这里处理返回事件
                    // 例如，显示一个确认对话框或者直接调用 finish()
                    if (webView?.canGoBack() == true) {
                        mWebView.goBack()
                    } else {
                        requireActivity().finish() // 或者其他处理逻辑
                    }
                }
            })
    }

    /**
     * 初始化 静态资源拦截
     */
    private fun initWebViewFileCacheLoader() {
        // 静态资源命中
        context?.let {
            mWebViewLoaderManager = WebViewLoaderManager(
                it,
                //listOf(WebViewFileCacheLoader(context, url), WebViewGlideLoader(context))
                listOf(WebViewFileCacheLoader(it, getUrl()))
            )
        }
    }


    // =============================================================================================


    /**
     * 业务层自行添加
     */
    abstract fun addJsInterface(bridgeWebView: BridgeWebView)


    // =============================================================================================


    /**
     * 下载拦截
     */
    open fun addDownloadWithDialogForWeb(
        requireActivity: FragmentActivity,
        tag: String,
        url: String?,
        title: String?,
        contentLength: Long,
        originalUrl: String,
    ) {
    }

    /**
     * 页面空白，请检查scheme是否加上，可以给默认落地页。
     *
     * @return mUrl
     */
    open fun getUrl(): String {
        return arguments?.getString(WebConstant.FIELD_URL) ?: ""
    }


    // ==================================== WebViewClient ==========================================


    /**
     * WebViewClient 回调
     */
    var firstLoadStartTime = 0L
    open fun onWebViewClientPageStarted(view: WebView, url: String, favicon: Bitmap?) {
        WebLogger.logD("onWebViewClientPageStarted $url")
        currentState = LoadStatus.LOADING
        firstLoadStartTime = System.currentTimeMillis()
        injectPerformanceScript(view)

    }

    open fun onWebViewClientPageFinished(view: WebView, url: String) {
        WebLogger.logD("onWebViewClientPageFinished $url")
        if (currentState == LoadStatus.LOADING) {
            checkFirstLoadStatus(url)
//            injectPerformanceScript()
        }


    }


    open fun onWebViewClientReceivedHttpError(
        view: WebView, request: WebResourceRequest, errorResponse: WebResourceResponse,
    ) {
        if (currentState == LoadStatus.LOADING && request.isForMainFrame()) {
            val statusCode = errorResponse.statusCode
            val reasonPhrase = errorResponse.reasonPhrase
            handleFirstLoadFailure(reasonPhrase, statusCode)
        }
    }

    open fun onWebViewClientReceivedError(
        view: WebView?, request: Int, error: String?, failingUrl: String?,
    ) {
        if (currentState == LoadStatus.LOADING) {
            handleFirstLoadFailure("WebResourceError: $error", -1);
        }
    }


    open fun onWebViewClientReceivedError(
        view: WebView?, request: WebResourceRequest, error: WebResourceError,
    ) {
        if (currentState == LoadStatus.LOADING) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                handleFirstLoadFailure(error.description.toString(), error.errorCode)
            }
        }
    }


    // =================================== WebChromeClient =========================================


    open fun onWebChromeClientReceivedTitle(view: WebView, title: String) {
        binding.tvTitle.text = title
    }

    open fun onWebChromeClientProgressChanged(view: WebView, newProgress: Int) {
        binding.progressBar.visibility = if (newProgress == 100) {
            View.GONE  // 完成加载，隐藏进度条
        } else {
            binding.progressBar.progress = newProgress
            View.VISIBLE // 显示进度条并更新进度
        }
    }

    open fun onWebChromeClientShowCustomView(
        view: View?, callback: WebChromeClient.CustomViewCallback?,
    ) {
        if (webCustomView != null) {
            //Logger.d(TAG, "onShowCustomView 当前存在CustomView，隐藏掉")
            callback?.onCustomViewHidden()
            return
        }
        val activity = activity
        if (activity == null || activity.isDestroyed || activity.isFinishing) return

        webCustomView = view
        originalSystemUiVisibility = activity.window.decorView.systemUiVisibility
        originalOrientation = activity.requestedOrientation
        customViewCallback = callback

        val decor = activity.window.decorView as? FrameLayout
        fullscreenContainer = FrameLayout(activity).apply {
            setBackgroundColor(Color.BLACK)
            addView(
                webCustomView, FrameLayout.LayoutParams(
                    ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT
                )
            )
        }
        decor?.addView(
            fullscreenContainer, FrameLayout.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT
            )
        )
        ImmersionBar.with(activity).hideBar(BarHide.FLAG_HIDE_BAR).init()
        activity.requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_SENSOR_LANDSCAPE
    }

    open fun onWebChromeClientHideCustomView() {
        //Logger.d(TAG, "onHideCustomView，退出全屏页")
        val activity = activity
        if (activity == null || activity.isDestroyed || activity.isFinishing) return
        val decor = activity.window.decorView as FrameLayout
        decor.removeView(fullscreenContainer)
        fullscreenContainer = null
        webCustomView = null
        ImmersionBar.with(activity).hideBar(BarHide.FLAG_SHOW_BAR).init()
        activity.requestedOrientation = originalOrientation
        customViewCallback!!.onCustomViewHidden()
        customViewCallback = null
    }


    // =============================================================================================


    private fun loadUrlOnly(): Boolean {
        return arguments?.getBoolean(WebConstant.FIELD_LOAD_URL_ONLY, false) ?: false
    }


    @SuppressLint("JavascriptInterface")
    private fun initWebContentMonitor(webView: BridgeWebView?) {
        webView?.addJavascriptInterface(LoadTimeInterface(), "AndroidInterface")
    }


    private fun checkFirstLoadStatus(string: String) {
        currentState = LoadStatus.SUCCESS
        val loadTime = System.currentTimeMillis() - firstLoadStartTime

        // 上报首次加载成功
        trackEvent(url = string, loadTime);
    }

    private fun handleFirstLoadFailure(errorMsg: String?, errorCode: Int) {
        currentState = LoadStatus.FAILED
        // 上报首次加载失败
        trackError(url = getUrl(), errorCode, errorMsg)
    }

    val loadInfoStats = LoadInfoStats()
    open fun trackEvent(url: String?, loadTime: Long) {
        WebLogger.logW("trackEvent --> loadTime = $loadTime --> url = $url")
        loadInfoStats.load_time = loadTime
        if (currentState == LoadStatus.SUCCESS) {
            loadInfoStats.isLoadSuccess = true
        }

    }

    open fun trackError(url: String?, errorCode: Int, errorMsg: String?) {
        WebLogger.logW("trackError --> errorCode = $errorCode --> url = $url --> errorMsg = $errorMsg")
        loadInfoStats.error_code = errorCode
        loadInfoStats.error_msg = errorMsg
    }


    /**
     * 注入JavaScript代码获取页面性能指标
     */
    private fun injectPerformanceScript(view1: WebView?) {
        val script = """
              (function() {
                window.addEventListener('load',
                function() {
                    try {
                        const intervalId = setInterval(function() {
                            var timing = window.performance.timing;
                            if (timing.loadEventEnd <= 0) {
                                console.log(timing.loadEventEnd);
                                return;
                            }
            
                            var stats = {
                                // 基本加载时间
                                dnsLookup: timing.domainLookupEnd - timing.domainLookupStart,
                                tcpConnect: timing.connectEnd - timing.connectStart,
                                requestTime: timing.responseStart - timing.requestStart,
                                responseTime: timing.responseEnd - timing.responseStart,
            
                                // DOM相关时间
                                domLoading: timing.domLoading - timing.navigationStart,
                                domInteractive: timing.domInteractive - timing.navigationStart,
                                domComplete: timing.domComplete - timing.navigationStart,
            
                                // 整体加载时间
                                loadEventTime: timing.loadEventEnd - timing.loadEventStart,
                                totalLoadTime: timing.loadEventEnd - timing.navigationStart,
            
                                // 加载状态
                                readyState: document.readyState,
                                success: document.readyState === 'complete'
                            };
                            clearInterval(intervalId)
                            console.log(timing.loadEventEnd)
                            console.warn(JSON.stringify(stats))
                            // 如果有Android接口可用，则通过接口报告
                            if (window.AndroidInterface) {
                                window.AndroidInterface.reportLoadStats(JSON.stringify(stats));
                            } else {
                                console.warn(window.AndroidInterface);
                                console.warn(stats);
                            }
            
                        },
                        1000);
                    } catch(e) {
                        if (window.AndroidInterface) {
                            window.AndroidInterface.reportLoadError(e.message);
                        } else {
                            console.error('Error collecting load stats:', e);
                        }
                    }
                })
            
            })()
            """;

        view1?.evaluateJavascript(script, null)
    }


    /**
     * JavaScript接口类，用于从JS接收加载时间数据
     */
    inner class LoadTimeInterface {
        @JavascriptInterface
        fun reportLoadError(errorMsg: String) {
            WebLogger.logD("reportLoadError: $errorMsg")
            val domPerformance = DomPerformance()
            domPerformance.errorMsg = errorMsg
            loadInfoStats.dom_performance = domPerformance
        }

        @JavascriptInterface
        fun reportLoadStats(json: String) {
            WebLogger.logD("reportLoadStats: $json")
            try {
                val domPerformance = DomPerformance()
                val jsonObject = JSONObject(json)
                domPerformance.dnsLookup = jsonObject.optLong("dnsLookup")
                domPerformance.tcpConnect = jsonObject.optLong("tcpConnect")
                domPerformance.requestTime = jsonObject.optLong("requestTime")
                domPerformance.responseTime = jsonObject.optLong("responseTime")
                domPerformance.domLoading = jsonObject.optLong("domLoading")
                domPerformance.domInteractive = jsonObject.optLong("domInteractive")
                domPerformance.domComplete = jsonObject.optLong("domComplete")
                domPerformance.loadEventTime = jsonObject.optLong("loadEventTime")
                domPerformance.totalLoadTime = jsonObject.optLong("totalLoadTime")
                domPerformance.readyState = jsonObject.optString("readyState")
                domPerformance.success = jsonObject.optBoolean("success")
                loadInfoStats.dom_performance = domPerformance
            } catch (e: Exception) {
                e.printStackTrace()
            }
        }
    }


}




