package com.transsion.lib_web.cache

import android.content.Context
import android.text.TextUtils
import android.view.ViewGroup
import androidx.annotation.MainThread
import com.blankj.utilcode.util.Utils
import com.github.lzyzsd.jsbridge.BridgeWebView

/**
 * @author shmizhangxinbing
 * @date : 2025/10/20 20:16
 * @description: 全局只有一个缓存的WebView
 */
object TWebViewCacheManager {

    /***当前缓存的地址*/
    private var mCurrentH5Url: String? = ""
    private var mCurrentWebView: BridgeWebView? = null


    // ================================== 预加载 ====================================================


    /** 预热单个目标 URL */
    @MainThread
    fun preload(url: String) {
        if (TextUtils.equals(url, mCurrentH5Url)) {
            TWebViewProvider.logCache("preload() --> 正在缓存的地址，无需重复创建 --> url = $url")
            return
        }

        // 标识当前正在缓存的地址
        mCurrentH5Url = url

        // 不存在重新创建
        if (mCurrentWebView == null) {
            mCurrentWebView =
                TWebViewProvider.getWebView(applicationContext = Utils.getApp().applicationContext)
            TWebViewProvider.logCache("preloadInternal() --> 不存在创建新的对象")
        }

        // 加载目标 URL
        mCurrentWebView?.apply {
            setTag(-1001, "${System.currentTimeMillis()}")
            stopLoading()
            clearHistory() // 清除历史记录
            // 清除缓存 不一定会立即释放磁盘空间
            // Android 系统可能会异步删除，短时间内文件夹可能还在。
            //
            // webView.clearCache(true) 会清除 WebView 的 内存缓存和磁盘缓存，参数为 true 会更彻底，
            // 但为了最大化隐私清除，通常要配合清 Cookie、WebStorage、历史记录等操作。
            clearCache(true)
            loadUrl("about:blank")
            loadUrl(url)
            TWebViewProvider.logCache("preloadInternal() --> 创建新对象加入缓存池 --> 预加载埋点 --> url = $url")
            TWebViewProvider.report(url = url, action = "preload")
            setTag(-1001, true)
            //Report.reportEventV2(event = event, map = map, pageName = pageName ?: "")
        }
    }


    // ====================================== 获取缓存 ==============================================


    /**
     * 获取 WebView
     * @param url 要加载的地址
     */
    @MainThread
    fun getWebView(url: String, context: Context?): BridgeWebView {
        return if (isCacheUrl(url)) {
            val valueWeb = mCurrentWebView
            mCurrentWebView = null
            mCurrentH5Url = null
            TWebViewProvider.logCache("get() --> 缓存命中，直接复用 --> url = $url")
            valueWeb
                ?: TWebViewProvider.getWebView(applicationContext = Utils.getApp().applicationContext)
        } else {
            TWebViewProvider.logCache("get() --> 没有缓存，直接创建一个新的 --> url = $url")
            TWebViewProvider.getWebView(
                applicationContext = context ?: Utils.getApp().applicationContext
            )
        }
    }


    // =============================================================================================


    /**
     * 是否是缓存URL
     */
    fun isCacheUrl(url: String): Boolean {
        return TextUtils.equals(url, mCurrentH5Url)
    }

    /**
     * 释放资源
     */
    fun destroy(webView: BridgeWebView?) {
        webView?.let {
            (it.parent as? ViewGroup)?.removeView(it)
            it.stopLoading()
            it.clearHistory()
            it.clearCache(true)
            it.loadUrl("about:blank")
            it.destroy()
        }
    }
}