package com.transsion.lib_web.cache

import android.annotation.SuppressLint
import android.content.Context
import android.util.Log
import android.view.View
import android.webkit.WebResourceRequest
import android.webkit.WebSettings
import android.webkit.WebView
import android.webkit.WebViewClient
import com.blankj.utilcode.util.GsonUtils
import com.github.lzyzsd.jsbridge.BridgeWebView
import com.tn.lib.util.networkinfo.NetworkUtil
import com.transsion.base.report.athena.Report

/**
 * @author shmizhangxinbing
 * @date : 2025/10/20 20:23
 * @description: 提供一个
 */
object TWebViewProvider {

    fun logCache(msg: String) {
        Log.i("zxb_cache_log", msg)
    }

    /**
     * 埋点上报
     */
    fun report(url: String, action: String) {
        val map = mutableMapOf<String, String?>()
        map["url"] = url
        map["action"] = action
        Report.reportEventV2(event = "preload", map = map, pageName = "web")
        logCache("上报 --> action = $action --> url = $url")
    }

    /**
     * 获取WebView
     *
     * 仅创建WebView,不加载URL
     */
    fun getWebView(applicationContext: Context): BridgeWebView {
        val webView = BridgeWebView(applicationContext)
        // 初始化 Web View 相关配置
        initWebOptions(webView)
        initSettings(webView)
        return webView
    }


    // =============================================================================================


    @SuppressLint("SetJavaScriptEnabled")
    private fun initSettings(webView: BridgeWebView) {
        webView.settings.apply {
            javaScriptEnabled = true
            setSupportZoom(true)
            builtInZoomControls = false
            //savePassword = false
            cacheMode = if (NetworkUtil.isNetworkConnected(webView.context)) {
                //根据cache-control获取数据。
                WebSettings.LOAD_DEFAULT
            } else {
                //没网，则从本地获取，即离线加载
                WebSettings.LOAD_CACHE_ELSE_NETWORK
            }
            //适配5.0不允许http和https混合使用情况
            mixedContentMode = WebSettings.MIXED_CONTENT_ALWAYS_ALLOW
            textZoom = 100
            databaseEnabled = true
            loadsImagesAutomatically = true
            setSupportMultipleWindows(false)
            // 是否阻塞加载网络图片  协议http or https
            blockNetworkImage = false
            // 允许加载本地文件html  file协议
            allowFileAccess = true
            // TODO 通过 file url 加载的 Javascript 读取其他的本地文件 .建议关闭
            //allowFileAccessFromFileURLs = false
            // TODO 允许通过 file url 加载的 Javascript 可以访问其他的源，包括其他的文件和 http，https 等其他的源
            //allowUniversalAccessFromFileURLs = false
            javaScriptCanOpenWindowsAutomatically = true
            layoutAlgorithm = WebSettings.LayoutAlgorithm.SINGLE_COLUMN
            loadWithOverviewMode = false
            useWideViewPort = false
            domStorageEnabled = true
            setNeedInitialFocus(true)
            defaultTextEncodingName = "utf-8" //设置编码格式

            defaultFontSize = 16
            minimumFontSize = 12 //设置 WebView 支持的最小字体大小，默认为 8

            setGeolocationEnabled(true)
        }
    }

    /**
     * 初始化 Web View配置
     */
    private fun initWebOptions(webView: BridgeWebView) {
        webView.apply {
            setGson(GsonUtils.getGson())
            setLayerType(View.LAYER_TYPE_HARDWARE, null)
            overScrollMode = WebView.OVER_SCROLL_NEVER
        }

        webView.webViewClient = object : WebViewClient() {

            override fun shouldOverrideUrlLoading(
                view: WebView?,
                request: WebResourceRequest?,
            ): Boolean {
                val urlString = request?.url?.toString() ?: ""
                return if (!urlString.startsWith("http") && !urlString.startsWith("https") && !urlString.startsWith(
                        "file"
                    )
                ) {
                    try {
                        // 页面加载完成，发送事件
                        loadFinish(webView, view?.url)
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }
                    true
                } else {
                    false
                }
            }

            override fun shouldOverrideUrlLoading(view: WebView, url: String): Boolean {
                val urlString = url
                return if (!urlString.startsWith("http") && !urlString.startsWith("https") && !urlString.startsWith(
                        "file"
                    )
                ) {
                    try {
                        // 因为 WebView 在解析跳转时遇到了 外部应用意图（Intent），它其实不会去渲染一个页面，自然也不会调用 onPageFinished。
                        // 页面加载完成，发送事件
                        loadFinish(webView, url)
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }
                    true
                } else {
                    false
                }
            }

            override fun onPageFinished(view: WebView?, url: String?) {
                super.onPageFinished(view, url)
                // 页面加载完成，发送事件
                loadFinish(webView, url)
            }
        }
    }


    private fun loadFinish(webView: BridgeWebView, url: String?) {
        val tag = webView.getTag(-1001) as? Boolean
        if (tag == true) {
            webView.setTag(-1001, false)
            report(url = url ?: "", action = "finish")
        }
    }
}