package com.transsion.lib_web.zip.loader

import android.content.Context
import com.transsion.lib_web.WebLogger
import com.transsion.lib_web.zip.WebViewCacheDownloader
import com.transsion.lib_web.zip.db.WebViewCacheConfigData
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import okhttp3.Request
import okhttp3.Response
import java.io.File
import java.io.FileOutputStream
import java.io.IOException
import java.util.zip.ZipInputStream

/**
 * @author: zhangxinbing
 * @date : 2024/12/11 14:03
 * @description: 下载任务
 */
data class WebViewCacheTask(val context: Context, val dbItem: WebViewCacheConfigData) {

    /**
     * 开始任务
     */
    suspend fun start() {
        withContext(Dispatchers.IO) {
            kotlin.runCatching {
                // 判断文件是否已经存在
                val localFile =
                    WebViewCacheDownloader.getLocalFile(context, dbItem) ?: return@withContext
                if (localFile.exists()) {
                    return@withContext
                }
                // 下载文件并进行MD5校验
                val downloadAndExtractZip = downloadAndExtractZip(dbItem, localFile)
                WebLogger.logD("${getClassTag()} --> checkDownloadZip() --> download success --> file =  $downloadAndExtractZip --> dbItem = $dbItem")
            }.getOrElse {
                WebLogger.logE("${getClassTag()} --> checkDownloadZip() --> it = $it --> dbItem = $dbItem")
            }
        }
    }

    // =============================================================================================


    private fun getClassTag(): String = javaClass.simpleName

    /**
     * 下载并且解压
     */
    private suspend fun downloadAndExtractZip(
        cacheConfigData: WebViewCacheConfigData?, targetDirectory: File
    ): File? {
        val zipUrl = cacheConfigData?.zipUrl
        if (zipUrl.isNullOrEmpty()) {
            return null
        }

        return withContext(Dispatchers.IO) {
            val request = Request.Builder().url(zipUrl).build()
            OkHttpProvider.getOkHttpClient().newCall(request).execute().use { response ->
                if (!response.isSuccessful) {
                    throw IOException("Unexpected code $response")
                }
                // 解压缩并且保存文件
                extractZipToFile(response, targetDirectory)
            }
        }
    }

    private fun extractZipToFile(
        response: Response,
        targetDirectory: File,
    ): File? {
        val inputStream = response.body?.byteStream() ?: return null
        val zipInputStream = ZipInputStream(inputStream)

        // 确保目标目录存在
        if (!targetDirectory.exists()) {
            targetDirectory.mkdirs()
        }

        var entry = zipInputStream.nextEntry
        while (entry != null) {
            val file = File(targetDirectory, entry.name)
            if (entry.isDirectory) {
                file.mkdirs()
            } else {
                file.parentFile?.mkdirs()
                FileOutputStream(file).use { fos ->
                    val buf = ByteArray(4096)
                    var len: Int
                    while (zipInputStream.read(buf).also { len = it } != -1) {
                        fos.write(buf, 0, len)
                    }
                }
            }
            zipInputStream.closeEntry()
            entry = zipInputStream.nextEntry
        }
        zipInputStream.close()
        return targetDirectory // 返回存储路径作为解压后的文件目录
    }
}

