package com.tn.lib.net.dns.or

import android.text.TextUtils
import com.blankj.utilcode.util.GsonUtils
import com.google.gson.reflect.TypeToken
import com.tn.lib.net.Http
import com.tn.lib.net.NetMMKVUtils
import com.tn.lib.net.cons.HeaderConst
import com.tn.lib.net.utils.MkvHeader
import okhttp3.Dns
import java.net.InetAddress
import java.util.concurrent.ConcurrentHashMap


/**
 * 默认pool数据，这里的数据是从服务器获取的，如果后续服务器下发内容有所更新，这里最好随之更新
 */
object CacheIpPool {


    private var mHostList: ArrayList<AddressItem>? = null
    private val mAddressMap = ConcurrentHashMap<String, AddressItem>()

    @Volatile
    private var mCurrentBaseUrl: String = ""


    @Volatile
    private var mHost: String = "api6.aoneroom.com"

    @Volatile
    private var mScheme: String = "https"


    private const val KEY_IP_LIST = "key_ip_list"
    private val defaultRelease by lazy {
        "[\n" +
                "{\n" +
                "\"host\": \"${mHost}\",\n" +
                "\"ip\": \"\",\n" +
                "\"scheme\": \"https\"\n" +
                "}\n" +
                "]"
    }

    private val defaultTest by lazy {
        "[{\n" +
                "    \"host\": \"test-mse-api.aoneroom.com\",\n" +
                "    \"ip\": \"8.219.92.106\",\n" +
                "    \"scheme\": \"https\"\n" +
                "}]"
    }

    const val DEFAULT_ONLINE = "api3.aoneroom.com"
    const val H3_ONLINE_v4 = "api4.aoneroom.com"
    const val H3_ONLINE_v4sg = "api4sg.aoneroom.com"
    const val H3_ONLINE_v5 = "api5.aoneroom.com"
    const val H3_ONLINE_v6 = "api6.aoneroom.com"
    const val H3_ONLINE_v6sg = "api6sg.aoneroom.com"
    const val H3_ONLINE_in = "api.inmoviebox.com"


    const val DEFAULT_DEDUG = "test-mse-api.aoneroom.com"


    private val defaultBaseUrl by lazy {
        if (Http.isDev()) {
            "https://test-mse-api.aoneroom.com"
        } else {
            "https://${mHost}"
        }
    }


    /**
     * 初始化读取可用ip
     */
    fun getCacheUrlList() {
        try {
            val type = object : TypeToken<ArrayList<AddressItem>?>() {}.type
            mHostList = GsonUtils.fromJson<ArrayList<AddressItem>?>(
                NetMMKVUtils.getString(
                    KEY_IP_LIST,
                    if (Http.isDev()) {
                        defaultTest
                    } else {
                        defaultRelease
                    }
                ), type
            )
            mHostList?.forEach { item ->
                if (item.host?.isNotEmpty() == true) {
                    item.ipEnable = true
                    mAddressMap[item.host] = item
                }
            }
            //拼接 baseurl出来
            mCurrentBaseUrl = if (mHostList?.isNotEmpty() == true) {
                mScheme = mHostList?.get(0)?.scheme.toString()
                mHost = mHostList?.get(0)?.host.toString()
                "$mScheme://$mHost"
            } else {
                mCurrentBaseUrl
            }
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }


    /**
     * 更新本地存储
     */

    fun saveConfig(json: String?) {
        if (TextUtils.isEmpty(json).not()) {
            NetMMKVUtils.putString(KEY_IP_LIST, json.toString())
        }
        getCacheUrlList()
    }


    /**
     * 获取可用的ip
     */
    fun getIpAddress(host: String): List<InetAddress> {
        return Dns.SYSTEM.lookup(host)
    }

    /**
     * 获取一个可用host
     */

    fun getBaseUrl(): String {
        if (TextUtils.isEmpty(mCurrentBaseUrl)) {
            mCurrentBaseUrl = defaultBaseUrl
        }
        return mCurrentBaseUrl
    }

    fun getHost(): String {
        val mockHost = MkvHeader.netMmkv?.getString(HeaderConst.MOCK_HOST_KEY, "")
        if (!TextUtils.isEmpty(mockHost)) {
            mHost = mockHost.toString()
        }
        return mHost
    }

    fun getScheme(): String {
        return mScheme
    }

    fun setHost(host: String) {
        mHost = host
    }


}