package com.tn.lib.net.interceptor.signature;

import com.google.gson.Gson;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.google.gson.JsonPrimitive;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

public final class JsonBodyToMapHelper {
    private static JsonBodyToMapHelper sInstance;

    public static JsonBodyToMapHelper getInstance() {
        if (sInstance == null) {
            synchronized (JsonBodyToMapHelper.class) {
                if (sInstance == null) {
                    sInstance = new JsonBodyToMapHelper();
                }
            }
        }

        return sInstance;
    }

    private JsonBodyToMapHelper() {
    }

    /**
     * RequestBody的json格式转成Map<String,String>
     * 1，json字段合并，{"a":1}, {"a":2} 处理后是 "a"=["1","2"]
     * 2，json对象，{"a":1} 处理后是 "a"="1"
     * 3，json嵌套:
     *   {
     *     "type": 100012,
     *     "data": [
     *       {
     *         "item": "first",
     *         "page": 0
     *       },
     *       {
     *         "item": "second",
     *         "page": 1
     *       }
     *     ]
     *   }
     * 结果："data"="[{\"item\":\"first\",\"page\":0},{\"item\":\"second\",\"page\":1}]", "type"="100012"
     * @param json String
     * @return Map<String, String>
     */
    public Map<String, String> toMap(String json) {
        Map<String, String> resultMap = new HashMap<>();
        JsonParser parser = new JsonParser();
        JsonElement jsonElement = null;
        try {
            jsonElement = parser.parse(json);
        } catch (Exception e) {
            e.printStackTrace();
        }
        if(jsonElement!=null) {
            if (jsonElement.isJsonArray()) {
                JsonArray jsonArray = (JsonArray) jsonElement;
                putArrayToMap(jsonArray, resultMap);
            } else if (jsonElement.isJsonObject()) {
                putObjectToMap((JsonObject) jsonElement, resultMap);
            }
        }
        return resultMap;
    }

    private void putArrayToMap(JsonArray jsonArray, Map<String, String> outMap) {
        int size = jsonArray.size();
        if (size == 1) {
            JsonObject jsonObject = jsonArray.get(0).getAsJsonObject();
            putObjectToMap(jsonObject, outMap);
        } else if (size > 1) {
            mergeArrayToMap(jsonArray, outMap);
        }
    }

    private void mergeArrayToMap(JsonArray jsonArray, Map<String, String> outMap) {
        int size = jsonArray.size();
        Map<String, List<String>> mergeMap = new HashMap<>();
        for (int i = 0; i < size; i++) {
            JsonObject jsonObject = jsonArray.get(i).getAsJsonObject();
            Map<String, String> singleObjectMap = jsonToMap(jsonObject);
            Set<String> keySet = singleObjectMap.keySet();
            //合并value到List<String>
            for (String key : keySet) {
                List<String> values = mergeMap.get(key);
                if (values == null) {
                    values = new ArrayList<>(2);
                    mergeMap.put(key, values);
                }
                values.add(singleObjectMap.get(key));
            }
        }
        //Map.Entry<String, List<String>>转成Map.Entry<String, String>
        Set<String> keySet = mergeMap.keySet();
        Gson gson = new Gson();
        for (String key : keySet) {
            List<String> values = mergeMap.get(key);
            outMap.put(key, gson.toJson(values));
        }
    }

    private void putObjectToMap(JsonObject jsonObject, Map<String, String> outMap) {
        outMap.putAll(jsonToMap(jsonObject));
    }

    private Map<String, String> jsonToMap(JsonObject jsonObject) {
        Map<String, String> resultMap = new HashMap<>();

        Set<String> keySet = jsonObject.keySet();
        for (String key : keySet) {
            JsonElement jsonElement = jsonObject.get(key);
            if (jsonElement instanceof JsonPrimitive && ((JsonPrimitive) jsonElement).isString()) {
                resultMap.put(key, jsonElement.getAsString());
            } else {
                resultMap.put(key, jsonElement.toString());
            }
        }
        return resultMap;
    }
}
