package com.tn.lib.net.manager

import android.util.Log
import com.blankj.utilcode.util.Utils
import com.google.net.cronet.okhttptransport.CronetInterceptor
import com.tn.lib.net.Http
import com.tn.lib.net.annotatioProcess.AnnotationDataManager
import com.tn.lib.net.annotatioProcess.RetryAnnoProcess
import com.tn.lib.net.annotation.IRequestApi
import com.tn.lib.net.annotation.InterfaceType
import com.tn.lib.net.annotation.RETRY
import com.tn.lib.net.cons.NetConst
import com.tn.lib.net.cons.NetTag
import com.tn.lib.net.convert.FastJsonConverterFactory
import com.tn.lib.net.dns.or.CacheIpPool
import com.tn.lib.net.env.Host
import com.tn.lib.net.interceptor.BaseUrlInterceptor
import com.tn.lib.net.interceptor.DynamicHostInterceptor
import com.tn.lib.net.interceptor.HttpLogInterceptor
import com.tn.lib.net.interceptor.HttpLoggingInterceptor
import com.tn.lib.net.interceptor.InterceptorsManager
import com.tn.lib.net.interceptor.header.HeaderFactory
import com.tn.lib.net.ssl.HttpsTrustManager
import com.transsion.api.gateway.interceptor.GatewayInterceptor
import io.reactivex.rxjava3.schedulers.Schedulers
import okhttp3.Interceptor
import okhttp3.OkHttpClient
import org.chromium.net.CronetEngine
import retrofit2.Retrofit
import retrofit2.adapter.rxjava3.RxJava3CallAdapterFactory
import retrofit2.http.GET
import retrofit2.http.POST
import java.net.Proxy
import java.util.concurrent.Executors
import java.util.concurrent.TimeUnit
import java.util.concurrent.atomic.AtomicInteger


/**
 * NetServiceGenerator管理类
 */
class NetServiceGenerator {

    private val TAG = "NetServiceGenerator"

    private var okHttpClient: OkHttpClient? = null
    private var retrofit: Retrofit? = null
    private val threadCounter = AtomicInteger(1)

    companion object {

        val instance: NetServiceGenerator by lazy(mode = LazyThreadSafetyMode.SYNCHRONIZED) {
            NetServiceGenerator()
        }

        val HTTP_THREAD_NAME = "one_room_net_thread"

    }

    init {
        CacheIpPool.getCacheUrlList()
        okHttpClient = newOkhttpClient()
        retrofit = Retrofit.Builder()
            .baseUrl(Host.getHost())
            .client(okHttpClient!!)
            .addConverterFactory(FastJsonConverterFactory.create())
            .addCallAdapterFactory(
                RxJava3CallAdapterFactory.createWithScheduler(
                    Schedulers.from(
                        Executors.newFixedThreadPool(NetConst.THREAD_POOL_SIZE_MAX) { r ->
                            Thread(r, "$HTTP_THREAD_NAME-${threadCounter.incrementAndGet()}").apply {
                                priority = Thread.MAX_PRIORITY
                            }
                        }
                    )
                )
            )
            .build()
    }


    private fun newOkhttpClient(): OkHttpClient {
        val logInterceptor = HttpLoggingInterceptor(HttpLogInterceptor()) //创建拦截对象
        logInterceptor.level = HttpLoggingInterceptor.Level.BODY
        val proxy = if (Http.isDebug()) {
            null
        } else {
            Proxy.NO_PROXY
        }
        val builder = OkHttpClient.Builder()
        builder.sslSocketFactory(
            HttpsTrustManager.createSSLSocketFactory(),
            HttpsTrustManager()
        ).hostnameVerifier(HttpsTrustManager.TrustAllHostnameVerifier())//配置ssl以支持抓包
        builder.proxy(proxy).connectTimeout(NetConst.DEFAULT_CONNECT_TIMEOUT, TimeUnit.SECONDS)
            .writeTimeout(NetConst.DEFAULT_WRITE_TIMEOUT, TimeUnit.SECONDS)
            .readTimeout(NetConst.DEFAULT_READ_TIMEOUT, TimeUnit.SECONDS)
            //由于cronet无法配置timeout，如果cronet请求不通会无法超时，这里设置全局超时
            .callTimeout(NetConst.DEFAULT_CONNECT_TIMEOUT, TimeUnit.SECONDS)
            .addInterceptor(HeaderFactory.createHttpHeader())//公共头部
            .addInterceptor(DynamicHostInterceptor())//动态域名
            .addInterceptor(GatewayInterceptor()) //网关签名拦截器
            .addInterceptor(BaseUrlInterceptor()) // 多BaseUrl 问题处理
        //添加外部拦截器
        InterceptorsManager.interceptorsList.forEach { interceptor ->
            builder.addInterceptor(interceptor)
        }
        val instance =
            builder.addInterceptor(logInterceptor) //日志打印
        val cronetInterceptor = getCronetInterceptor()
        if (cronetInterceptor != null) {
            instance.addInterceptor(cronetInterceptor)//cronet拦截器 必须为最后一个
        }
        return instance.build()
    }


    private fun getCronetInterceptor(): Interceptor? {
        try {
            val builder = CronetEngine.Builder(Utils.getApp())
            builder.enableQuic(true)
                .enableBrotli(true)
                .addQuicHint(CacheIpPool.DEFAULT_ONLINE, 443, 443)
                .addQuicHint(CacheIpPool.H3_ONLINE_v4, 443, 443)
                .addQuicHint(CacheIpPool.H3_ONLINE_v4sg, 443, 443)
                .addQuicHint(CacheIpPool.H3_ONLINE_v5, 443, 443)
                .addQuicHint(CacheIpPool.H3_ONLINE_v6, 443, 443)
                .addQuicHint(CacheIpPool.H3_ONLINE_v7, 443, 443)
                .addQuicHint(CacheIpPool.H3_ONLINE_v8, 443, 443)
                .addQuicHint(CacheIpPool.H3_ONLINE_v6sg, 443, 443)
                .addQuicHint(CacheIpPool.H3_ONLINE_in, 443, 443)
                .addQuicHint(CacheIpPool.DEFAULT_DEDUG, 443, 443)
            val cronetEngine: CronetEngine = builder.build()
            return CronetInterceptor.newBuilder(cronetEngine).build()
        } catch (e: Throwable) {
            Log.e("error", "e ${e.message}")
        }
        return null;

    }







    /**
     * 创建Retrofit服务代理对象
     * 使用动态代理设计模式
     */
    fun <T> getService(service: Class<T>): T {
        val api = retrofit?.create(service) ?: generateDefaultRetrofit(service)
        try {
            resetAnnotation()
            if (api is IRequestApi) {
                return agent(api)
            }
        } catch (e: Exception) {
            e.printStackTrace()
            return api
        }
        return api
    }


    fun getHttpClient(): OkHttpClient? {
        return okHttpClient
    }

    private fun <T> agent(newApi: IRequestApi): T {
        val proxy = java.lang.reflect.Proxy.newProxyInstance(
            newApi.javaClass.classLoader,
            newApi.javaClass.interfaces
        ) { proxy, method, args ->
//            Logger.d(TAG, "method:${method.name} args=${args.size}")

            val annotations = method.annotations
            var url = ""
            annotations.forEach {
                url = getMethodUrl(it)
            }

            annotations.forEach {
                parseMethodAnnotation(it, url)
            }
            method.invoke(newApi, *args)
        }
        return proxy as T
    }

    private fun parseMethodAnnotation(annotation: Annotation, url: String) {
        when (annotation) {
            is RETRY -> {
                RetryAnnoProcess.retryMaxNum = annotation.retryCount
                RetryAnnoProcess.retryTimestamp = annotation.timestamp
            }
            is InterfaceType -> {
                AnnotationDataManager.setType(url, annotation.type)
            }
        }
    }

    private fun getMethodUrl(annotation: Annotation): String {
        var url = ""
        if (annotation is GET) {
            url = annotation.value
        } else if (annotation is POST) {
            url = annotation.value
        }
        return url
    }

    private fun resetAnnotation() {
        RetryAnnoProcess.retryMaxNum = 0
        RetryAnnoProcess.retryTimestamp = 0
    }

    /**
     * 创建默认值，避免空指针
     */
    private fun <T> generateDefaultRetrofit(service: Class<T>): T {
//        Logger.d(TAG, "--- generateDefaultRetrofit ---")
        return Retrofit.Builder().build().create(service)
    }


    /**
     * okHttpclide for glide
     * @return OkHttpClient
     */
    var okHttpClientForGlide: OkHttpClient? = null

    fun newGlideOkhttpClient(): OkHttpClient? {
        if (okHttpClientForGlide != null) {
            return okHttpClientForGlide
        }
        val loginInterceptor = HttpLogInterceptor()
        loginInterceptor.replaceTag(NetTag.LOG_IMG_TAG)
        val logInterceptor = HttpLoggingInterceptor(loginInterceptor) //创建拦截对象
        logInterceptor.level = HttpLoggingInterceptor.Level.BASIC
        val builder = OkHttpClient.Builder()
        //避免时间错误带来的ssl异常
        builder.sslSocketFactory(
            HttpsTrustManager.createSSLSocketFactory(),
            HttpsTrustManager()
        ).hostnameVerifier(HttpsTrustManager.TrustAllHostnameVerifier())//配置ssl以支持抓包
        val instance =
            builder.connectTimeout(NetConst.DEFAULT_CONNECT_TIMEOUT, TimeUnit.SECONDS)
                .writeTimeout(NetConst.DEFAULT_WRITE_TIMEOUT, TimeUnit.SECONDS)
                .readTimeout(NetConst.DEFAULT_READ_TIMEOUT, TimeUnit.SECONDS)
                .addInterceptor(logInterceptor) //日志打印
        val cronetInterceptor = getCronetInterceptorForGlide()
        if (cronetInterceptor != null) {
            instance.addInterceptor(cronetInterceptor)//cronet拦截器 必须为最后一个
        }
        okHttpClientForGlide = instance.build()
        return okHttpClientForGlide
    }

    private fun getCronetInterceptorForGlide(): Interceptor? {
        try {
            val builder = CronetEngine.Builder(Utils.getApp())
            builder.enableQuic(true)
                .enableBrotli(true)
                .addQuicHint("pacdn.aoneroom.com", 443, 443)
            val cronetEngine: CronetEngine = builder.build()
            return CronetInterceptor.newBuilder(cronetEngine).build()
        } catch (e: Throwable) {
            Log.e("error", "e ${e.message}")
        }
        return null;

    }


}