package com.transsion.player.control

import android.content.Context
import android.util.AttributeSet
import android.view.ViewGroup
import android.widget.FrameLayout
import androidx.core.view.isVisible
import com.transsion.player.enum.PipStatus
import com.transsion.player.ui.IORVideoView

/**
 *
 * Created by kun.xu on 2024/11/13 20:53
 * Description:
 * 视频控制层
 * 外部添加各自控制层view：播放按钮，进度条，错误提示，完成页面，等页面所以控制层
 */
abstract class BaseVideoController<Data, PlayStatus> : FrameLayout, IVideoController<Data, PlayStatus> {

    private var controlWrapper: ControlWrapper<Data, PlayStatus>? = null

    //保存了所有的控制组件
    protected var controlComponents = LinkedHashMap<IControlComponent<Data, PlayStatus>, Boolean>()

    //进入画中画之前展示的comp
    protected val pipStatusVisibilityComponents = mutableListOf<IControlComponent<Data, PlayStatus>>()

    protected var videoContentData: Data? = null

    constructor(context: Context) : this(context, null)
    constructor(context: Context, attrs: AttributeSet?) : this(context, attrs, 0)
    constructor(context: Context, attrs: AttributeSet?, defStyleAttr: Int) : super(
        context,
        attrs,
        defStyleAttr
    )

    fun setPayerView(playerView: IORVideoView) {
        controlWrapper = ControlWrapper(playerView, this)
        //绑定ControlComponent和Controller
        for ((component) in controlComponents.entries) {
            component.attach(controlWrapper!!)
        }
    }

    fun setVideoData(videoData: Data) {
        this.videoContentData = videoData
        for ((component) in controlComponents.entries) {
            component.setVideoData(videoData)
        }
    }

    /**
     * 添加控制组件，最后面添加的在最下面，合理组织添加顺序，可让ControlComponent位于不同的层级
     */
    fun addControlComponent(vararg component: IControlComponent<Data, PlayStatus>) {
        for (item in component) {
            addControlComponent(item, false)
        }
    }

    /**
     * 添加控制组件，可让ControlComponent位于不同的层级
     * todo 重点：添加顺序要注意，就是addView，最先添加的放在view最下面。别错乱了导致业务问题
     *
     * @param isDissociate 是否为游离的控制组件, true:可以不是view，只处理逻辑，不处理控制view
     */
    fun addControlComponent(component: IControlComponent<Data, PlayStatus>, isDissociate: Boolean) {
        controlComponents[component] = isDissociate
        if (controlWrapper != null) {
            component.attach(controlWrapper!!)
        }
        val view = component.getView()
        if (view != null && !isDissociate) {
            addView(view, LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT))
        }
    }

    /**
     * 移除某个控制组件
     */
    fun removeControlComponent(component: IControlComponent<Data, PlayStatus>) {
        removeView(component.getView())
        controlComponents.remove(component)
    }

    /**
     * 移除所有控制组件
     */
    fun removeAllControlComponent() {
        for ((key) in controlComponents.entries) {
            removeView(key.getView())
        }
        controlComponents.clear()
    }

    /**
     * 移除所有的游离控制组件
     * 关于游离控制组件的定义请看 [.addControlComponent] 关于 isDissociate 的解释
     */
    fun removeAllDissociateComponents() {
        val it: MutableIterator<Map.Entry<IControlComponent<Data, PlayStatus>, Boolean>> = controlComponents.entries.iterator()
        while (it.hasNext()) {
            val next = it.next()
            if (next.value) {
                it.remove()
            }
        }
    }

    override fun getControlComponentMap(): LinkedHashMap<IControlComponent<Data, PlayStatus>, Boolean> {
        return controlComponents
    }


    fun onPipStatusChange(pipStatus: PipStatus) {
        if (pipStatus == PipStatus.DEFAULT) {
            pipStatusVisibilityComponents.forEach {
                it.getView()?.isVisible = true
            }
        } else {
            pipStatusVisibilityComponents.clear()
            for ((component) in controlComponents.entries) {
                val view = component.getView()
                if (view?.isVisible == true) {
                    view.isVisible = false
                    pipStatusVisibilityComponents.add(component)
                }
            }
        }

    }

}