package com.transsion.player.mediasession

import android.os.Bundle
import android.support.v4.media.MediaMetadataCompat
import android.support.v4.media.session.MediaSessionCompat
import android.support.v4.media.session.PlaybackStateCompat
import com.transsion.player.orplayer.global.TnPlayerManager


/**
 * @author xinbing.zhang
 * @date :2023/9/28 16:28
 * @description: 控制器操作回调的地方 -- 音视频都会走到这里来
 */
class MediaMediaSessionCompatCallBack(
    private val mediaSession: MediaSessionCompat?, private val mediaService: MediaService
) : MediaSessionCompat.Callback() {


    companion object {
        const val KEY_MEDIAITEM = "MediaItem"
        const val ACTION_PLAY = "transsion_play"
        const val ACTION_PAUSE = "transsion_pause"
    }


    private fun getTagClass(): String = javaClass.simpleName

    private var mediaItem: MediaItem? = null


    // =============================== 通知栏事件回调 =================================================


    override fun onPlay() {
        super.onPlay()
        SessionLogger.log("${getTagClass()} --> mediaSessionCallback --> onPlay()")
        // 设置当前的播放状态 --> 发送通知
        changePlayState(setPlayState(PlaybackStateCompat.STATE_PLAYING))
        // 调用播放器的播放能力
        TnPlayerManager.currentFocusPlayer?.play()
    }

    override fun onPause() {
        super.onPause()
        SessionLogger.log("${getTagClass()} --> mediaSessionCallback --> onPause()")

        // 暂停 --> 通知
        changePlayState(setPlayState(PlaybackStateCompat.STATE_PAUSED))

        // 调用播放器的暂停能力
        TnPlayerManager.currentFocusPlayer?.pause()
    }

    override fun onStop() {
        super.onStop()
        SessionLogger.log("${getTagClass()} --> mediaSessionCallback --> onStop()")
        setPlayState(PlaybackStateCompat.STATE_STOPPED)
        TnPlayerManager.currentFocusPlayer?.release()
    }

    override fun onSkipToNext() {
        super.onSkipToNext()
        SessionLogger.log("${getTagClass()} --> mediaSessionCallback --> onSkipToNext() -- 暂不处理该事件")
        // 暂不处理该事件 setPlayState(PlaybackStateCompat.STATE_SKIPPING_TO_NEXT)
    }

    override fun onSkipToPrevious() {
        super.onSkipToPrevious()
        SessionLogger.log("${getTagClass()} --> mediaSessionCallback --> onSkipToPrevious() -- 暂不处理该事件")
        // 暂不处理该事件 setPlayState(PlaybackStateCompat.STATE_SKIPPING_TO_PREVIOUS)
    }

    override fun onSeekTo(pos: Long) {
        super.onSeekTo(pos)
        mediaItem?.position = pos
        SessionLogger.log("${getTagClass()} --> mediaSessionCallback --> onSeekTo() --> mediaItem = $mediaItem")
        //setPlayState(PlaybackStateCompat.STATE_PLAYING)
        // 调用播放器的修改进度能力
        TnPlayerManager.currentFocusPlayer?.seekTo(pos)
    }


    // =============================================================================================


    override fun onCustomAction(action: String?, extras: Bundle?) {
        super.onCustomAction(action, extras)
        mediaItem = extras?.getParcelable(KEY_MEDIAITEM)

        //SessionLogger.log("${getTagClass()} --> onCustomAction --> action = $action --> mediaItem = $mediaItem")
        val playbackStateCompat = when (action) {
            ACTION_PLAY -> {
                // 播放信息设置
                mediaSession?.setMetadata(
                    MediaMetadataCompat.Builder().putString(
                        MediaMetadataCompat.METADATA_KEY_MEDIA_ID, mediaItem?.mediaId
                    ).putLong(
                        MediaMetadataCompat.METADATA_KEY_DURATION, mediaItem?.duration ?: 0
                    ).build()
                )
                setPlayState(PlaybackStateCompat.STATE_PLAYING)
            }

            ACTION_PAUSE -> {
                setPlayState(PlaybackStateCompat.STATE_PAUSED)
            }

            else -> {
                null
            }
        }

        changePlayState(playbackStateCompat)
    }


    // =============================================================================================


    /**
     * 设置 播放器当前的状态
     *
     * 这里要设置真实的进度 .setState(state, mediaPlayer.getCurrentPosition(), 1.0f)
     *
     * @param state state
     */
    private fun setPlayState(@PlaybackStateCompat.State state: Int): PlaybackStateCompat {
        val position = mediaItem?.position ?: 0
        val playbackState = PlaybackStateCompat.Builder().setActions(
            MediaService.PLAY_BACK_STATE_COMPAT_ACTIONS
        ).setState(state, position, 1.0f).build()
        mediaSession?.setPlaybackState(playbackState)
        return playbackState
    }

    /**
     * 修改播放状态
     */
    private fun changePlayState(playbackStateCompat: PlaybackStateCompat?) {
        playbackStateCompat?.let {
            MediaNotification.notifyNotification(
                mediaService, mediaItem, playbackStateCompat.state, mediaSession
            )
        }
    }

}