package com.transsion.player.orplayer.global

import android.annotation.SuppressLint
import com.transsion.player.MediaSource
import com.transsion.player.mediasession.MediaBrowserCompatHelper
import com.transsion.player.mediasession.SessionLogger
import com.transsion.player.orplayer.IPlayerListener
import com.transsion.player.orplayer.PlayError
import com.transsion.player.tracks.TnTracks
import java.text.SimpleDateFormat
import java.util.TimeZone
import java.util.concurrent.CopyOnWriteArrayList

/**
 * @author: zhangxinbing
 * @date : 2024/10/24 10:47
 * @description: 播放器事件拦截 + 播放器状态回调
 *
 * 单例会导致当前多个播放器回调错乱
 */
class PlayerListenerIntercept : IPlayerListener {

    private var listeners: CopyOnWriteArrayList<IPlayerListener> = CopyOnWriteArrayList()

    private fun getCLassTag(): String = javaClass.simpleName

    fun addPlayerListener(listener: IPlayerListener) {
        if (listeners.contains(listener).not()) {
            listeners.add(listener)
        }
    }

    fun removePlayerListener(listener: IPlayerListener) {
        listeners.remove(listener)
    }


    // =============================================================================================


    override fun onSetDataSource() {
        super.onSetDataSource()
        listeners.forEach {
            it.onSetDataSource()
        }
    }

    override fun initPlayer() {
        super.initPlayer()
        listeners.forEach {
            it.initPlayer()
        }
    }

    override fun onRenderFirstFrame() {
        super.onRenderFirstFrame()
        listeners.forEach {
            it.onRenderFirstFrame()
        }
    }

    override fun onPlayerRelease(mediaSource: MediaSource?) {
        super.onPlayerRelease(mediaSource)
        listeners.forEach {
            it.onPlayerRelease(mediaSource)
        }
    }

    override fun onPrepare(mediaSource: MediaSource?) {
        super.onPrepare(mediaSource)
        listeners.forEach {
            it.onPrepare(mediaSource)
        }
    }

    override fun onLoopingStart() {
        super.onLoopingStart()
        listeners.forEach {
            it.onLoopingStart()
        }
    }

    override fun onPlayError(errorInfo: PlayError, mediaSource: MediaSource?) {
        super.onPlayError(errorInfo, mediaSource)
        listeners.forEach {
            it.onPlayError(errorInfo, mediaSource)
        }
    }

    override fun onVideoSizeChanged(width: Int, height: Int) {
        super.onVideoSizeChanged(width, height)
        listeners.forEach {
            it.onVideoSizeChanged(width, height)
        }
    }

    override fun onLoadingBegin(mediaSource: MediaSource?) {
        super.onLoadingBegin(mediaSource)
        listeners.forEach {
            it.onLoadingBegin(mediaSource)
        }
    }

    override fun onBufferedPosition(progress: Long, mediaSource: MediaSource?) {
        super.onBufferedPosition(progress, mediaSource)
        listeners.forEach {
            it.onBufferedPosition(progress, mediaSource)
        }
    }

    override fun onLoadingProgress(percent: Int, netSpeed: Float, mediaSource: MediaSource?) {
        super.onLoadingProgress(percent, netSpeed, mediaSource)
        listeners.forEach {
            it.onLoadingProgress(percent, netSpeed, mediaSource)
        }
    }

    override fun onLoadingEnd(mediaSource: MediaSource?) {
        super.onLoadingEnd(mediaSource)
        listeners.forEach {
            it.onLoadingEnd(mediaSource)
        }
    }

    override fun onPlayerReset() {
        super.onPlayerReset()
        listeners.forEach {
            it.onPlayerReset()
        }
    }

    override fun onIsPlayingChanged(isPlaying: Boolean) {
        super.onIsPlayingChanged(isPlaying)
        listeners.forEach {
            it.onIsPlayingChanged(isPlaying)
        }
    }

    override fun onFocusChange(focusLoss: Boolean) {
        super.onFocusChange(focusLoss)
        listeners.forEach {
            it.onFocusChange(focusLoss)
        }
    }

    override fun onMediaItemTransition(uuid: String?) {
        super.onMediaItemTransition(uuid)
        listeners.forEach {
            it.onMediaItemTransition(uuid)
        }
    }

    override fun onTracksChange(tracks: TnTracks) {
        super.onTracksChange(tracks)
        listeners.forEach {
            it.onTracksChange(tracks)
        }
    }

    override fun onTracksVideoBitrateChange(videoBitrate: Int) {
        super.onTracksVideoBitrateChange(videoBitrate)
        listeners.forEach {
            it.onTracksVideoBitrateChange(videoBitrate)
        }
    }

    override fun onTracksAudioBitrateChange(audioBitrate: Int) {
        super.onTracksAudioBitrateChange(audioBitrate)
        listeners.forEach {
            it.onTracksAudioBitrateChange(audioBitrate)
        }
    }

    override fun onPlayErrorChangePayer(errorType: TnPlayerType, mediaSource: MediaSource?) {
        super.onPlayErrorChangePayer(errorType, mediaSource)
        listeners.forEach {
            it.onPlayErrorChangePayer(errorType, mediaSource)
        }
    }

    override fun onAliyunDecodeErrorChangeSoftwareDecoder(mediaSource: MediaSource?) {
        super.onAliyunDecodeErrorChangeSoftwareDecoder(mediaSource)
        listeners.forEach {
            it.onAliyunDecodeErrorChangeSoftwareDecoder(mediaSource)
        }
    }


    // =============================================================================================


    override fun onVideoStart(mediaSource: MediaSource?) {
        super.onVideoStart(mediaSource)
        listeners.forEach {
            it.onVideoStart(mediaSource)
        }
        SessionLogger.log(
            "${getCLassTag()} --> onVideoStart() --> 刷新通知栏 --> position = ${
                millisecondToTimeWithHour(
                    mediaSource?.mediaItem?.position ?: 0
                )
            } --> position = ${mediaSource?.mediaItem?.position}"
        )
        mediaSource?.mediaItem?.duration = TnPlayerManager.currentFocusPlayer?.getDuration()
        mediaSource?.mediaItem?.position = TnPlayerManager.currentFocusPlayer?.getCurrentPosition()
        MediaBrowserCompatHelper.instance.play(mediaSource?.mediaItem)
    }

    override fun onProgress(progress: Long, mediaSource: MediaSource?) {
        super.onProgress(progress, mediaSource)
        listeners.forEach {
            it.onProgress(progress, mediaSource)
        }

        if (System.currentTimeMillis() - lastTimestamp > DEFAULT_GAP_TIME) {
            lastTimestamp = System.currentTimeMillis()
//            SessionLogger.log(
//                "${getCLassTag()} --> onProgress() --> 刷新通知栏 --> position = ${
//                    millisecondToTimeWithHour(
//                        mediaSource?.mediaItem?.position ?: 0
//                    )
//                } --> position = ${mediaSource?.mediaItem?.position}"
//            )
            mediaSource?.mediaItem?.duration = TnPlayerManager.currentFocusPlayer?.getDuration()
            mediaSource?.mediaItem?.position = progress
            MediaBrowserCompatHelper.instance.play(mediaSource?.mediaItem)
        }
    }

    override fun onVideoPause(mediaSource: MediaSource?) {
        super.onVideoPause(mediaSource)
        listeners.forEach {
            it.onVideoPause(mediaSource)
        }
        SessionLogger.log(
            "${getCLassTag()} --> onVideoPause() --> 刷新通知栏 --> position = ${
                millisecondToTimeWithHour(
                    mediaSource?.mediaItem?.position ?: 0
                )
            } --> position = ${mediaSource?.mediaItem?.position}"
        )
        mediaSource?.mediaItem?.duration = TnPlayerManager.currentFocusPlayer?.getDuration()
        mediaSource?.mediaItem?.position = TnPlayerManager.currentFocusPlayer?.getCurrentPosition()
        MediaBrowserCompatHelper.instance.pause(mediaSource?.mediaItem)
    }

    override fun setOnSeekCompleteListener() {
        super.setOnSeekCompleteListener()
        listeners.forEach {
            it.setOnSeekCompleteListener()
        }
        val mediaItem = TnPlayerManager.currentFocusPlayer?.currentMediaSource()?.mediaItem
        mediaItem?.duration = TnPlayerManager.currentFocusPlayer?.getDuration()
        mediaItem?.position = TnPlayerManager.currentFocusPlayer?.getCurrentPosition()
        SessionLogger.log(
            "${getCLassTag()} --> setOnSeekCompleteListener() --> 刷新通知栏 --> position = ${
                millisecondToTimeWithHour(
                    mediaItem?.position ?: 0
                )
            } --> position = ${mediaItem?.position}"
        )
        MediaBrowserCompatHelper.instance.play(mediaItem)
    }

    override fun onCompletion(mediaSource: MediaSource?) {
        super.onCompletion(mediaSource)
        listeners.forEach {
            it.onCompletion(mediaSource)
        }
        SessionLogger.log("${getCLassTag()} --> onCompletion() --> 刷新通知栏")
        MediaBrowserCompatHelper.instance.pause(mediaSource?.mediaItem)
    }


    // =============================================================================================


    /**
     * 控制通知栏刷新频率
     */
    private var lastTimestamp = 0L

    private val DEFAULT_GAP_TIME = 3000

    /**
     * 时间戳转换
     */
    @SuppressLint("SimpleDateFormat")
    private fun millisecondToTimeWithHour(millisecond: Long): String {
        val format = if (millisecond >= 3600_000) {
            SimpleDateFormat("H:mm:ss")
        } else {
            SimpleDateFormat("mm:ss")
        }
        format.timeZone = TimeZone.getTimeZone("GMT+0:00")
        return format.format(millisecond)
    }
}