package com.transsion.player.exo

import android.content.Context
import android.os.Build
import android.os.Handler
import android.os.Looper
import android.util.Log
import android.view.SurfaceView
import android.view.TextureView
import androidx.media3.common.AudioAttributes
import androidx.media3.common.C
import androidx.media3.common.DeviceInfo
import androidx.media3.common.MediaItem
import androidx.media3.common.MediaMetadata
import androidx.media3.common.Metadata
import androidx.media3.common.MimeTypes
import androidx.media3.common.PlaybackException
import androidx.media3.common.PlaybackParameters
import androidx.media3.common.Player
import androidx.media3.common.Player.DiscontinuityReason
import androidx.media3.common.Player.PlayWhenReadyChangeReason
import androidx.media3.common.Player.PlaybackSuppressionReason
import androidx.media3.common.Player.TimelineChangeReason
import androidx.media3.common.Timeline
import androidx.media3.common.TrackSelectionOverride
import androidx.media3.common.TrackSelectionParameters
import androidx.media3.common.Tracks
import androidx.media3.common.VideoSize
import androidx.media3.common.text.Cue
import androidx.media3.common.text.CueGroup
import androidx.media3.common.util.UnstableApi
import androidx.media3.datasource.DefaultDataSource
import androidx.media3.datasource.DefaultHttpDataSource
import androidx.media3.datasource.HttpDataSource
import androidx.media3.datasource.rtmp.RtmpDataSource
import androidx.media3.exoplayer.DefaultLivePlaybackSpeedControl
import androidx.media3.exoplayer.ExoPlaybackException
import androidx.media3.exoplayer.ExoPlayer
import androidx.media3.exoplayer.RenderersFactory
import androidx.media3.exoplayer.analytics.AnalyticsListener
import androidx.media3.exoplayer.dash.DashMediaSource
import androidx.media3.exoplayer.drm.DrmSessionManager
import androidx.media3.exoplayer.hls.HlsMediaSource
import androidx.media3.exoplayer.source.DefaultMediaSourceFactory
import androidx.media3.exoplayer.source.ProgressiveMediaSource
import androidx.media3.exoplayer.util.EventLogger
import com.transsion.player.MediaSource
import com.transsion.player.enum.PlayMimeType
import com.transsion.player.enum.ScaleMode
import com.transsion.player.orplayer.AudioFocusHelper
import com.transsion.player.orplayer.AudioFocusPlay
import com.transsion.player.orplayer.IPlayerListener
import com.transsion.player.orplayer.ORPlayer
import com.transsion.player.orplayer.PlayError
import com.transsion.player.orplayer.global.TnErrorInterceptorListener
import com.transsion.player.orplayer.global.TnPlayerPool
import com.transsion.player.orplayer.global.TnPlayerType
import com.transsion.player.tracks.TnFormat
import com.transsion.player.tracks.TnTracks
import com.transsion.player.tracks.TnTracksGroup
import com.transsion.player.tracks.findTrackGroup
import com.transsion.player.tracks.toTnFormat
import com.transsion.player.tracks.toTnTracks
import com.transsion.player.ui.render.RenderScaleMode
import com.transsion.player.ui.render.SurfaceRenderView
import com.transsion.player.ui.render.TextureRenderView
import com.transsion.player.utils.ORPlayerLog
import java.util.concurrent.CopyOnWriteArrayList
import kotlin.math.abs

@UnstableApi
class ORExoPlayer(
    val context: Context,
    val config: VideoConfig = DefaultConfig(),
    openAudioFocus: Boolean = true
) : ORPlayer {

    private var isPrepare = false
    private var isRenderFirstFrame = false
    private var surfaceView: SurfaceRenderView? = null
    private var textureView: TextureRenderView? = null
    private var audioFocusHelper: AudioFocusHelper? = null

    private var mediaSource: MediaSource? = null
    private var volume: Float? = null
    private var isAutoPlay: Boolean = false
    private var isLoop: Boolean = false
    private var mute: Boolean = false
    private var speed: Float = 1f

    private var videoBitrate = 0
    private var audioBitrate = 0

    private var isCompletion = false//防止Completion多次回调

    private var errorInterceptorListener: TnErrorInterceptorListener? = null

    private var exoPlayer: ExoPlayer? = null

    /** 当前视频的编码类型,参考：[MimeTypes]*/
    private var videoMimeType: String? = null

    /** 当前播放器使用的解码器*/
    private var curDecoderType = ORExoDecoderType.HARDWARE

    companion object {
        private const val TAG = "ORExoPlayer"
    }

    private fun initAudioFocus() {
        val audioFocusPlayer = object : AudioFocusPlay {
            override fun play() {
                exoPlayer?.play()
            }

            override fun pause() {
                exoPlayer?.pause()
            }

            override fun isMute(): Boolean {
                return exoPlayer?.isDeviceMuted ?: false
            }

            override fun setVolume(volume: Float) {
                exoPlayer?.volume = volume
            }
        }
        audioFocusHelper = AudioFocusHelper(audioFocusPlayer) { focusLoss ->
            listeners.forEach { listener ->
                listener.onFocusChange(focusLoss)
            }
        }
    }

    private val exoPlayerListener: Player.Listener = object : Player.Listener {

        override fun onTimelineChanged(timeline: Timeline, reason: @TimelineChangeReason Int) {}
        override fun onMediaMetadataChanged(mediaMetadata: MediaMetadata) {}
        override fun onPlaylistMetadataChanged(mediaMetadata: MediaMetadata) {}
        override fun onLoadingChanged(isLoading: Boolean) {}
        override fun onAvailableCommandsChanged(availableCommands: Player.Commands) {}
        override fun onTrackSelectionParametersChanged(parameters: TrackSelectionParameters) {}
        override fun onPlayerStateChanged(playWhenReady: Boolean, playbackState: @Player.State Int) {}
        override fun onPlayWhenReadyChanged(playWhenReady: Boolean, reason: @PlayWhenReadyChangeReason Int) {}
        override fun onPlaybackSuppressionReasonChanged(playbackSuppressionReason: @PlaybackSuppressionReason Int) {}
        override fun onRepeatModeChanged(repeatMode: @Player.RepeatMode Int) {}
        override fun onShuffleModeEnabledChanged(shuffleModeEnabled: Boolean) {}
        override fun onPlayerErrorChanged(error: PlaybackException?) {}
        override fun onPositionDiscontinuity(reason: @DiscontinuityReason Int) {}
        override fun onPlaybackParametersChanged(playbackParameters: PlaybackParameters) {}
        override fun onSeekBackIncrementChanged(seekBackIncrementMs: Long) {}
        override fun onSeekForwardIncrementChanged(seekForwardIncrementMs: Long) {}
        override fun onMaxSeekToPreviousPositionChanged(maxSeekToPreviousPositionMs: Long) {}
        override fun onAudioSessionIdChanged(audioSessionId: Int) {}
        override fun onAudioAttributesChanged(audioAttributes: AudioAttributes) {}
        override fun onVolumeChanged(volume: Float) {}
        override fun onDeviceInfoChanged(deviceInfo: DeviceInfo) {}
        override fun onDeviceVolumeChanged(volume: Int, muted: Boolean) {}
        override fun onSurfaceSizeChanged(width: Int, height: Int) {}
        override fun onCues(cues: MutableList<Cue>) {}
        override fun onCues(cueGroup: CueGroup) {}
        override fun onMetadata(metadata: Metadata) {}

        private var lastState = Player.STATE_IDLE
        override fun onPlaybackStateChanged(playbackState: Int) {
            Log.e(TAG, "onPlaybackStateChanged  playbackState:$playbackState ")
            when (playbackState) {
                ExoPlayer.STATE_READY -> {
                    if (!isPrepare) {
                        isPrepare = true
                        if (errorInterceptorListener?.onPrepare(mediaSource) == true) {
                            Log.i(TAG, "exo onPrepare 被外部拦截掉")
                        } else {
                            Log.i(TAG, "onPlaybackStateChanged onPrepare")
                            listeners.forEach { listener ->
                                listener?.onPrepare(mediaSource)
                            }
                        }
                        onPrepare2GetBitrate()
                    }
                    if (lastState == Player.STATE_BUFFERING) {
                        Log.i(TAG, "onPlaybackStateChanged onLoadingEnd")
                        listeners.forEach { listener ->
                            listener?.onLoadingEnd()
                        }
                    }
                }

                ExoPlayer.STATE_ENDED -> {
                    Log.i(TAG, "onPlaybackStateChanged STATE_ENDED")
                    if (isCompletion.not()) {
                        isCompletion = true
                        listeners.forEach { listener ->
                            listener?.onCompletion(mediaSource)
                        }
                    }
                }

                Player.STATE_BUFFERING -> {
                    Log.e(TAG, "onPlaybackStateChanged STATE_BUFFERING")
                    listeners.forEach { listener ->
                        listener?.onLoadingBegin(mediaSource)
                    }
                }

                Player.STATE_IDLE -> {
                }
            }
            lastState = playbackState
        }

        override fun onIsLoadingChanged(isLoading: Boolean) {
        }

        override fun onIsPlayingChanged(isPlaying: Boolean) {
            audioFocusHelper?.isPlaying = isPlaying
            listeners.forEach { listener ->
                listener?.onIsPlayingChanged(isPlaying)
            }
            if (isPlaying) {
                listeners.forEach { listener ->
                    listener?.onVideoStart(mediaSource)
                }
                handler.post(timerTask)
            } else {
                if (exoPlayer?.playWhenReady != true) {
                    listeners.forEach { listener ->
                        listener?.onVideoPause(mediaSource)
                    }
                }
                handler.removeCallbacks(timerTask)
                if (abs((exoPlayer?.duration ?: 0) - (exoPlayer?.currentPosition ?: 0)) < 2000) {
                    Log.e(TAG, "onIsPlayingChanged onCompletion")
                    if (isCompletion.not()) {
                        isCompletion = true
                        listeners.forEach { listener ->
                            listener?.onCompletion(mediaSource)
                        }
                    }
                }
            }
            Log.e(TAG, "onIsPlayingChanged:$isPlaying")
        }

        override fun onRenderedFirstFrame() {
            if (!isRenderFirstFrame) {
                isRenderFirstFrame = true
                Log.e(TAG, "onRenderedFirstFrame:")
                listeners.forEach { listener ->
                    listener?.onRenderFirstFrame()
                }
            }
        }

        override fun onPlayerError(error: PlaybackException) {
            isPrepare = false
            if (error is ExoPlaybackException) {
                error.cause?.let {
                    if (it is HttpDataSource.InvalidResponseCodeException && it.responseCode == 404) {
                        val totalDuration = getDuration()
                        val progress = getCurrentPosition()
                        val isEndProgress = progress > 0 && totalDuration > 0 && (progress * 1f / totalDuration > 0.99f)
                        if (isEndProgress) {//404 缺少分片的，增加进度判断，否则有误判
                            //todo 这个异常特殊处理 视频转码可能有问题，最后可能缺少一片
                            if (isCompletion.not()) {
                                isCompletion = true
                                listeners.forEach { listener ->
                                    listener?.onCompletion(mediaSource)
                                }
                            }
                            return
                        }
                    }
                }
            }
            ORPlayerLog.e(TAG, "exo onPlayerError，MimeType:$videoMimeType, code:${error.errorCode},msg:${error.message}, cause:${error.cause}", true)
            //当前硬解播放失败，切换到软解播放
            if (curDecoderType == ORExoDecoderType.HARDWARE
                && mediaSource?.isLive == false
                && error.cause !is HttpDataSource.InvalidResponseCodeException
            ) {//http错误，换软解也没用，直接报错
                ORPlayerLog.e(TAG, "exo 硬解失败，切换到软解，MimeType:$videoMimeType, code:${error.errorCode},msg:${error.message}", true)
                changeToSoftDecoder()?.let { mediaSource ->
                    setDataSource(mediaSource)
                    prepare()
                }
                return
            }
            val playError = PlayError(error.errorCode, error.message + "|${error.cause.toString()}").apply {
                exoCause = error.cause
            }

            if (errorInterceptorListener?.onPlayError(TnPlayerType.EXO, playError, mediaSource) == true) {
                Log.e(TAG, "exo onError 被外部拦截掉")
            } else {
                listeners.forEach { listener ->
                    listener?.onPlayError(playError)
                }
            }
        }

        override fun onVideoSizeChanged(videoSize: VideoSize) {
            surfaceView?.setVideoRotation(videoSize.unappliedRotationDegrees)
            textureView?.setVideoRotation(videoSize.unappliedRotationDegrees)
            surfaceView?.setVideoSize(videoSize.width, videoSize.height)
            textureView?.setVideoSize(videoSize.width, videoSize.height)
            listeners.forEach { listener ->
                listener?.onVideoSizeChanged(videoSize.width, videoSize.height)
            }
            Log.e(TAG, "onVideoSizeChanged width:${videoSize.width}  height:${videoSize.height}")
        }

        override fun onEvents(player: Player, events: Player.Events) {
        }

        override fun onPositionDiscontinuity(
            oldPosition: Player.PositionInfo,
            newPosition: Player.PositionInfo,
            reason: Int
        ) {
            val currentWindowIndex: Int = newPosition.mediaItemIndex
            val currentPositionMs: Long = newPosition.positionMs
            Log.d(
                TAG, "onPositionDiscontinuity: currentWindowIndex=" + currentWindowIndex
                        + ", currentPositionMs=" + currentPositionMs
                        + ", reason=" + reason
            )
            if (reason == Player.DISCONTINUITY_REASON_SEEK || reason == Player.DISCONTINUITY_REASON_SEEK_ADJUSTMENT) {
                listeners.forEach { listener -> listener?.setOnSeekCompleteListener() }
            }
            listeners.forEach { listener -> listener?.onProgress(currentPositionMs, mediaSource) }
        }


        override fun onMediaItemTransition(mediaItem: MediaItem?, reason: Int) {
            Log.e(TAG, "onMediaItemTransition  reason:$reason ")
            if (reason == Player.MEDIA_ITEM_TRANSITION_REASON_AUTO || reason == Player.MEDIA_ITEM_TRANSITION_REASON_REPEAT) {
                listeners.forEach { listener -> listener?.onMediaItemTransition(mediaItem?.localConfiguration?.customCacheKey) }
            }
        }

        override fun onTracksChanged(tracks: Tracks) {
            listeners.forEach { listener -> listener?.onTracksChange(tracks.toTnTracks()) }
        }

    }

    init {
        if (openAudioFocus) {
            initAudioFocus()
        }
        createPlayer(ORExoDecoderType.HARDWARE)
    }

    var rendererFactory: RenderersFactory? = null

    private var listeners: CopyOnWriteArrayList<IPlayerListener> = CopyOnWriteArrayList()

    private fun createPlayer(decoderType: ORExoDecoderType) {
        ORPlayerLog.d(TAG, "createPlayer, DecoderType:$decoderType", true)
        rendererFactory = DemoUtil.buildRenderersFactory(context, decoderType)
        exoPlayer = ExoPlayer.Builder(context)
//            .setLivePlaybackSpeedControl(createLiveSpeedControl())
            .setMediaSourceFactory(
                DefaultMediaSourceFactory(context).setDataSourceFactory(
                    DemoUtil.getDataSourceFactory(
                        context
                    )
                )
            )
            .setRenderersFactory(rendererFactory!!)
            .build()
            .apply {
                playWhenReady = false
//                repeatMode = Player.REPEAT_MODE_ALL
                addListener(exoPlayerListener)
                addAnalyticsListener(object : EventLogger(TAG) {

                    override fun onTracksChanged(eventTime: AnalyticsListener.EventTime, tracks: Tracks) {
                        super.onTracksChanged(eventTime, tracks)
                        onExoTracksChanged(eventTime, tracks)
                    }

                    override fun onAudioDecoderInitialized(
                        eventTime: AnalyticsListener.EventTime,
                        decoderName: String,
                        initializedTimestampMs: Long,
                        initializationDurationMs: Long
                    ) {
                        super.onAudioDecoderInitialized(eventTime, decoderName, initializedTimestampMs, initializationDurationMs)
                        ORPlayerLog.v(TAG, "onExoAudioDecoderInitialize, decoderName:$decoderName", true)
                    }

                    override fun onVideoDecoderInitialized(
                        eventTime: AnalyticsListener.EventTime,
                        decoderName: String,
                        initializedTimestampMs: Long,
                        initializationDurationMs: Long
                    ) {
                        super.onVideoDecoderInitialized(eventTime, decoderName, initializedTimestampMs, initializationDurationMs)
                        ORPlayerLog.v(TAG, "onExoVideoDecoderInitialize, decoderName:$decoderName", true)
                    }

                })
            }
    }

    private fun onPrepare2GetBitrate() {
        exoPlayer?.currentTracks?.let { tracks ->
            for (groupIndex in tracks.groups.indices) {
                val trackGroup: Tracks.Group = tracks.groups[groupIndex]
                for (index in 0 until trackGroup.length) {
                    val format = trackGroup.getTrackFormat(index)
                    if (MimeTypes.isVideo(format.sampleMimeType)) {
                        if (exoPlayer?.videoFormat?.bitrate == format.bitrate) {
                            videoBitrate = format.bitrate
                            ORPlayerLog.d(TAG, "--onPrepare2GetBitrate  MimeType:${format.sampleMimeType}，videoBitrate：$videoBitrate")
                            listeners.forEach { listener ->
                                listener.onTracksVideoBitrateChange(videoBitrate)
                            }
                        }
                    } else if (MimeTypes.isAudio(format.sampleMimeType)) {
                        if (exoPlayer?.audioFormat?.bitrate == format.bitrate) {
                            audioBitrate = format.bitrate
                            ORPlayerLog.d(TAG, "--onPrepare2GetBitrate  MimeType:${format.sampleMimeType}，audioBitrate：$audioBitrate")
                            listeners.forEach { listener ->
                                listener.onTracksAudioBitrateChange(audioBitrate)
                            }
                        }
                    }
                }
            }
        }

    }

    /**
     * 播放之前获取音视频track信息
     * [MimeTypes]
     * 该方法最先调用，然后初始化解码器
     */
    private fun onExoTracksChanged(eventTime: AnalyticsListener.EventTime, tracks: Tracks) {
        //vp9视频-4.4以上，现在全机型都支持，无需扩展解码器
        //id=1, mimeType=video/x-vnd.on2.vp9, res=1920x1080, color=BT709/Limited range/SDR SMPTE 170M/8/8, fps=24.192997, supported=YES
        //id=2, mimeType=audio/mp4a-latm, bitrate=129157, codecs=mp4a.40.2, channels=2, sample_rate=24000, language=en, supported=YES

        //av1视频-10+支持，以下的版本需要扩展解码器，10+的无需扩展解码器
        //id=1, mimeType=video/av01, res=1920x1080, color=BT709/Limited range/SDR SMPTE 170M/8/8, fps=25.0, supported=NO_UNSUPPORTED_TYPE
        //id=2, mimeType=audio/mp4a-latm, bitrate=128000, codecs=mp4a.40.2, channels=2, sample_rate=44100, language=en, supported=YES

        //he265视频-看设备视频，硬解有些支持，有些不支持，需要动态切换
        //id=1, mimeType=video/hevc, codecs=hvc1.1.6.L90.90, res=860x360, color=NA/NA/NA/8/8, fps=23.97602, supported=YES
        //id=2, mimeType=audio/mp4a-latm, bitrate=62794, codecs=mp4a.40.2, channels=2, sample_rate=22050, language=und, supported=YES

        //10以下的av1视频，能播放成功不报错，但是只有音频(有onAudioDecoderInitialized回调)没有视频(没有onAudioDecoderInitialized回调)，
        // 所以需要判断超时，切换av1扩展编码器，
        // 如果失败那跟ffmpeg一样直接切换即可
        for (groupIndex in tracks.groups.indices) {
            val trackGroup: Tracks.Group = tracks.groups[groupIndex]
            for (index in 0 until trackGroup.length) {
                val format = trackGroup.getTrackFormat(index)
                ORPlayerLog.v(TAG, "onGetTracksInfo  MimeType:${format.sampleMimeType}， bitrate：${format.bitrate}")
                if (format.sampleMimeType?.contains(MimeTypes.BASE_TYPE_VIDEO) == true) {
                    videoMimeType = format.sampleMimeType
                    ORPlayerLog.i(TAG, "--------onGetTracksInfo curVideoMimeType:$videoMimeType")
                }
            }
        }
        if (curDecoderType != ORExoDecoderType.AV1
            && videoMimeType == MimeTypes.VIDEO_AV1
            && Build.VERSION.SDK_INT <= Build.VERSION_CODES.Q
        ) {
            //针对<=10的，编码为av1的，直接强制切到软解播放
            ORPlayerLog.e(TAG, "exo av1，强制切换到软解， MimeType:$videoMimeType", true)
            changeToSoftDecoder()?.let { mediaSource ->
                setDataSource(mediaSource)
                prepare()
            }
        }
    }

    /**
     * 切换到软解
     * 需要重新创建新播放器
     */
    private fun changeToSoftDecoder(): MediaSource? {
        val curMediaSource = mediaSource
        val curVideoMimeType = videoMimeType
        //切换编码，需要重新创建播放器，所以把之前的release掉
        innerRelease()
        curDecoderType = when (curVideoMimeType) {
            MimeTypes.VIDEO_AV1 -> {
                ORExoDecoderType.AV1
            }

            else -> {//vp9设备就支持，其他类型的用ffmpeg基本可以支持
                ORExoDecoderType.FFMPEG
            }
        }
        ORPlayerLog.d(TAG, "重新创建播放器切换编码, DecoderType:$curDecoderType,MimeTyp:$curVideoMimeType", true)
        createPlayer(curDecoderType)
        initNewPlayer()//重新创建播放器，配置需要重新设置
        return curMediaSource
    }

    private fun initNewPlayer() {
        setMute(mute)
        volume?.let {
            setVolume(it)
        }
        setLooping(isLoop)
        setAutoPlay(isAutoPlay)
        setSpeed(speed)
        if (null != surfaceView) {
            setSurfaceView(surfaceView)
        }
        if (null != textureView) {
            setTextureView(textureView)
        }
        setScaleMode(scaleMode)
    }

    override fun isPrepared(): Boolean {
        return exoPlayer?.playbackState == Player.STATE_READY
    }

    override fun setSurfaceView(surfaceView: SurfaceView?) {
        this.surfaceView = surfaceView as? SurfaceRenderView
        exoPlayer?.setVideoSurfaceView(surfaceView)
    }

    override fun setTextureView(textureView: TextureView?) {
        this.textureView = textureView as? TextureRenderView
        exoPlayer?.setVideoTextureView(textureView)
    }

    override fun clearSurfaceOnly() {
        exoPlayer?.setVideoSurfaceView(null)
        exoPlayer?.setVideoTextureView(null)
        surfaceView?.holder?.surface?.release()
        textureView?.surfaceTexture?.release()
    }

    override fun getBitrate(): Pair<Int, Int> {
        return Pair(videoBitrate, audioBitrate)
    }

    override fun setPlayerListener(listener: IPlayerListener) {
        addPlayerListener(listener)
    }

    override fun addPlayerListener(listener: IPlayerListener) {
        super.addPlayerListener(listener)
        if (!listeners.contains(listener)) {
            listeners.add(listener)
        }
    }

    override fun removePlayerListener(listener: IPlayerListener) {
        super.removePlayerListener(listener)
        listeners.remove(listener)
    }


    override fun prepare() {
        Log.e(TAG, "prepare")
        exoPlayer?.prepare()
        isCompletion = false
    }

    override fun play() {
        Log.e(TAG, "play")
        if (exoPlayer?.playerError != null) {
            exoPlayer?.prepare()
        }
        exoPlayer?.play()
        audioFocusHelper?.isUserPause = false
        audioFocusHelper?.requestFocus()
        isCompletion = false
    }

    override fun pause() {
        audioFocusHelper?.isUserPause = true
        Log.e(TAG, "pause")
        exoPlayer?.pause()
        exoPlayer?.playWhenReady
        audioFocusHelper?.abandonFocus()
    }

    override fun stop() {
        Log.e(TAG, "stop    ")
        audioFocusHelper?.isUserPause = true
        exoPlayer?.stop()
        isPrepare = false
        isCompletion = false
        isRenderFirstFrame = false
        audioFocusHelper?.abandonFocus()
    }

    override fun seekToDefaultPosition() {
        exoPlayer?.seekToDefaultPosition()
    }

    override fun release() {
        isPrepare = false
        isCompletion = false
        isRenderFirstFrame = false
        Log.e(TAG, "release")
        if (TnPlayerPool.needReleasePlayer(this)) {
            exoPlayer?.release()
        } else {
            exoPlayer?.stop()
            exoPlayer?.clearMediaItems()
            exoPlayer?.clearVideoSurface()

        }
        listeners.forEach { listener ->
            listener?.onPlayerRelease(mediaSource)
        }
        audioFocusHelper?.abandonFocus()
        videoMimeType = null
        curDecoderType = ORExoDecoderType.HARDWARE
        mediaSource = null
    }

    private fun innerRelease() {
        isPrepare = false
        isCompletion = false
        isRenderFirstFrame = false
        Log.e(TAG, "innerRelease")
        if (TnPlayerPool.needReleasePlayer(this)) {
            exoPlayer?.release()
        } else {
            exoPlayer?.stop()
            exoPlayer?.clearMediaItems()
            exoPlayer?.clearVideoSurface()
        }
    }

    override fun reset() {
        Log.e(TAG, "reset    ")
        isPrepare = false
        isCompletion = false
        isRenderFirstFrame = false
        exoPlayer?.stop()
        exoPlayer?.clearMediaItems()
        curDecoderType = ORExoDecoderType.HARDWARE
        mediaSource = null
        listeners.forEach { listener ->
            listener?.onPlayerReset()
        }
    }

    override fun clearScreen() {
    }

    override fun seekTo(mills: Long) {
        Log.e(TAG, "seekTo    mills:$mills")
        if (exoPlayer?.playerError != null) {
            exoPlayer?.prepare()
        }
        exoPlayer?.seekTo(mills)
    }

    override fun getDuration(): Long = exoPlayer?.duration ?: 0L

    override fun isPlaying(): Boolean {
        return exoPlayer?.isPlaying ?: false
    }

    override fun isComplete(): Boolean {
        return exoPlayer?.playbackState == Player.STATE_ENDED
    }

    override fun setLooping(isLoop: Boolean) {
        this.isLoop = isLoop
        exoPlayer?.repeatMode = if (isLoop) Player.REPEAT_MODE_ONE else Player.REPEAT_MODE_OFF
    }

    override fun isMute(): Boolean {
        return exoPlayer?.isDeviceMuted ?: false
    }

    override fun setMute(mute: Boolean) {
        this.mute = mute
        exoPlayer?.setDeviceMuted(mute, C.VOLUME_FLAG_SHOW_UI)
    }

    override fun setVolume(volume: Float) {
        val mVolume = volume.coerceIn(0f, 2f)
        (rendererFactory as? ORRenderersFactory)?.setVolume(mVolume)
        exoPlayer?.volume = mVolume
        this.volume = mVolume
        Log.d("ORExoPlayer_Volume", "setVolume  volume:$volume")
    }

    override fun getVolume(): Float? {
        return volume
    }

    override fun setAutoPlay(isAutoPlay: Boolean) {
        this.isAutoPlay = isAutoPlay
        exoPlayer?.playWhenReady = isAutoPlay
    }

    override fun enableHardwareDecoder(enable: Boolean) {
        if (enable.not() && curDecoderType == ORExoDecoderType.HARDWARE) {
            //硬解切到软解
            ORPlayerLog.d(TAG, "1-硬解切到软解，重新创建设置", true)
            changeToSoftDecoder()
        } else if (enable && curDecoderType != ORExoDecoderType.HARDWARE) {
            //软解切到硬解
            curDecoderType = ORExoDecoderType.HARDWARE
            ORPlayerLog.d(TAG, "2-软解切换到硬解，重新创建设置", true)
            createPlayer(curDecoderType)
            initNewPlayer()
        }
    }

    override fun setDataSource(mediaSource: MediaSource) {
        ORPlayerLog.d(TAG, "setDataSource url:${mediaSource.url}", true)
        listeners.forEach {
            it.onSetDataSource()
        }
        if (isStreamType(mediaSource)) {
            addStreamDataSource(mediaSource)
            return
        }
        if (isRTMPType(mediaSource)) {
            addRtmpDataSource(mediaSource)
            return
        }

        val mediaItem = MediaItem.Builder()
            .setUri(mediaSource.url)
            .setMediaId(mediaSource.weights.toString())
            .setCustomCacheKey(mediaSource.key)
            .build()
        exoPlayer?.setMediaItem(mediaItem)
        this.mediaSource = mediaSource
    }

    private fun isStreamType(mediaSource: MediaSource): Boolean {
        if (mediaSource.mimeType == PlayMimeType.DASH ||
            mediaSource.mimeType == PlayMimeType.HLS
        ) {
            return true
        }
        if (mediaSource.url.contains(".mpd")) {
            mediaSource.mimeType = PlayMimeType.DASH
            return true
        }
        if (mediaSource.url.contains(".m3u8")) {
            mediaSource.mimeType = PlayMimeType.HLS
            return true
        }
        return false
    }

    private fun isRTMPType(mediaSource: MediaSource): Boolean {
        if (mediaSource.mimeType == PlayMimeType.RTMP) {
            return true
        }
        if (mediaSource.url.lowercase().startsWith("rtmp://")) {
            mediaSource.mimeType = PlayMimeType.RTMP
            return true
        }
        return false
    }

    private fun addRtmpDataSource(mediaSource: MediaSource){
        val rtmpDataSourceFactory = RtmpDataSource.Factory()
        ORPlayerLog.d(TAG, "addRtmpDataSource ", true)
        val newSource = ProgressiveMediaSource.Factory(rtmpDataSourceFactory)
            .createMediaSource(MediaItem.fromUri(mediaSource.url))
        exoPlayer?.setMediaSource(newSource)
    }

    override fun seekTo(uuid: String, mills: Long) {
        super.seekTo(uuid, mills)
        for (i in 0 until (exoPlayer?.mediaItemCount ?: 0)) {
            if (exoPlayer?.getMediaItemAt(i)?.localConfiguration?.customCacheKey == uuid) {
                if (exoPlayer?.playerError != null) {
                    prepare()
                }
                if (exoPlayer?.currentMediaItemIndex != i) {
                    Log.e(TAG, "seekTo  index:$i uuid:$uuid")
                    exoPlayer?.seekTo(i, mills)
                } else {
                    if (isComplete()) {
                        exoPlayer?.seekTo(mills)
                    }
                    Log.e(TAG, "seekTo   uuid:$uuid mills:$mills  是当前视频直接播放")
                }
                play()
                return
            }
        }
        Log.e(TAG, "seekTo   uuid:$uuid mills:$mills  没有找到")
        pause()
    }

    private val handler = Handler(Looper.myLooper()!!)
    private val timerTask = Runnable {
        if (exoPlayer?.isPlaying != false) {
            val progress = exoPlayer?.currentPosition ?: 0L
            listeners.forEach {
                it.onProgress(progress, mediaSource)
            }
            loop()
        }
    }

    private fun loop() {
        handler.removeCallbacks(timerTask)
        handler.postDelayed(timerTask, 500)
    }


    fun addStreamDataSource(mediaSource: MediaSource): Boolean {
        val path = mediaSource.localPath ?: mediaSource.url
        val mediaItem = MediaItem.Builder()
            .setUri(path)
            .setMediaId(mediaSource.weights.toString())
            .setCustomCacheKey(mediaSource.key)
            .setTag(mediaSource.id)
            .build()

        val factory = if (path.startsWith("http")) {
            DefaultHttpDataSource.Factory().apply {
                mediaSource.headers?.let {
                    setDefaultRequestProperties(it)
                }
            }
        } else {
            DefaultDataSource.Factory(context)
        }

        Log.d(TAG, "dash addStreamDataSource, mimeType:${mediaSource.mimeType}")
        val mediaSourceFactory = when (mediaSource.mimeType) {
            PlayMimeType.DASH -> DashMediaSource.Factory(factory)
            PlayMimeType.HLS -> HlsMediaSource.Factory(factory)
            else -> return false
        }
        mediaSourceFactory.setDrmSessionManagerProvider { _: MediaItem? -> DrmSessionManager.DRM_UNSUPPORTED }
            .createMediaSource(mediaItem).apply {
                exoPlayer?.setMediaSource(this)
            }

        this.mediaSource = mediaSource
        return true
    }

    override fun currentMediaSource(): MediaSource? {
        return mediaSource
    }

    override fun getCurrentTracks(): TnTracks {
        return exoPlayer?.currentTracks?.toTnTracks() ?: TnTracks()
    }

    override fun getCurrentPosition(): Long {
        return exoPlayer?.currentPosition ?: 0L
    }

    override fun getCurrentVideoFormat(): TnFormat? {
        return exoPlayer?.videoFormat?.toTnFormat()
    }

    override fun changeTrackSelection(mediaTrackGroup: TnTracksGroup, index: Int) {
        kotlin.runCatching {
            videoBitrate = mediaTrackGroup.formats[index].bitrate
            listeners.forEach { listener ->
                listener.onTracksVideoBitrateChange(videoBitrate)
            }
        }
        Log.d(TAG, "changeTrackSelection  mediaTrackGroup:$mediaTrackGroup  index:$index, videoBitrate:$videoBitrate")
        exoPlayer?.currentTracks?.findTrackGroup(mediaTrackGroup)?.let {
            exoPlayer!!.trackSelectionParameters =
                exoPlayer!!.trackSelectionParameters
                    .buildUpon()
                    .setOverrideForType(
                        TrackSelectionOverride(/* mediaTrackGroup = */ it, /* trackIndex= */
                            index
                        )
                    )
                    .build()
        }
    }

    override fun setSpeed(speed: Float) {
        this.speed = speed
        exoPlayer?.setPlaybackSpeed(speed)
    }

    private var scaleMode: ScaleMode = ScaleMode.SCALE_TO_FILL
    override fun setScaleMode(scaleMode: ScaleMode) {
        super.setScaleMode(scaleMode)
        this.scaleMode = scaleMode
        val renderScaleMode: RenderScaleMode = when (scaleMode) {
            ScaleMode.SCALE_TO_FILL -> {
                RenderScaleMode.SCREEN_SCALE_MATCH_PARENT
            }

            ScaleMode.SCALE_ASPECT_FIT -> {
                RenderScaleMode.SCREEN_SCALE_DEFAULT
            }

            ScaleMode.SCALE_ASPECT_FILL -> {
                RenderScaleMode.SCREEN_SCALE_CENTER_CROP
            }
        }
        textureView?.setScaleType(renderScaleMode)
        surfaceView?.setScaleType(renderScaleMode)
    }

    override fun setErrorInterceptor(listener: TnErrorInterceptorListener) {
        errorInterceptorListener = listener
    }

    // 低延迟播放控制配置
    private fun createLiveSpeedControl(): DefaultLivePlaybackSpeedControl {
        return DefaultLivePlaybackSpeedControl.Builder()
            .setTargetLiveOffsetIncrementOnRebufferMs(3000)// 目标延迟 3 秒
            .setFallbackMinPlaybackSpeed(0.95f)// 最低播放速度
            .setFallbackMaxPlaybackSpeed(1.05f)// 最高播放速度
            .build().apply {
            }
    }
}