package com.transsion.player.orplayer

import android.content.Context
import android.graphics.SurfaceTexture
import android.os.Handler
import android.os.HandlerThread
import android.os.Looper
import android.util.Log
import android.view.Surface
import android.view.SurfaceHolder
import android.view.SurfaceView
import android.view.TextureView
import com.aliyun.player.AliPlayer
import com.aliyun.player.AliPlayerFactory
import com.aliyun.player.AliPlayerGlobalSettings
import com.aliyun.player.IPlayer
import com.aliyun.player.bean.ErrorCode
import com.aliyun.player.bean.InfoCode
import com.aliyun.player.nativeclass.CacheConfig
import com.aliyun.player.nativeclass.PlayerConfig
import com.aliyun.player.nativeclass.TrackInfo
import com.aliyun.player.source.UrlSource
import com.blankj.utilcode.util.GsonUtils
import com.transsion.player.MediaSource
import com.transsion.player.config.ORPlayerConfig
import com.transsion.player.config.VodConfig
import com.transsion.player.enum.ScaleMode
import com.transsion.player.orplayer.global.TnErrorInterceptorListener
import com.transsion.player.orplayer.global.TnPlayerType
import com.transsion.player.utils.MD5
import com.transsion.player.utils.ORPlayerLog
import java.io.File
import java.net.URI
import java.util.concurrent.CopyOnWriteArrayList


/**
 * @author xugaopan
 * @date 2022/8/9
 */
class ORPlayerImpl : ORPlayer {

    private val TAG = "ORPlayerImpl"
    private var aliPlayer: AliPlayer? = null
    private var audioFocusHelper: AudioFocusHelper? = null
    private var onPlayerListener: IPlayerListener? = null
    private var curUrl = ""
    private var mediaSource: MediaSource? = null

    private var isPlaying = false
    private var isLoading = false
    private var isReset = false
    private var isComplete = false
    private var isPreparer = false

    private var listeners: CopyOnWriteArrayList<IPlayerListener> = CopyOnWriteArrayList()

    private var errorInterceptorListener: TnErrorInterceptorListener? = null
    private var context: Context

    private var currentPosition = 0L
    private var videoBitrate = 0
    private var audioBitrate = 0

    //是否是内部解码错误切换视频
    private var isInnerChangeDecoder = false

    constructor(context: Context, vodConfig: VodConfig?) {
        this.context = context
        aliPlayer = AliPlayerFactory.createAliPlayer(context).apply {
            setTraceId(vodConfig?.traceId)
            enableHardwareDecoder(vodConfig?.enableHardwareDecoder ?: true)
        }
        if (vodConfig?.openAudioFocus != false) {
            initAudioFocus()
        }
        setPlayerConfig(vodConfig ?: ORPlayerConfig.getVodConfig())
        initListener(vodConfig?.openCache ?: true)
    }

    private fun initAudioFocus() {
        val audioFocusPlayer = object : AudioFocusPlay {
            override fun play() {
                aliPlayer?.start()
            }

            override fun pause() {
                aliPlayer?.pause()
            }

            override fun isMute(): Boolean {
                return aliPlayer?.isMute ?: false
            }

            override fun setVolume(volume: Float) {
                aliPlayer?.volume = volume
            }
        }
        audioFocusHelper = AudioFocusHelper(audioFocusPlayer) { focusLoss ->
            onPlayerListener?.onFocusChange(focusLoss)
            listeners.forEach { listener ->
                listener.onFocusChange(focusLoss)
            }
        }
    }

    private fun initListener(cacheable: Boolean) {
        setCache(ORPlayerConfig.getVodConfig(), ORPlayerConfig.openCache() && cacheable)
        aliPlayer?.setOnRenderingStartListener {
            aliPlayer?.mediaInfo?.trackInfos?.forEach { trackInfo ->
                ORPlayerLog.d(TAG, "setOnInfoListener onRenderFirstFrame ${GsonUtils.toJson(trackInfo)}", true)
                when (trackInfo.mType) {
                    TrackInfo.Type.TYPE_VIDEO -> {
                        videoBitrate = trackInfo.videoBitrate
                        onPlayerListener?.onTracksVideoBitrateChange(videoBitrate)
                        listeners.forEach { listener ->
                            listener.onTracksVideoBitrateChange(videoBitrate)
                        }
                    }

                    TrackInfo.Type.TYPE_AUDIO -> {
                        audioBitrate = trackInfo.videoBitrate
                        onPlayerListener?.onTracksAudioBitrateChange(audioBitrate)
                        listeners.forEach { listener ->
                            listener.onTracksAudioBitrateChange(audioBitrate)
                        }
                    }

                    else -> {}
                }
            }

            Log.d(TAG, "setOnInfoListener onRenderFirstFrame，videoBitrate:$videoBitrate, audioBitrate:$audioBitrate")
            onPlayerListener?.onRenderFirstFrame()
            listeners.forEach { listener ->
                listener.onRenderFirstFrame()
            }
        }

        aliPlayer?.setOnVideoSizeChangedListener { width, height ->
            onPlayerListener?.onVideoSizeChanged(width, height)
            listeners.forEach { listener ->
                listener.onVideoSizeChanged(width, height)
            }
        }

        aliPlayer?.setOnErrorListener {
            Log.d(TAG, "onError code:${it.code},msg:${it.msg}")
            val errorInfo = PlayError(it.code.value, it.msg)
            if (handleDecodeError(errorInfo)) {
                return@setOnErrorListener
            }
            if (errorInterceptorListener?.onPlayError(TnPlayerType.ALIYUN, errorInfo, mediaSource) == true) {
                ORPlayerLog.e(TAG, "aliyun onError 被外部拦截掉", true)
                return@setOnErrorListener
            } else {
                onPlayerListener?.onPlayError(errorInfo, mediaSource)
                listeners.forEach { listener ->
                    listener.onPlayError(errorInfo, mediaSource)
                }
            }
            if (it.code == ErrorCode.ERROR_DECODE_AUDIO) {
                prepare()
                play()
            }
        }

        aliPlayer?.setOnLoadingStatusListener(object : IPlayer.OnLoadingStatusListener {
            override fun onLoadingBegin() {
                Log.d(TAG, "onLoadingBegin--->")
                isLoading = true
                onPlayerListener?.onLoadingBegin(mediaSource)
                listeners.forEach { listener ->
                    listener.onLoadingBegin(mediaSource)
                }
            }

            override fun onLoadingProgress(percent: Int, netSpeed: Float) {
                Log.d(TAG, "onLoadingProgress percent:$percent,netSpeed:$netSpeed")
                //这个方法没有回调
//                onPlayerListener?.onLoadingProgress(percent,netSpeed)
            }

            override fun onLoadingEnd() {
                Log.d(TAG, "onLoadingEnd----->")
                isLoading = false
                onPlayerListener?.onLoadingEnd(mediaSource)
                listeners.forEach { listener ->
                    listener.onLoadingEnd(mediaSource)
                }
            }

        })

        aliPlayer?.setOnPreparedListener {
            isPreparer = true
            Log.d(TAG, "setOnInfoListener onPrepare")
            if (errorInterceptorListener?.onPrepare(mediaSource) == true) {
                ORPlayerLog.d(TAG, "aliyun onPrepare 被外部拦截掉", true)
            } else if (isInnerChangeDecoder && currentPosition > 0) {
                //没有播放进度则为首次，直接抛出去处理
                ORPlayerLog.d(TAG, "aliyun onPrepare 内部的解码切换，已有播放进度，直接播放", true)
                seekTo(currentPosition)
                play()
            } else {
                onPlayerListener?.onPrepare(mediaSource)
                listeners.forEach { listener ->
                    listener.onPrepare(mediaSource)
                }
            }
        }

        aliPlayer?.setOnInfoListener { infoBean ->
            val extraValue = infoBean.extraValue
            if (infoBean.code == InfoCode.CurrentPosition) {
                currentPosition = extraValue
                ////extraValue为当前播放进度，单位为毫秒
                onPlayerListener?.onProgress(extraValue, mediaSource)
                listeners.forEach { listener ->
                    listener.onProgress(extraValue, mediaSource)
                }
            }

            if (infoBean.code == InfoCode.LoopingStart) {
                onPlayerListener?.onLoopingStart()
                listeners.forEach { listener ->
                    listener.onLoopingStart()
                }
            }

            if (infoBean.code == InfoCode.BufferedPosition) {
                onPlayerListener?.onBufferedPosition(extraValue, mediaSource)
                listeners.forEach { listener ->
                    listener.onBufferedPosition(extraValue, mediaSource)
                }
            }
            if (infoBean.code == InfoCode.SwitchToSoftwareVideoDecoder) {
                //切换到软解
                ORPlayerLog.i(TAG, "aliyun 硬解失败，自动切换到软件", writeToFile = true)
            }
        }

        aliPlayer?.setOnStateChangedListener {
            Log.d(TAG, "setOnStateChangedListener status:${it}")
            isPlaying = (it == PlayStatus.STARTED)
            onPlayerListener?.onIsPlayingChanged(isPlaying)
            listeners.forEach { listener ->
                listener.onIsPlayingChanged(isPlaying)
            }
            isComplete = (it == PlayStatus.COMPLETION)
            audioFocusHelper?.isPlaying = isPlaying
            when (it) {
                PlayStatus.IDLE -> {}
                PlayStatus.INITALIZED -> {}
                PlayStatus.PREPARED -> {}
                PlayStatus.STARTED -> {
                    onPlayerListener?.onVideoStart(mediaSource)
                    listeners.forEach { listener ->
                        listener.onVideoStart(mediaSource)
                    }
                }

                PlayStatus.PAUSED -> {
                    onPlayerListener?.onVideoPause(mediaSource)
                    listeners.forEach { listener ->
                        listener.onVideoPause(mediaSource)
                    }
                }

                PlayStatus.STOPPED -> {}
                PlayStatus.COMPLETION -> {
                    onPlayerListener?.onCompletion(mediaSource)
                    listeners.forEach { listener ->
                        listener.onCompletion(mediaSource)
                    }
                }

                PlayStatus.ERROR -> {}
            }

        }

        aliPlayer?.setOnSeekCompleteListener {
            onPlayerListener?.setOnSeekCompleteListener()
            listeners.forEach {
                it.setOnSeekCompleteListener()
            }
        }
    }


    private var handler: Handler? = null
    override fun setSurfaceView(surfaceView: SurfaceView?) {
        clearSurface()
        this.surfaceView = surfaceView
        val currentThread = Thread.currentThread()
        if (null != surfaceView) {
            handler = if (currentThread is HandlerThread) {
                Handler(currentThread.looper)
            } else {
                Handler(Looper.getMainLooper())
            }
        }
        surfaceView?.holder?.addCallback(mSurfaceCallback)
        surfaceView?.holder?.surface?.let {
            if (it.isValid) {
                aliPlayer?.setSurface(it)
            }
        }
    }

    private val mSurfaceCallback = object : SurfaceHolder.Callback {
        override fun surfaceCreated(holder: SurfaceHolder) {
            handler?.post {
                val surface = holder.surface
                if (surface.isValid) {
                    aliPlayer?.setSurface(surface)
                }
            }
        }

        override fun surfaceChanged(
            holder: SurfaceHolder,
            format: Int,
            width: Int,
            height: Int
        ) {
            runOnPlayerThread {
                aliPlayer?.surfaceChanged()
            }
        }

        override fun surfaceDestroyed(holder: SurfaceHolder) {
            runOnPlayerThread {
                aliPlayer?.setSurface(null)
            }
        }
    }

    private fun runOnPlayerThread(block: () -> Unit) {
        val h = handler
        if (null == h || Thread.currentThread() == h.looper.thread) {
            block()
        } else {
            h.post {
                block()
            }
        }
    }

    private var textureView: TextureView? = null
    private var surfaceView: SurfaceView? = null
    override fun setTextureView(textureView: TextureView?) {
        clearSurface()
        this.textureView = textureView
        val currentThread = Thread.currentThread()
        if (null != surfaceView) {
            handler = if (currentThread is HandlerThread) {
                Handler(currentThread.looper)
            } else {
                Handler(Looper.getMainLooper())
            }
        }
        textureView?.surfaceTextureListener = mSurfaceTextureListener
        textureView?.surfaceTexture?.let {
            val surface = Surface(it)
            if (surface.isValid) {
                aliPlayer?.setSurface(surface)
            }
        }
    }

    private fun clearSurface() {
        this.surfaceView?.holder?.removeCallback(mSurfaceCallback)
        this.surfaceView = null

        this.textureView?.surfaceTextureListener = null
        this.textureView = null
    }

    private val mSurfaceTextureListener = object : TextureView.SurfaceTextureListener {
        override fun onSurfaceTextureAvailable(
            surfaceTexture: SurfaceTexture,
            width: Int,
            height: Int
        ) {
            runOnPlayerThread {
                val surface = Surface(surfaceTexture)
                if (surface.isValid) {
                    aliPlayer?.setSurface(surface)
                }
            }
        }

        override fun onSurfaceTextureSizeChanged(
            surface: SurfaceTexture,
            width: Int,
            height: Int
        ) {
            runOnPlayerThread {
                aliPlayer?.surfaceChanged()
            }
        }

        override fun onSurfaceTextureDestroyed(surface: SurfaceTexture): Boolean {
            runOnPlayerThread {
                aliPlayer?.setSurface(null)
            }
            return false
        }

        override fun onSurfaceTextureUpdated(surface: SurfaceTexture) {
        }
    }

    override fun clearSurfaceOnly() {
        aliPlayer?.setSurface(null)
        surfaceView?.holder?.surface?.release()
        textureView?.surfaceTexture?.release()
        clearSurface()
    }

    override fun getBitrate(): Pair<Int, Int> {
        return Pair(videoBitrate, audioBitrate)
    }

    private fun setCache(vodConfig: VodConfig, enable: Boolean) {
        val cacheConfig = CacheConfig()
        //开启缓存功能
        cacheConfig.mEnable = enable
        //能够缓存的单个文件最大时长。超过此长度则不缓存
        cacheConfig.mMaxDurationS = 120
        //缓存目录的位置
        val path = context.externalCacheDir?.absolutePath + File.separator + "video_cache"
        cacheConfig.mDir = path
        //缓存目录的最大大小。超过此大小，将会删除最旧的缓存文件
        cacheConfig.mMaxSizeMB = vodConfig.cacheMaxSizeMB
        //设置缓存配置给到播放器
        aliPlayer?.setCacheConfig(cacheConfig)

        // 开启本地缓存，开启之后，会缓存到本地文件中。
        // @param enable - 本地缓存功能开关。true：开启本地缓存，false：关闭，默认关闭。
        // @param maxBufferMemoryKB - 新版本已废弃，暂无作用
        // @param localCacheDir - 本地缓存的文件目录，为绝对路径。
        //使用MediaLoader的预加载，必须这里开启才能生效
        runCatching {
            AliPlayerGlobalSettings.enableLocalCache(enable, 0, path)
            AliPlayerGlobalSettings.setCacheUrlHashCallback { url ->
                val videoUrlKey = try {
                    val uri = URI(url)
                    MD5.getStringMD5(
                        URI(
                            uri.scheme,
                            uri.authority,
                            uri.path,
                            null,
                            uri.fragment
                        ).toString()
                    )
                } catch (e: Exception) {
                    MD5.getStringMD5(url ?: "")
                }
                return@setCacheUrlHashCallback videoUrlKey
            }
        }
    }

    override fun setPlayerConfig(vodConfig: VodConfig) {
        val config = aliPlayer?.config as PlayerConfig
        config.mClearFrameWhenStop = vodConfig.clearFrameWhenStop
        config.mEnableLocalCache = vodConfig.openCache
        config.mPositionTimerIntervalMs = vodConfig.positionTimerIntervalMs
        config.mMaxBufferDuration = vodConfig.maxBufferDuration
        config.mHighBufferDuration = vodConfig.highBufferDuration
        config.mStartBufferDuration = vodConfig.startBufferDuration
        config.mNetworkRetryCount = vodConfig.retryCount
        config.mNetworkTimeout = vodConfig.networkTimeout
        config.mDisableAudio = vodConfig.disableAudio
        config.mDisableVideo = vodConfig.disableVideo
//        config.mHttpProxy = "10.150.194.242:8888"
        aliPlayer?.config = config
    }

    override fun setPlayerListener(listener: IPlayerListener) {
        onPlayerListener = listener
    }

    override fun addPlayerListener(listener: IPlayerListener) {
        if (listeners.contains(listener).not()) {
            listeners.add(listener)
        }
    }

    override fun removePlayerListener(listener: IPlayerListener) {
        listeners.remove(listener)
    }

    override fun setDataSource(mediaSource: MediaSource) {
        val source = UrlSource()
        curUrl = mediaSource.url
        this.mediaSource = mediaSource
        source.uri = curUrl
        onPlayerListener?.onSetDataSource()
        listeners.forEach {
            it.onSetDataSource()
        }
        aliPlayer?.setDataSource(source)
        isPreparer = false
        isInnerChangeDecoder = false
    }


    override fun prepare() {
        aliPlayer?.prepare()
        isReset = false
    }

    override fun play() {
        onPlayerListener?.initPlayer()
        listeners.forEach {
            it.initPlayer()
        }
        aliPlayer?.start()
        audioFocusHelper?.isUserPause = false
        audioFocusHelper?.requestFocus()
    }

    override fun pause() {
        audioFocusHelper?.isUserPause = true
        aliPlayer?.pause()
        audioFocusHelper?.abandonFocus()
    }

    override fun stop() {
        curUrl = ""
        audioFocusHelper?.isUserPause = true
        aliPlayer?.stop()
        audioFocusHelper?.abandonFocus()
        isInnerChangeDecoder = false
    }

    override fun release() {
        curUrl = ""
        aliPlayer?.release()
        onPlayerListener?.onPlayerRelease()
        listeners.forEach {
            it.onPlayerRelease()
        }
        audioFocusHelper?.abandonFocus()
        clearSurface()
        isPreparer = false
        isInnerChangeDecoder = false
    }

    override fun reset() {
        aliPlayer?.stop()
        aliPlayer?.reset()
        aliPlayer?.clearScreen()
        audioFocusHelper?.abandonFocus()
        if (!isReset) {
            onPlayerListener?.onPlayerReset()
            listeners.forEach {
                it.onPlayerReset()
            }
        }
        isReset = true
        isPreparer = false
        isInnerChangeDecoder = false
    }

    override fun clearScreen() {
        aliPlayer?.clearScreen()
    }

    override fun seekTo(mills: Long) {
//        aliPlayer?.seekTo(mills)
        aliPlayer?.setMaxAccurateSeekDelta(150_000)
        aliPlayer?.seekTo(mills, IPlayer.SeekMode.Accurate)
    }

    override fun getDuration(): Long {
        return aliPlayer?.duration ?: 0L
    }

    override fun isPlaying(): Boolean {
        return isPlaying
    }

    override fun isComplete(): Boolean {
        return isComplete
    }

    override fun setLooping(isLoop: Boolean) {
        aliPlayer?.isLoop = isLoop
    }

    override fun isMute(): Boolean {
        return aliPlayer?.isMute ?: false
    }

    override fun setMute(mute: Boolean) {
        aliPlayer?.isMute = mute
    }

    override fun setVolume(volume: Float) {
        Log.d("ORAliPlayer_Volume","setVolume   volume:$volume")
        aliPlayer?.volume = volume.coerceIn(0f, 2f)
    }

    override fun getVolume(): Float? {
        return aliPlayer?.volume
    }

    override fun isLoading(): Boolean {
        return isLoading
    }

    override fun getVideoWidth(): Int {
        return aliPlayer?.videoWidth ?: super.getVideoWidth()
    }

    override fun getVideoHeight(): Int {
        return aliPlayer?.videoHeight ?: super.getVideoHeight()
    }

    override fun getDownloadBitrate(): Any? {
        return aliPlayer?.getOption(IPlayer.Option.DownloadBitrate)
    }

    override fun setSpeed(speed: Float) {
        aliPlayer?.speed = speed
    }

    override fun isPrepared(): Boolean {
        return isPreparer
    }

    override fun getCurrentPosition(): Long {
        return currentPosition
    }

    override fun setScaleMode(scaleMode: ScaleMode) {
        when (scaleMode) {
            ScaleMode.SCALE_ASPECT_FILL -> {
                aliPlayer?.scaleMode = IPlayer.ScaleMode.SCALE_ASPECT_FILL
            }

            ScaleMode.SCALE_TO_FILL -> {
                aliPlayer?.scaleMode = IPlayer.ScaleMode.SCALE_TO_FILL
            }

            ScaleMode.SCALE_ASPECT_FIT -> {
                aliPlayer?.scaleMode = IPlayer.ScaleMode.SCALE_ASPECT_FIT
            }
        }

    }

    override fun currentMediaSource(): MediaSource? {
        return mediaSource
    }

    override fun setAutoPlay(isAutoPlay: Boolean) {

    }

    override fun enableHardwareDecoder(enable: Boolean) {
        aliPlayer?.enableHardwareDecoder(enable)
    }

    override fun setErrorInterceptor(listener: TnErrorInterceptorListener) {
        errorInterceptorListener = listener
    }

    private fun handleDecodeError(info: PlayError): Boolean {
        //视频，音频解码失败
        if (info.errorCode != 537133057 && info.errorCode != 537133058) {
            return false
        }
        if (isInnerChangeDecoder) {
            //已经切过软解，还是出错不再处理
            ORPlayerLog.d(TAG, "aliyun 已经切过软解，还是出错不再处理", writeToFile = true)
            return false
        }
        ORPlayerLog.e(TAG, "aliyun onError -- 解码失败，强制切到软解", true)
        onPlayerListener?.onAliyunDecodeErrorChangeSoftwareDecoder(mediaSource)
        listeners.forEach { listener ->
            listener.onAliyunDecodeErrorChangeSoftwareDecoder(mediaSource)
        }
        //重置
        stop()
        reset()
        aliPlayer?.enableHardwareDecoder(false)
        mediaSource?.let { mediaSource ->
            setDataSource(mediaSource)
            prepare()
        }
        isInnerChangeDecoder = true
        return true
    }

}