package com.transsion.player.mediasession

import android.content.ComponentName
import android.support.v4.media.MediaBrowserCompat
import android.support.v4.media.MediaMetadataCompat
import android.support.v4.media.session.MediaControllerCompat
import android.support.v4.media.session.PlaybackStateCompat
import android.util.Log
import androidx.core.os.bundleOf
import com.blankj.utilcode.util.ThreadUtils
import com.blankj.utilcode.util.Utils
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch

/**
 * @author xinbing.zhang
 * @date :2023/9/25 10:28
 * @description: MediaBrowserCompat（客户端）: --> 初始化 --> 获取到MediaControllerCompat（控制器）--> 通过MediaControllerCompat，我们可以实现在Activity中控制音乐。
 */
class MediaBrowserCompatHelper {

    companion object {
        val instance by lazy {
            MediaBrowserCompatHelper()
        }
    }


    // =============================================================================================


    private fun getClassTag(): String = javaClass.simpleName

    /**
     * 浏览器订阅的接口，数据的回调
     */
    private val subscriptionCallback: MediaBrowserCompat.SubscriptionCallback =
        object : MediaBrowserCompat.SubscriptionCallback() {
            override fun onChildrenLoaded(
                parentId: String, children: List<MediaBrowserCompat.MediaItem>
            ) {
                super.onChildrenLoaded(parentId, children)
                // 从MediaService发送回来列表的数据
                // children 即为Service发送回来的媒体数据集合
                // 在onChildrenLoaded可以执行刷新列表UI的操作
                SessionLogger.log("mediaSource --> subscriptionCallback --> onChildrenLoaded() --> parentId = $parentId -- children = $children")
            }
        }

    /**
     * 通过MediaControllerCompat，我们可以拿到MediaControllerCompat.TransportControls对象，
     * 该对象可以控制播放音乐的一些常规操作：
     * 除此之外，我们还可以通过MediaControllerCompat来实现MediaControllerCompat.Callback接口，
     *
     *
     * 该接口的作用是获取音乐播放的状态改变，从而控制我们UI界面的更新，例如进度条，播放信息
     */
    private val mediaControllerCallback: MediaControllerCompat.Callback =
        object : MediaControllerCompat.Callback() {
            override fun onPlaybackStateChanged(state: PlaybackStateCompat?) {
                super.onPlaybackStateChanged(state)
                // 这里根据播放状态的改变，本地ui做相应的改变，例如播放模式，播放、暂停，进度条等
                //SessionLogger.log("mediaSource --> mediaControllerCallback --> onPlaybackStateChanged() --> state = $state")
            }

            override fun onMetadataChanged(metadata: MediaMetadataCompat) {
                super.onMetadataChanged(metadata)
                // 歌曲的信息，例如播放时长，歌曲名称等
                //SessionLogger.log("mediaSource --> mediaControllerCallback --> onMetadataChanged() -- metadata = $metadata")
            }
        }

    /**
     * 客户端注册状态回调 --> 注册成功之后就可以获取到 MediaControllerCompat 控制器
     *
     *
     * 同时通过MediaBrowserCompat.ConnectionCallback（上面的代码片中的mConnectionCallback）接口来实现和服务端连接的回调，
     * 在回调中，我们可以通过返回的MediaSessionCompat.Token获取到MediaControllerCompat（控制器）
     * ，通过MediaControllerCompat，我们可以实现在Activity中控制音乐。
     */
    private val connectionCallback: MediaBrowserCompat.ConnectionCallback =
        object : MediaBrowserCompat.ConnectionCallback() {
            override fun onConnected() {
                super.onConnected()
                //SessionLogger.log("mediaSource --> connectionCallback --> onConnected() --> 表示已经连接")
                try {
                    //表示已经连接
                    //val mediaId =
                    mediaBrowser?.root?.let {
                        mediaBrowser?.unsubscribe(it)
                        // 订阅消息 MediaBrowserCompat订阅：
                        // MediaBrowserCompat连接上MediaBrowserServiceCompat之后，
                        // 我们可以通过MediaBrowserCompat向MediaBrowserServiceCompat发起订阅请求，
                        // 例如需要获取某一列表的数据，或者某一首歌的数据，这时就需要
                        // 做如下操作：
                        // 需要先解除订阅，再发起订阅（这好像是官方的一个Bug，必须这么做）
                        mediaBrowser?.subscribe(it, subscriptionCallback)
                    }

                    // MediaControllerCompat（控制器）：
                    mediaController =
                        MediaControllerCompat(Utils.getApp(), mediaBrowser!!.sessionToken)
                    mediaController?.registerCallback(mediaControllerCallback)
                    initCallbacks.forEach {
                        it.invoke(mediaController?.transportControls)
                    }
                    initCallbacks.clear()
                } catch (e: Exception) {
                    e.printStackTrace()
                    SessionLogger.log(
                        "mediaSource --> connectionCallback --> e = " + Log.getStackTraceString(
                            e
                        )
                    )
                } finally {
                    isConnecting = false
                }
            }

            override fun onConnectionFailed() {
                super.onConnectionFailed()
                isConnecting = false
            }

            override fun onConnectionSuspended() {
                super.onConnectionSuspended()
            }
        }

    /**
     * MediaBrowserCompat（客户端）:
     */
    private var mediaBrowser: MediaBrowserCompat? = null

    /**
     * 媒体播放控制器
     */
    private var mediaController: MediaControllerCompat? = null

    private var initCallbacks = mutableListOf<(MediaControllerCompat.TransportControls?) -> Unit>()

    private var isConnecting = false


    // =============================================================================================


    /**
     * 初始化 客户端
     *
     *  开启服务
     *  val intent = Intent(it, MediaService::class.java)
     *  it.startService(intent)
     */
    fun init(callback: ((transportControls: MediaControllerCompat.TransportControls?) -> Unit)? = null) {
        //SessionLogger.log("mediaSource --> 初始化了")

        // 如果当前 已经连接了那就不重复初始化了
        runCatching {
            if (mediaBrowser?.isConnected == true) {
                callback?.invoke(mediaController?.transportControls)
                //SessionLogger.log("mediaSource --> 已经连接了那就不重复初始化了")
                return
            }


            if (isConnecting) {
                callback?.let {
                    initCallbacks.add(it)
                }
                //SessionLogger.log("mediaSource --> 正在连接中等待")
                return
            }
            isConnecting = true

            // 初始化
            Utils.getApp()?.let {
                // MediaBrowserCompat（客户端）:
                // 声明MediaBrowserCompat（客户端），通过MediaBrowserCompat来和MediaBrowserServiceCompat（服务端）连接
                if (ThreadUtils.isMainThread()) {//增加线程判断，确保在主线程初始化
                    callback?.let { callback ->
                        initCallbacks.add(callback)
                    }
                    mediaBrowser = MediaBrowserCompat(
                        it, ComponentName(it, MediaService::class.java), connectionCallback, null
                    )
                    mediaBrowser?.connect()
                } else {
                    CoroutineScope(Dispatchers.Main).launch {
                        callback?.let { callback ->
                            initCallbacks.add(callback)
                        }
                        mediaBrowser = MediaBrowserCompat(
                            it, ComponentName(it, MediaService::class.java), connectionCallback, null
                        )
                        mediaBrowser?.connect()
                    }
                }
            }
        }
    }

    fun removeNotification(mediaId: String) {
//        if (mMediaItem == null || mMediaItem?.mediaId != mediaId) {
//            return
//        }
//        ORPlayerMiddle.getORPlayer().release()
        //NotificationUtil.removeNotification()
        //MediaNotification.removeNotification()
    }

    fun destroy() {
        mediaBrowser?.disconnect()
    }


    // ===================== 播放器操作之后通知控制器刷新通知 ============================================


    /**
     * 刷新通知
     */
    private fun requestTransportControls(callback: ((transportControls: MediaControllerCompat.TransportControls?) -> Unit)? = null) {
        init(callback)
    }

    /**
     * 播放
     */
    fun play(mediaItem: MediaItem?) {
        //SessionLogger.log("${getClassTag()} --> play() --> 刷新通知 --> mediaItem = $mediaItem")
        requestTransportControls {
            it?.sendCustomAction(
                MediaMediaSessionCompatCallBack.ACTION_PLAY, bundleOf(
                    Pair(MediaMediaSessionCompatCallBack.KEY_MEDIAITEM, mediaItem)
                )
            )
        }
    }

    /**
     * 暂停
     */
    fun pause(mediaItem: MediaItem?) {
        //SessionLogger.log("${getClassTag()} --> pause() --> 刷新通知 --> mediaItem = $mediaItem")
        requestTransportControls {
            it?.sendCustomAction(
                MediaMediaSessionCompatCallBack.ACTION_PAUSE, bundleOf(
                    Pair(MediaMediaSessionCompatCallBack.KEY_MEDIAITEM, mediaItem)
                )
            )
        }
    }

    /**
     * 进度条
     */
    fun seekTo(pos: Long) {
        SessionLogger.log("${getClassTag()} --> seekTo() --> 刷新通知 --> pos = $pos")
        requestTransportControls {
            it?.seekTo(pos)
        }
    }

    fun skipToNext() {
        requestTransportControls {
            it?.skipToNext()
        }
    }

    fun skipToPrevious() {
        requestTransportControls {
            it?.skipToPrevious()
        }
    }

    fun stop() {
        requestTransportControls {
            it?.stop()
        }
    }

}