package com.transsion.player.shorttv.preload

import android.content.Context
import android.net.Uri
import android.os.Handler
import android.os.HandlerThread
import android.os.Message
import android.util.Log
import androidx.media3.common.util.UnstableApi
import androidx.media3.exoplayer.offline.Download
import androidx.media3.exoplayer.offline.DownloadManager
import androidx.media3.exoplayer.offline.DownloadRequest
import androidx.media3.exoplayer.offline.DownloadService
import androidx.media3.exoplayer.scheduler.Requirements
import com.blankj.utilcode.util.Utils
import com.transsion.player.MediaSource
import com.transsion.player.exo.DemoDownloadService
import com.transsion.player.exo.DemoUtil
import com.transsion.player.utils.MD5
import com.transsion.player.utils.ORPlayerLog
import java.util.concurrent.ConcurrentHashMap

@UnstableApi
object VideoPreloadManager {

    private const val TAG = "VideoPreloadUtil"
    private val activeTasks: ConcurrentHashMap<String, VideoDownloadBean> = ConcurrentHashMap()

    private val context: Context by lazy { Utils.getApp() }

    private val videoPreloadThread: HandlerThread = HandlerThread(TAG)
    private val handler: Handler

    private val updateInterval = 100L
    private val updateIntervalRunnable = {

        for (currentDownload in DemoUtil.getDownloadManager(context).currentDownloads) {
            if (currentDownload.state == Download.STATE_DOWNLOADING) {
                Log.d(
                    TAG, "progress 111:" +
                            "  download.requestId:${currentDownload.request.id}   " +
                            "  bytesDownloaded:${currentDownload.bytesDownloaded}" +
                            "  contentLength:${currentDownload.contentLength}" +
                            "  percentDownloaded:${currentDownload.percentDownloaded}"
                )
                activeTasks[currentDownload.request.id]?.apply {
                    downloadLength = currentDownload.bytesDownloaded
                    contentLength = currentDownload.contentLength

                    if (maxLength in 1 until downloadLength) {
                        removePreload(request.id)
                    }
                }
            }

        }
        if (isStartLoop) {
            loop()
        }
    }

    private val listener = object : DownloadManager.Listener {

        override fun onInitialized(downloadManager: DownloadManager) {
        }

        override fun onDownloadChanged(
            downloadManager: DownloadManager,
            download: Download,
            finalException: Exception?
        ) {
            Log.e(
                TAG, "onDownloadChanged:" +
                        "  download.requestId:${download.request.id}   " +
                        "  bytesDownloaded:${download.bytesDownloaded}" +
                        "  contentLength:${download.contentLength}" +
                        "  percentDownloaded:${download.percentDownloaded}"
            )
            activeTasks[download.request.id]?.apply {
                downloadLength = download.bytesDownloaded
                contentLength = download.contentLength
                if (download.state == Download.STATE_DOWNLOADING) {
                    if (maxLength in 1 until downloadLength) {
                        stopPreload(request.id)
                    }
                } else if (download.state == Download.STATE_STOPPED || download.state == Download.STATE_COMPLETED) {
                    removePreload(request.id)
                }
            }
            startLoop()
        }

        override fun onIdle(downloadManager: DownloadManager) {
            Log.e(TAG, "onIdle:")
            stopLoop()
        }

        override fun onDownloadsPausedChanged(
            downloadManager: DownloadManager,
            downloadsPaused: Boolean
        ) {
            Log.e(TAG, "onDownloadsPausedChanged:")
        }

        override fun onDownloadRemoved(downloadManager: DownloadManager, download: Download) {
            Log.e(TAG, "onDownloadRemoved:")
        }

        override fun onRequirementsStateChanged(
            downloadManager: DownloadManager,
            requirements: Requirements,
            notMetRequirements: Int
        ) {
            Log.e(TAG, "onRequirementsStateChanged:")
        }

        override fun onWaitingForRequirementsChanged(
            downloadManager: DownloadManager,
            waitingForRequirements: Boolean
        ) {
            Log.e(TAG, "onWaitingForRequirementsChanged:")
        }
    }

    init {
        videoPreloadThread.start()
        handler = object : Handler(videoPreloadThread.looper) {
            override fun dispatchMessage(msg: Message) {
                try {
                    super.dispatchMessage(msg)
                } catch (e: Exception) {
                    e.printStackTrace()
                }
            }
        }
        DemoUtil.getDownloadManager(context).addListener(listener)
    }

    private var isStartLoop = false
    private fun startLoop() {
        if (isStartLoop) return
        isStartLoop = true
        loop()
    }

    private fun stopLoop() {
        isStartLoop = false
        handler.removeCallbacks(updateIntervalRunnable)
    }


    private fun loop() {
        handler.removeCallbacks(updateIntervalRunnable)
        handler.postDelayed(updateIntervalRunnable, updateInterval)
    }

    /**
     * 预计加载视频
     * @param shortTVItem 视频缓存的 key 和视频的 url
     * @return 返回下载的一个信息
     */
    public fun preloadVideo(shortTVItem: MediaSource) {
        runOnPreloadThread {
            val shortTVDownloadBean = createDownloadRequest(shortTVItem)
            preloadVideo(shortTVDownloadBean)
        }
    }

    /**
     * 预计加载视频
     * @param shortTVItem 视频缓存的 key 和视频的 url
     * @return 返回下载的一个信息
     */
    public fun preloadVideoAsync(shortTVItem: MediaSource): VideoDownloadBean {
        val shortTVDownloadBean = createDownloadRequest(shortTVItem)
        preloadVideo(shortTVDownloadBean)
        return shortTVDownloadBean
    }

    public fun preloadVideo(shortTVDownloadBean: VideoDownloadBean): VideoDownloadBean {
        runOnPreloadThread {
            Log.e(
                TAG,
                "preloadVideo id:${shortTVDownloadBean.id} downloadLength:${shortTVDownloadBean.downloadLength}  maxLength:${shortTVDownloadBean.maxLength} contentLength:${shortTVDownloadBean.contentLength}"
            )
            if (shortTVDownloadBean.maxLength == -1L || shortTVDownloadBean.downloadLength < shortTVDownloadBean.maxLength) {
                if (shortTVDownloadBean.isAdd) {
                    DownloadService.sendSetStopReason(
                        context,
                        DemoDownloadService::class.java,
                        shortTVDownloadBean.id,
                        Download.STOP_REASON_NONE,
                        false
                    )
                    Log.e(TAG, "checkToStart restart  shortTVItem:$shortTVDownloadBean")
                } else {
                    DownloadService.sendAddDownload(
                        context,
                        DemoDownloadService::class.java,
                        shortTVDownloadBean.request,
                        false
                    )
                    Log.e(TAG, "checkToStart add    shortTVItem:$shortTVDownloadBean")
                }
            }
        }
        return shortTVDownloadBean
    }

    public fun stopPreload(key: String) {
        runOnPreloadThread {
            ORPlayerLog.e(TAG, "stopPreload   key:$key")
            DownloadService.sendSetStopReason(
                context,
                DemoDownloadService::class.java,
                key,
                VideoPreloadHelper.PAUSE_REASON,
                false
            )
        }
    }

    public fun removePreload(key: String) {
        activeTasks.remove(key)
        runOnPreloadThread {
            ORPlayerLog.e(TAG, "removePreload   key:$key")
            DownloadService.sendRemoveDownload(
                context,
                DemoDownloadService::class.java,
                key,
                false
            )
        }
    }

    public fun stopALlPreload() {
        ORPlayerLog.e(TAG, "stopALlPreload  ")
        DownloadService.sendSetStopReason(
            context,
            DemoDownloadService::class.java,
            null,
            VideoPreloadHelper.PAUSE_REASON,
            false
        )
    }

    public fun createDownloadRequest(shortTVItem: MediaSource): VideoDownloadBean {
        val data = activeTasks[shortTVItem.key]
        if (null != data) {
            ORPlayerLog.d(TAG, "createDownloadRequest [from map]  shortTVItem:$shortTVItem")
            return data
        }
        val currentDownload =
            DemoUtil.getDownloadManager(Utils.getApp()).downloadIndex.getDownload(shortTVItem.key)
        if (currentDownload?.request?.id == shortTVItem.key) {
            ORPlayerLog.d(
                TAG,
                "createDownloadRequest [from currentDownloads]   shortTVItem:$shortTVItem"
            )
            return VideoDownloadBean(
                shortTVItem.key,
                shortTVItem.url,
                currentDownload.request,
                currentDownload.contentLength,
                currentDownload.bytesDownloaded,
                VideoPreloadHelper.PRELOAD_FIRST_SIZE,
                true
            )
        }

        val cacheKey = shortTVItem.key
        val url = shortTVItem.url
        val shortTVDownloadBean = VideoDownloadBean(
            cacheKey,
            url,
            DownloadRequest.Builder(cacheKey, Uri.parse(url))
                .setCustomCacheKey(cacheKey)
                .build(),
            -1,
            0,
            VideoPreloadHelper.PRELOAD_FIRST_SIZE,
            false
        )
        activeTasks[cacheKey] = shortTVDownloadBean
        ORPlayerLog.d(TAG, "createDownloadRequest [from new]  shortTVItem:$shortTVItem")
        return shortTVDownloadBean

    }

    fun isCache(url: String?): Boolean {
        return if (url.isNullOrEmpty()) {
            false
        } else if (url.startsWith("http")) {
            DemoUtil.getDownloadCache(Utils.getApp()).isCached(urlToCacheKey(url), 0L, 200L)
        } else {
            true
        }
    }


    fun urlToCacheKey(url: String?): String {
        val index = url?.indexOf("?") ?: -1
        val key = if (null != url && index >= 0) {
            MD5.getStringMD5(url.split("?")[0])
        } else {
            MD5.getStringMD5(url ?: "")
        }
        return key
    }

    fun runOnPreloadThread(runnable: Runnable) {
        handler.post(runnable)
    }

}
