package com.transsion.player.orplayer

import com.aliyun.loader.MediaLoader
import com.aliyun.player.AliPlayerGlobalSettings
import com.blankj.utilcode.util.Utils
import com.transsion.player.utils.ORPlayerLog
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import java.io.File
import java.util.concurrent.CopyOnWriteArrayList

/**
 * @FileName:    视频预加载管理类
 * @Author:      kun.xu
 * @Date:        2022/10/19 16:32
 * @Description: 使用阿里云播放器的[MediaLoader]加载，前提是[AliPlayerGlobalSettings]开启本地缓存
 * 23-5-5 改为单任务下载，当前预加载一个视频，确保当前视频播放流畅
 */
class ORPlayerPreloadManager : MediaLoader.OnLoadStatusListener {

    private val TAG = "ORPlayerPreload"
    private val DEF_DURATION = 4000L//预加载时长
    private var preloadDuration = DEF_DURATION//预加载时长
    private var curUrl = ""//当前预加载
    private val urlsPaused = CopyOnWriteArrayList<String>()//暂停列表
    private val urlsPreloaded = CopyOnWriteArrayList<String>()//预加载成功的列表

    private val urlsCache = CopyOnWriteArrayList<String>()//当前预加载等待列表-队列预加载
    private var lastPreloadTime = 0L

    companion object {
        val instance by lazy { ORPlayerPreloadManager() }
    }

    init {
        MediaLoader.getInstance().setOnLoadStatusListener(this)
        val path = Utils.getApp().externalCacheDir?.absolutePath + File.separator + "video_cache"
        runCatching {
            AliPlayerGlobalSettings.enableLocalCache(true, 0, path)
        }
    }

    fun setPreloadDuration(duration: Long){
        preloadDuration = duration
    }

    fun preload(url: String) {
        preload(url, DEF_DURATION)
    }

    fun preload(url: String, duration: Long) {
        if (duration > 0) {
            preloadDuration = duration
        }
        if (isPreloading(url)) {
            ORPlayerLog.d(TAG, "start preload, contains return~ url = $url")
            return
        }
        if (urlsPaused.contains(url)) {
            ORPlayerLog.d(TAG, "start preload, pause contains resume~ url = $url")
            if (curUrl.isNotEmpty()) {
                urlsCache.add(url)
            } else {
                curUrl = url
                MediaLoader.getInstance().resume(url)
            }
            urlsPaused.remove(url)
            return
        }
        if (urlsPreloaded.contains(url)) {
            ORPlayerLog.d(TAG, "is preloaded, return, url = $url")
            return
        }
        if (curUrl.isNotEmpty()) {
            if (System.currentTimeMillis() - lastPreloadTime > 5000) {//上一个预加载已经超过5秒还没回调，则预加载新的
                ORPlayerLog.v(TAG, "last is more than 5 sec, preload new url")
                if (urlsCache.isNotEmpty()) {
                    curUrl = urlsCache.first()
                    startPreload(curUrl)
                    urlsCache.remove(curUrl)
                    urlsCache.add(url)
                } else {
                    curUrl = url
                    startPreload(url)
                }
            } else {
                ORPlayerLog.d(TAG, "start preload, add to cache, url = $url")
                urlsCache.add(url)
            }
        } else {
            curUrl = url
            startPreload(url)
        }
    }

    fun pause(url: String) {
        if (!isPreloading(url)) {
            ORPlayerLog.d(TAG, "pause, but not contains, PausedContains =  ${urlsPaused.contains(url)}, url = $url")
            return
        }
        ORPlayerLog.d(TAG, "pause, url = $url")
        urlsPaused.add(url)
        if (curUrl == url){//当前正在下载
            curUrl = ""
            MediaLoader.getInstance().pause(url)
            if (urlsCache.isNotEmpty()) {
                curUrl = urlsCache.first()
                startPreload(curUrl)
                urlsCache.remove(curUrl)
            }
        } else if (urlsCache.contains(url)){
            urlsCache.remove(url)
        }

    }

    fun pauseAll() {
        ORPlayerLog.d(TAG, "pauseAll, paused size = ${urlsPaused.size}, urlsCache size = ${urlsCache.size}, curUrl = $curUrl ")
        if (curUrl.isNotEmpty()) {
            MediaLoader.getInstance().pause(curUrl)
            urlsPaused.add(curUrl)
            curUrl = ""
        }
        if (urlsCache.isNotEmpty()) {
            urlsPaused.addAll(urlsCache)
            urlsCache.clear()
        }
    }

    fun resumeAll() {
        ORPlayerLog.d(TAG, "resumeAll, paused size = ${urlsPaused.size}, urlsCache size = ${urlsCache.size}, curUrl = $curUrl ")
        if (curUrl.isNotEmpty()) {
            return
        }
        if (urlsPaused.isNotEmpty()) {
            for (paused in urlsPaused) {
                if (curUrl.isNotEmpty()) {
                    urlsCache.add(paused)
                } else {
                    curUrl = paused
                    MediaLoader.getInstance().resume(paused)
                }
            }
        }
        urlsPaused.clear()
    }

    fun cancelAll() {
        ORPlayerLog.d(TAG, "cancelAll")
        curUrl = ""
        urlsCache.clear()
        urlsPaused.clear()
        CoroutineScope(Dispatchers.IO).launch {
            MediaLoader.getInstance().cancel("")
        }
    }

    fun isPreloading(url: String): Boolean {
        return curUrl.isNotEmpty() && url == curUrl || urlsCache.contains(url)
    }

    fun isPreloaded(url: String): Boolean {
        return urlsPreloaded.contains(url)
    }

    override fun onError(url: String?, errorCode: Int, errorMsg: String?) {

        ORPlayerLog.d(
            TAG,
            "preload error, ${if (errorCode == -300) "is preloaded code" else "errorCode"} = $errorCode, errorMsg= $errorMsg, url = $url"
        )
        //失败不做自动重试，需外部再次调用预加载, code 为300则为已预加载成功
        if (urlsCache.isNotEmpty()) {
            curUrl = urlsCache.first()
            startPreload(curUrl)
            urlsCache.remove(curUrl)
        }
        urlsPaused.remove(url)
        if (errorCode == -300){
            urlsPreloaded.add(url)
        }
    }

    override fun onCompleted(url: String?) {
        ORPlayerLog.v(TAG, "preload complete, url = $url")
        urlsPaused.remove(url)
        if (urlsCache.isNotEmpty()) {
            curUrl = urlsCache.first()
            startPreload(curUrl)
            urlsCache.remove(curUrl)
        } else {
            curUrl = ""
        }
        url?.let { urlsPreloaded.add(it) }
    }

    private fun startPreload(url: String) {
        lastPreloadTime = System.currentTimeMillis()
        ORPlayerLog.d(TAG, "startPreload, url = $url")
        MediaLoader.getInstance().load(url, preloadDuration)
    }

    override fun onCanceled(url: String?) {
    }


}