package com.transsion.player.shorttv

import android.content.Context
import android.os.Build
import android.os.Handler
import android.os.Looper
import android.view.SurfaceView
import android.view.TextureView
import androidx.media3.common.AudioAttributes
import androidx.media3.common.C
import androidx.media3.common.DeviceInfo
import androidx.media3.common.MediaItem
import androidx.media3.common.MediaMetadata
import androidx.media3.common.Metadata
import androidx.media3.common.MimeTypes
import androidx.media3.common.PlaybackException
import androidx.media3.common.PlaybackParameters
import androidx.media3.common.Player
import androidx.media3.common.Player.DiscontinuityReason
import androidx.media3.common.Player.PlayWhenReadyChangeReason
import androidx.media3.common.Player.PlaybackSuppressionReason
import androidx.media3.common.Player.TimelineChangeReason
import androidx.media3.common.Timeline
import androidx.media3.common.TrackSelectionParameters
import androidx.media3.common.Tracks
import androidx.media3.common.VideoSize
import androidx.media3.common.text.Cue
import androidx.media3.common.text.CueGroup
import androidx.media3.common.util.UnstableApi
import androidx.media3.exoplayer.ExoPlayer
import androidx.media3.exoplayer.analytics.AnalyticsListener
import androidx.media3.exoplayer.source.DefaultMediaSourceFactory
import androidx.media3.exoplayer.util.EventLogger
import com.blankj.utilcode.util.Utils
import com.transsion.player.MediaSource
import com.transsion.player.enum.ScaleMode
import com.transsion.player.exo.DemoUtil
import com.transsion.player.exo.ORExoDecoderType
import com.transsion.player.orplayer.AudioFocusHelper
import com.transsion.player.orplayer.AudioFocusPlay
import com.transsion.player.orplayer.IPlayerListener
import com.transsion.player.orplayer.PlayError
import com.transsion.player.shorttv.preload.DefaultConfig
import com.transsion.player.shorttv.preload.VideoPreloadConfig
import com.transsion.player.shorttv.preload.VideoPreloadHelper
import com.transsion.player.ui.render.RenderScaleMode
import com.transsion.player.ui.render.SurfaceRenderView
import com.transsion.player.ui.render.TextureRenderView
import com.transsion.player.utils.ORPlayerLog
import java.util.concurrent.CopyOnWriteArrayList


@UnstableApi
class ORExoShortTvPlayer(val config: VideoPreloadConfig = DefaultConfig()) : ORShortTvPlayer {

    private val TAG = "ShortTv-ORExoPlayer"
    private val mediaSourceMap: MutableMap<String, MediaSource> = hashMapOf()
    private var exoPlayer: ExoPlayer? = null

    private var mediaSource: MediaSource? = null
    private var surfaceView: SurfaceRenderView? = null
    private var textureView: TextureRenderView? = null
    private var volume: Float? = null
    private var isAutoPlay: Boolean = false
    private var isLoop: Boolean = false
    private var mute: Boolean = false
    private var speed: Float = 1f
    private var scaleMode: ScaleMode = ScaleMode.SCALE_TO_FILL

    private var videoBitrate = 0
    private var audioBitrate = 0

    /** 当前视频的编码类型,参考：[MimeTypes]*/
    private var curVideoMimeType: String? = null
    private var curDecoderType = ORExoDecoderType.HARDWARE

    private var audioFocusHelper: AudioFocusHelper? = null

    private fun initAudioFocus() {
        val audioFocusPlayer = object : AudioFocusPlay {
            override fun play() {
                exoPlayer?.play()
            }

            override fun pause() {
                exoPlayer?.pause()
            }

            override fun isMute(): Boolean {
                return exoPlayer?.isDeviceMuted ?: false
            }

            override fun setVolume(volume: Float) {
                exoPlayer?.volume = volume
            }
        }
        audioFocusHelper = AudioFocusHelper(audioFocusPlayer) { focusLoss ->
            listeners.forEach { listener ->
                listener.onFocusChange(focusLoss)
            }
        }
    }

    private val exoPlayerListener: Player.Listener = object : Player.Listener {
        override fun onTimelineChanged(timeline: Timeline, reason: @TimelineChangeReason Int) {}
        override fun onMediaMetadataChanged(mediaMetadata: MediaMetadata) {}
        override fun onPlaylistMetadataChanged(mediaMetadata: MediaMetadata) {}
        override fun onLoadingChanged(isLoading: Boolean) {}
        override fun onAvailableCommandsChanged(availableCommands: Player.Commands) {}
        override fun onTrackSelectionParametersChanged(parameters: TrackSelectionParameters) {}
        override fun onPlayerStateChanged(playWhenReady: Boolean, playbackState: @Player.State Int) {}
        override fun onPlayWhenReadyChanged(playWhenReady: Boolean, reason: @PlayWhenReadyChangeReason Int) {}
        override fun onPlaybackSuppressionReasonChanged(playbackSuppressionReason: @PlaybackSuppressionReason Int) {}
        override fun onRepeatModeChanged(repeatMode: @Player.RepeatMode Int) {}
        override fun onShuffleModeEnabledChanged(shuffleModeEnabled: Boolean) {}
        override fun onPlayerErrorChanged(error: PlaybackException?) {}
        override fun onPositionDiscontinuity(reason: @DiscontinuityReason Int) {}
        override fun onPlaybackParametersChanged(playbackParameters: PlaybackParameters) {}
        override fun onSeekBackIncrementChanged(seekBackIncrementMs: Long) {}
        override fun onSeekForwardIncrementChanged(seekForwardIncrementMs: Long) {}
        override fun onMaxSeekToPreviousPositionChanged(maxSeekToPreviousPositionMs: Long) {}
        override fun onAudioSessionIdChanged(audioSessionId: Int) {}
        override fun onAudioAttributesChanged(audioAttributes: AudioAttributes) {}
        override fun onVolumeChanged(volume: Float) {}
        override fun onDeviceInfoChanged(deviceInfo: DeviceInfo) {}
        override fun onDeviceVolumeChanged(volume: Int, muted: Boolean) {}
        override fun onSurfaceSizeChanged(width: Int, height: Int) {}
        override fun onCues(cues: MutableList<Cue>) {}
        override fun onCues(cueGroup: CueGroup) {}
        override fun onMetadata(metadata: Metadata) {}

        private var lastState = Player.STATE_IDLE
        override fun onPlaybackStateChanged(playbackState: Int) {
            ORPlayerLog.e(TAG, "onPlaybackStateChanged  playbackState:$playbackState ")
            when (playbackState) {
                ExoPlayer.STATE_READY -> {
                    onPrepare2GetBitrate()
                    listeners.forEach { listener ->
                        listener?.onPrepare()
                    }
                    if (lastState == Player.STATE_BUFFERING) {
                        listeners.forEach { listener ->
                            listener?.onLoadingEnd()
                        }
                    }
                }

                ExoPlayer.STATE_ENDED -> {
                    listeners.forEach { listener ->
                        listener?.onCompletion()
                    }
                }

                Player.STATE_BUFFERING -> {
                    ORPlayerLog.e(TAG, "onPlaybackStateChanged STATE_BUFFERING")
                    listeners.forEach { listener ->
                        listener?.onLoadingBegin()
                    }
                }

                Player.STATE_IDLE -> {
                }
            }
            lastState = playbackState
        }

        override fun onIsLoadingChanged(isLoading: Boolean) {
        }

        override fun onIsPlayingChanged(isPlaying: Boolean) {
            audioFocusHelper?.isPlaying = isPlaying
            if (isPlaying) {
                listeners.forEach { listener ->
                    listener?.onVideoStart()
                }
                handler.post(timerTask)
            } else {
                if (exoPlayer?.playWhenReady != true) {
                    listeners.forEach { listener ->
                        listener?.onVideoPause()
                    }
                }
                handler.removeCallbacks(timerTask)
            }
            ORPlayerLog.e(TAG, "onIsPlayingChanged:$isPlaying")
        }

        override fun onRenderedFirstFrame() {
            ORPlayerLog.e(TAG, "onRenderedFirstFrame:")
            listeners.forEach { listener ->
                listener?.onRenderFirstFrame()
            }
        }

        override fun onPlayerError(error: PlaybackException) {
            ORPlayerLog.e(TAG, "onPlayerError:$error")
            //短剧是列表提前设置到播放器的，如何切换播放器直接导致后续视频无法播放的，短剧视频格式较单一，这里移除切换
            //当前硬解播放失败，切换到软解播放
//            if (curDecoderType == ORExoDecoderType.HARDWARE) {
//                ORPlayerLog.d(TAG, "exo 硬解失败，切换到软解，MimeType:$curVideoMimeType, code:${error.errorCode},msg:${error.message}", true)
//                changeToSoftDecoder()?.let { mediaSource ->
//                    setDataSource(mediaSource)
//                }
//                return
//            }
            listeners.forEach { listener ->
                listener?.onPlayError(PlayError(error.errorCode, error.message))
            }
        }

        override fun onVideoSizeChanged(videoSize: VideoSize) {
            surfaceView?.setVideoRotation(videoSize.unappliedRotationDegrees)
            textureView?.setVideoRotation(videoSize.unappliedRotationDegrees)
            surfaceView?.setVideoSize(videoSize.width, videoSize.height)
            textureView?.setVideoSize(videoSize.width, videoSize.height)
            listeners.forEach { listener ->
                listener?.onVideoSizeChanged(videoSize.width, videoSize.height)
            }
            ORPlayerLog.e(TAG, "onVideoSizeChanged width:${videoSize.width}  height:${videoSize.height}")
        }

        override fun onEvents(player: Player, events: Player.Events) {
//            ORPlayerLog.e(TAG, "onEvents events:$events")
        }

        override fun onPositionDiscontinuity(
            oldPosition: Player.PositionInfo,
            newPosition: Player.PositionInfo,
            reason: Int
        ) {
            val currentWindowIndex: Int = newPosition.mediaItemIndex
            val currentPositionMs: Long = newPosition.positionMs
            ORPlayerLog.d(
                TAG, "onPositionDiscontinuity: currentWindowIndex=" + currentWindowIndex
                        + ", currentPositionMs=" + currentPositionMs
                        + ", reason=" + reason
            )
            if (reason == Player.DISCONTINUITY_REASON_SEEK || reason == Player.DISCONTINUITY_REASON_SEEK_ADJUSTMENT) {
                listeners.forEach { listener -> listener?.setOnSeekCompleteListener() }
            }
            listeners.forEach { listener -> listener?.onProgress(currentPositionMs) }
            videoPreloadHelper?.changeSelect(currentWindowIndex)
        }


        override fun onMediaItemTransition(mediaItem: MediaItem?, reason: Int) {
            ORPlayerLog.e(TAG, "onMediaItemTransition  reason:$reason ")
            if (reason == Player.MEDIA_ITEM_TRANSITION_REASON_AUTO || reason == Player.MEDIA_ITEM_TRANSITION_REASON_REPEAT) {
                listeners.forEach { listener -> listener?.onMediaItemTransition(mediaItem?.localConfiguration?.customCacheKey) }
            }
        }

    }

    private var listeners: CopyOnWriteArrayList<IPlayerListener> = CopyOnWriteArrayList()

    init {
        initAudioFocus()
        createPlayer(ORExoDecoderType.HARDWARE)
    }
    private fun createPlayer(extRendererType: ORExoDecoderType) {
        val context: Context = Utils.getApp()
        exoPlayer = ExoPlayer.Builder(context)
            .setMediaSourceFactory(
                DefaultMediaSourceFactory(context).setDataSourceFactory(
                    DemoUtil.getDataSourceFactory(
                        context
                    )
                )
            )
            .setRenderersFactory(DemoUtil.buildRenderersFactory(context, extRendererType))
            .build()
            .apply {
                playWhenReady = false
//                repeatMode = Player.REPEAT_MODE_ALL
                addListener(exoPlayerListener)
                addAnalyticsListener(object : EventLogger(TAG) {

                    override fun onTracksChanged(eventTime: AnalyticsListener.EventTime, tracks: Tracks) {
                        super.onTracksChanged(eventTime, tracks)
                        onExoTracksChanged(eventTime, tracks)
                    }

                    override fun onAudioDecoderInitialized(
                        eventTime: AnalyticsListener.EventTime,
                        decoderName: String,
                        initializedTimestampMs: Long,
                        initializationDurationMs: Long
                    ) {
                        super.onAudioDecoderInitialized(eventTime, decoderName, initializedTimestampMs, initializationDurationMs)
                        ORPlayerLog.v(TAG, "onExoAudioDecoderInitialize, decoderName:$decoderName", true)
                    }

                    override fun onVideoDecoderInitialized(
                        eventTime: AnalyticsListener.EventTime,
                        decoderName: String,
                        initializedTimestampMs: Long,
                        initializationDurationMs: Long
                    ) {
                        super.onVideoDecoderInitialized(eventTime, decoderName, initializedTimestampMs, initializationDurationMs)
                        ORPlayerLog.v(TAG, "onExoVideoDecoderInitialize, decoderName:$decoderName", true)
                    }

                })
            }
    }

    private fun onPrepare2GetBitrate() {
        exoPlayer?.currentTracks?.let { tracks ->
            for (groupIndex in tracks.groups.indices) {
                val trackGroup: Tracks.Group = tracks.groups[groupIndex]
                for (index in 0 until trackGroup.length) {
                    val format = trackGroup.getTrackFormat(index)
                    if (MimeTypes.isVideo(format.sampleMimeType)) {
                        if (exoPlayer?.videoFormat?.bitrate == format.bitrate) {
                            videoBitrate = format.bitrate
                            ORPlayerLog.d(TAG, "--onPrepare2GetBitrate  MimeType:${format.sampleMimeType}，videoBitrate：$videoBitrate")
                            listeners.forEach { listener ->
                                listener.onTracksVideoBitrateChange(videoBitrate)
                            }
                        }
                    } else if (MimeTypes.isAudio(format.sampleMimeType)) {
                        if (exoPlayer?.audioFormat?.bitrate == format.bitrate) {
                            audioBitrate = format.bitrate
                            ORPlayerLog.d(TAG, "--onPrepare2GetBitrate  MimeType:${format.sampleMimeType}，audioBitrate：$audioBitrate")
                            listeners.forEach { listener ->
                                listener.onTracksAudioBitrateChange(audioBitrate)
                            }
                        }
                    }
                }
            }
        }

    }

    private fun onExoTracksChanged(eventTime: AnalyticsListener.EventTime, tracks: Tracks) {
        //vp9视频-4.4以上，现在全机型都支持，无需扩展解码器
        //id=1, mimeType=video/x-vnd.on2.vp9, res=1920x1080, color=BT709/Limited range/SDR SMPTE 170M/8/8, fps=24.192997, supported=YES
        //id=2, mimeType=audio/mp4a-latm, bitrate=129157, codecs=mp4a.40.2, channels=2, sample_rate=24000, language=en, supported=YES

        //av1视频-10+支持，以下的版本需要扩展解码器，10+的无需扩展解码器
        //id=1, mimeType=video/av01, res=1920x1080, color=BT709/Limited range/SDR SMPTE 170M/8/8, fps=25.0, supported=NO_UNSUPPORTED_TYPE
        //id=2, mimeType=audio/mp4a-latm, bitrate=128000, codecs=mp4a.40.2, channels=2, sample_rate=44100, language=en, supported=YES

        //he265视频-看设备视频，硬解有些支持，有些不支持，需要动态切换
        //id=1, mimeType=video/hevc, codecs=hvc1.1.6.L90.90, res=860x360, color=NA/NA/NA/8/8, fps=23.97602, supported=YES
        //id=2, mimeType=audio/mp4a-latm, bitrate=62794, codecs=mp4a.40.2, channels=2, sample_rate=22050, language=und, supported=YES

        //10以下的av1视频，能播放成功不报错，但是只有音频(有onAudioDecoderInitialized回调)没有视频(没有onAudioDecoderInitialized回调)，
        // 所以需要判断超时，切换av1扩展编码器，
        // 如果失败那跟ffmpeg一样直接切换即可
        for (groupIndex in tracks.groups.indices) {
            val trackGroup: Tracks.Group = tracks.groups[groupIndex]
            for (index in 0 until trackGroup.length) {
                val format = trackGroup.getTrackFormat(index)
                if (MimeTypes.isVideo(format.sampleMimeType)) {
                    videoBitrate = format.bitrate
                } else if (MimeTypes.isAudio(format.sampleMimeType)) {
                    audioBitrate = format.bitrate
                }
                ORPlayerLog.v(TAG, "onTracksChanged  MimeType:${format.sampleMimeType}")
                if (format.sampleMimeType?.contains(MimeTypes.BASE_TYPE_VIDEO) == true) {
                    curVideoMimeType = format.sampleMimeType
                    ORPlayerLog.i(TAG, "--------onTracksChanged  curVideoMimeType:$curVideoMimeType")
                }
            }
        }
        if (curVideoMimeType == MimeTypes.VIDEO_AV1 && Build.VERSION.SDK_INT <= Build.VERSION_CODES.Q) {
            //针对<=10的，编码为av1的，直接强制切到软解播放
            ORPlayerLog.e(TAG, "exo av1，强制切换到软解， MimeType:$curVideoMimeType", true)
            changeToSoftDecoder()?.let { mediaSource ->
                setDataSource(mediaSource)
                prepare()
            }
        }
    }

    /**
     * 切换到软解
     * 需要重新创建新播放器
     */
    private fun changeToSoftDecoder(): MediaSource? {
        val curMediaSource = mediaSource
        reset()
        //切换编码，需要重新创建播放器，所以把之前的release掉
        release()
        curDecoderType = when (curVideoMimeType) {
            MimeTypes.VIDEO_AV1 -> {
                ORExoDecoderType.AV1
            }

            else -> {//vp9设备就支持，其他类型的用ffmpeg基本可以支持
                ORExoDecoderType.FFMPEG
            }
        }
        ORPlayerLog.d(TAG, "重新创建播放器切换编码, DecoderType:$curDecoderType,MimeTyp:$curVideoMimeType", true)
        createPlayer(curDecoderType)
        initNewPlayer()
        return curMediaSource
    }

    private fun initNewPlayer() {
        setMute(mute)
        volume?.let {
            setVolume(it)
        }
        setLooping(isLoop)
        setAutoPlay(isAutoPlay)
        setSpeed(speed)
        if (null != surfaceView) {
            setSurfaceView(surfaceView)
        }
        if (null != textureView) {
            setTextureView(textureView)
        }
        setScaleMode(scaleMode)
    }

    override fun isPrepared(): Boolean {
        return exoPlayer?.playbackState == Player.STATE_READY
    }

    override fun setSpeed(speed: Float) {
        this.speed = speed
        exoPlayer?.setPlaybackSpeed(speed)
    }

    override fun setSurfaceView(surfaceView: SurfaceView?) {
        this.surfaceView = surfaceView as? SurfaceRenderView
        exoPlayer?.setVideoSurfaceView(surfaceView)
    }

    override fun setTextureView(textureView: TextureView?) {
        this.textureView = textureView as? TextureRenderView
        exoPlayer?.setVideoTextureView(textureView)
    }

    override fun clearSurfaceOnly() {
        exoPlayer?.setVideoSurfaceView(null)
        exoPlayer?.setVideoTextureView(null)
        surfaceView?.holder?.surface?.release()
        textureView?.surfaceTexture?.release()
    }

    override fun setPlayerListener(listener: IPlayerListener) {
        addPlayerListener(listener)
    }

    override fun addPlayerListener(listener: IPlayerListener) {
        super.addPlayerListener(listener)
        ORPlayerLog.i(TAG, "addPlayerListener")
        if (!listeners.contains(listener)) {
            listeners.add(listener)
        }
    }

    override fun removePlayerListener(listener: IPlayerListener) {
        super.removePlayerListener(listener)
        ORPlayerLog.i(TAG, "removePlayerListener")
        listeners.remove(listener)
    }

    override fun setDataSource(mediaSource: MediaSource) {
        this.mediaSource = mediaSource
        val mediaItem = MediaItem.fromUri(mediaSource.url)
        exoPlayer?.setMediaItem(mediaItem)
        listeners.forEach { listener -> listener?.onSetDataSource() }
        ORPlayerLog.e(TAG, "setDataSource")
    }

    override fun prepare() {
        ORPlayerLog.e(TAG, "prepare")
        exoPlayer?.prepare()
    }

    override fun play() {
        ORPlayerLog.e(TAG, "play")
        if (exoPlayer?.playerError != null) {
            exoPlayer?.prepare()
        }
        exoPlayer?.play()
        audioFocusHelper?.isUserPause = false
        audioFocusHelper?.requestFocus()
    }

    override fun pause() {
        audioFocusHelper?.isUserPause = true
        ORPlayerLog.e(TAG, "pause")
        exoPlayer?.pause()
        audioFocusHelper?.abandonFocus()
    }

    override fun stop() {
        audioFocusHelper?.isUserPause = true
        ORPlayerLog.e(TAG, "stop    ")
        exoPlayer?.stop()
        audioFocusHelper?.abandonFocus()
    }

    override fun release() {
        ORPlayerLog.e(TAG, "release")
        exoPlayer?.release()
        listeners.forEach { listener ->
            listener?.onPlayerRelease()
        }
        listeners.clear()
        videoPreloadHelper?.release()
        mediaSource = null
        audioFocusHelper?.abandonFocus()
    }

    override fun reset() {
        ORPlayerLog.e(TAG, "reset    ")
        exoPlayer?.stop()
        listeners.forEach { listener ->
            listener?.onPlayerReset()
        }
    }

    override fun seekTo(mills: Long) {
        ORPlayerLog.e(TAG, "seekTo    mills:$mills")
        if (exoPlayer?.playerError != null) {
            exoPlayer?.prepare()
        }
        exoPlayer?.seekTo(mills)
    }

    override fun getDuration(): Long = exoPlayer?.duration ?: 0L

    override fun isPlaying(): Boolean {
        return exoPlayer?.isPlaying ?: false
    }

    override fun isComplete(): Boolean {
        return exoPlayer?.playbackState == Player.STATE_ENDED
    }

    override fun setLooping(isLoop: Boolean) {
        this.isLoop = isLoop
        exoPlayer?.repeatMode = if (isLoop) Player.REPEAT_MODE_ONE else Player.REPEAT_MODE_OFF
    }

    override fun isMute(): Boolean {
        return exoPlayer?.isDeviceMuted ?: false
    }

    override fun setMute(mute: Boolean) {
        this.mute = mute
        exoPlayer?.setDeviceMuted(mute, C.VOLUME_FLAG_SHOW_UI)
    }

    override fun setVolume(volume: Float) {
        this.volume = volume
        exoPlayer?.volume = volume
    }

    override fun getVolume(): Float? {
        return exoPlayer?.volume
    }

    override fun setAutoPlay(isAutoPlay: Boolean) {
        this.isAutoPlay = isAutoPlay
        exoPlayer?.playWhenReady = isAutoPlay
    }

    override fun getCurrentPosition(): Long {
        return exoPlayer?.currentPosition ?: 0L
    }

    override fun enableHardwareDecoder(enable: Boolean) {
        if (enable.not() && curDecoderType == ORExoDecoderType.HARDWARE) {
            changeToSoftDecoder()
        } else if (enable && curDecoderType != ORExoDecoderType.HARDWARE) {
            ORPlayerLog.d(TAG, "软解切换到硬解，重新创建设置", true)
            curDecoderType = ORExoDecoderType.HARDWARE
            createPlayer(curDecoderType)
            initNewPlayer()
        }
    }

    override fun addDataSource(mediaSource: MediaSource): Boolean {
        this.mediaSource = mediaSource
        val mediaItem = MediaItem.Builder()
            .setUri(mediaSource.url)
            .setMediaId(mediaSource.weights.toString())
            .setCustomCacheKey(mediaSource.key)
            .setTag(mediaSource.id)
            .build()
        val count = exoPlayer?.mediaItemCount ?: 0
        var findIndex = -1
        for (i in 0 until count) {
            val currentItem = exoPlayer?.getMediaItemAt(i)
            val currentWeight = currentItem?.mediaId?.toIntOrNull() ?: 0
            val vid = currentItem?.localConfiguration?.tag
            if (vid == mediaSource.id) {
                ORPlayerLog.e(TAG, "addDataSource 已经在列表里 index:$i vid:${mediaSource.id}")
                return false
            }
            if (findIndex < 0 && currentWeight > mediaSource.weights) {
                findIndex = i
            }
        }
        if (findIndex >= 0) {
            videoPreloadHelper?.add(findIndex, mediaSource)
            exoPlayer?.addMediaItem(findIndex, mediaItem)
            ORPlayerLog.e(TAG, "addDataSource  index:$findIndex key:${mediaSource.id}")
        } else {
            ORPlayerLog.e(TAG, "addDataSource  index:$count key:${mediaSource.id}")
            exoPlayer?.addMediaItem(mediaItem)
        }
        videoPreloadHelper?.add(mediaSource)
        mediaSourceMap[mediaSource.id ?: ""] = mediaSource
        return true
    }

    override fun addDataSource(mediaSource: List<MediaSource>): Boolean {
        var addResult = false
        mediaSource.forEach {
            if (addDataSource(it)) {
                addResult = true
            }
        }
        if (addResult) {
            videoPreloadHelper?.loadNext()
        }
        return addResult
    }

    override fun removeDataSource(mediaSource: MediaSource): Boolean {
        val count = exoPlayer?.mediaItemCount ?: 0
        var findIndex = -1
        for (i in 0 until count) {
            val currentItem = exoPlayer?.getMediaItemAt(i)
            val currentKey = currentItem?.localConfiguration?.tag
            if (currentKey == mediaSource.id) {
                ORPlayerLog.e(TAG, "removeDataSource 在列表里 index:$i key:${mediaSource.id}")
                findIndex = i
                break
            }
        }
        return if (findIndex < 0) {
            false
        } else {
            videoPreloadHelper?.remove(mediaSource.id ?: "")
            exoPlayer?.removeMediaItem(findIndex)
            mediaSourceMap.remove(mediaSource.id ?: "")
            true
        }
    }

    override fun getBitrate(): Pair<Int, Int> {
        return Pair(videoBitrate, audioBitrate)
    }

    override fun seekTo(uuid: String, mills: Long) {
        super.seekTo(uuid, mills)
        for (i in 0 until (exoPlayer?.mediaItemCount ?: 0)) {
            if (exoPlayer?.getMediaItemAt(i)?.localConfiguration?.tag == uuid) {
                if (exoPlayer?.playerError != null) {
                    prepare()
                }
                if (exoPlayer?.currentMediaItemIndex != i) {
                    ORPlayerLog.e(TAG, "seekTo  index:$i uuid:$uuid")
                    exoPlayer?.seekTo(i, mills)
                } else {
                    if (isComplete()) {
                        exoPlayer?.seekTo(mills)
                    }
                    ORPlayerLog.e(TAG, "seekTo   uuid:$uuid mills:$mills  是当前视频直接播放")
                }
                play()
                return
            }
        }
        ORPlayerLog.e(TAG, "seekTo   uuid:$uuid mills:$mills  没有找到")
        pause()
    }

    private val handler = Handler(Looper.myLooper()!!)
    private val timerTask = Runnable {
        if (exoPlayer?.isPlaying == true) {
            listeners.forEach {
                exoPlayer?.currentPosition?.let { currentPosition ->
                    it.onProgress(currentPosition, currentMediaSource())
                }
            }
            loop()
        }
    }

    private fun loop() {
        handler.removeCallbacks(timerTask)
        handler.postDelayed(timerTask, 500)
    }

    override fun setScaleMode(scaleMode: ScaleMode) {
        super.setScaleMode(scaleMode)
        this.scaleMode = scaleMode
        val renderScaleMode: RenderScaleMode = when (scaleMode) {
            ScaleMode.SCALE_TO_FILL -> {
                RenderScaleMode.SCREEN_SCALE_MATCH_PARENT
            }

            ScaleMode.SCALE_ASPECT_FIT -> {
                RenderScaleMode.SCREEN_SCALE_DEFAULT
            }

            ScaleMode.SCALE_ASPECT_FILL -> {
                RenderScaleMode.SCREEN_SCALE_CENTER_CROP
            }
        }
        textureView?.setScaleType(renderScaleMode)
        surfaceView?.setScaleType(renderScaleMode)
    }

    private val videoPreloadHelper: VideoPreloadHelper? by lazy {
        if (config.enable) {
            VideoPreloadHelper(Utils.getApp())
        } else {
            null
        }
    }


    override fun updateDataSource(mediaSource: MediaSource): Boolean {
        val mediaItem = MediaItem.Builder()
            .setUri(mediaSource.localPath ?: mediaSource.url)
            .setMediaId(mediaSource.weights.toString())
            .setCustomCacheKey(mediaSource.key)
            .setTag(mediaSource.id)
            .build()
        val count = exoPlayer?.mediaItemCount ?: 0
        for (i in 0 until count) {
            val currentItem = exoPlayer?.getMediaItemAt(i)
            val currentKey = currentItem?.localConfiguration?.tag
            if (currentKey == mediaSource.id) {
                ORPlayerLog.e(TAG, "updateDataSource 在列表里更新数据 index:$i key:${mediaSource.id}")
                exoPlayer?.replaceMediaItem(i, mediaItem)
                exoPlayer?.prepare()
                mediaSourceMap[mediaSource.id ?: ""] = mediaSource
                return true
            }
        }

        return false
    }

    override fun getDataSourceMap(): MutableMap<String, MediaSource> {
        return mediaSourceMap
    }


    override fun currentMediaSource(): MediaSource? {
        val vid =
            exoPlayer?.getMediaItemAt(exoPlayer?.currentMediaItemIndex ?: 0)?.localConfiguration?.tag?.toString()
        return mediaSourceMap[vid]
    }
}