package com.transsion.player.orplayer.global

import android.view.SurfaceView
import android.view.TextureView
import androidx.media3.common.util.UnstableApi
import com.blankj.utilcode.util.Utils
import com.transsion.player.MediaSource
import com.transsion.player.config.VodConfig
import com.transsion.player.enum.ScaleMode
import com.transsion.player.exo.ORExoPlayer
import com.transsion.player.orplayer.IPlayerListener
import com.transsion.player.orplayer.ORPlayer
import com.transsion.player.orplayer.PlayError
import com.transsion.player.tracks.TnFormat
import com.transsion.player.tracks.TnTracks
import com.transsion.player.tracks.TnTracksGroup
import com.transsion.player.utils.ORPlayerLog
import java.util.concurrent.CopyOnWriteArrayList

@UnstableApi
class TnPlayer : ORPlayer {
    companion object {
        const val TAG = "TnPlayerProxy"

    }

    @Volatile
    var mOrPlayer: ORPlayer? = null

    @Volatile
    private var mediaSource: MediaSource? = null


    private var surfaceView: SurfaceView? = null
    private var textureView: TextureView? = null
    private var listener: IPlayerListener? = null
    private var listeners: CopyOnWriteArrayList<IPlayerListener> = CopyOnWriteArrayList()

    private var volume: Float? = null
    private var isAutoPlay: Boolean = false
    private var isLoop: Boolean = false
    private var vodConfig: VodConfig? = null
    private var mute: Boolean = false
    private var enableHardwareDecoder: Boolean = true
    private var speed: Float = 1f
    private var scaleMode: ScaleMode = ScaleMode.SCALE_ASPECT_FIT

    private var isPlayerChanged = false
    private var curProgress = 0L

    private val playerListenerIntercept = PlayerListenerIntercept()

    override fun setSurfaceView(surfaceView: SurfaceView?) {
        this.textureView = null
        this.surfaceView = surfaceView
        mOrPlayer?.setSurfaceView(surfaceView)
    }

    override fun setTextureView(textureView: TextureView?) {
        this.surfaceView = null
        this.textureView = textureView
        mOrPlayer?.setTextureView(textureView)
    }

    override fun clearSurfaceOnly() {
        mOrPlayer?.clearSurfaceOnly()
        surfaceView = null
        textureView = null
    }

    override fun getBitrate() = mOrPlayer?.getBitrate() ?: Pair(0, 0)

    override fun setPlayerListener(listener: IPlayerListener) {
        this.listener = listener
        mOrPlayer?.setPlayerListener(listener)
    }

    override fun addPlayerListener(listener: IPlayerListener) {
//        if (listeners.contains(listener).not()) {
//            listeners.add(listener)
//            mOrPlayer?.addPlayerListener(listener)
//        }
        playerListenerIntercept.addPlayerListener(listener)
    }

    override fun removePlayerListener(listener: IPlayerListener) {
//        listeners.remove(listener)
//        mOrPlayer?.removePlayerListener(listener)
        playerListenerIntercept.removePlayerListener(listener)
    }


    override fun setDataSource(mediaSource: MediaSource) {
        TnPlayerManager.requestFocus(this, "setDataSource")
        this.mediaSource = mediaSource
        checkPlayer()
        // 中转一下
        mOrPlayer?.addPlayerListener(playerListenerIntercept)
        mOrPlayer?.setDataSource(mediaSource)
    }


    override fun prepare() {
        mOrPlayer?.prepare()
    }

    override fun play() {
        TnPlayerManager.requestFocus(this, "play")
        TnPlayerManager.play(this, mediaSource)
        mOrPlayer?.play()
    }

    override fun pause() {
        TnPlayerManager.pause(this, mediaSource)
        mOrPlayer?.pause()
    }

    override fun stop() {
        resetErrorInterceptor()
        mOrPlayer?.stop()
    }

    override fun release(id: String?) {
        release()
    }

    override fun release() {
        resetErrorInterceptor()
        mOrPlayer?.release()
        this.mOrPlayer = null
        this.listener = null
        this.surfaceView = null
        this.textureView = null
        TnPlayerManager.releaseFocus(this)
    }

    override fun reset() {
        resetErrorInterceptor()
        mOrPlayer?.reset()
    }

    override fun clearScreen() {
        mOrPlayer?.clearScreen()
    }

    override fun seekTo(mills: Long) {
        mOrPlayer?.seekTo(mills)
        mediaSource?.mediaItem?.position= mills
        TnPlayerManager.seekTo(this, mediaSource)
    }

    override fun seekTo(uuid: String, mills: Long) {
        mOrPlayer?.seekTo(uuid, mills)
    }

    override fun getDuration(): Long {
        return mOrPlayer?.getDuration() ?: 0
    }

    override fun isPlaying(): Boolean {
        return mOrPlayer?.isPlaying() ?: false
    }

    override fun isComplete(): Boolean {
        return mOrPlayer?.isComplete() ?: false
    }

    override fun setLooping(isLoop: Boolean) {
        this.isLoop = isLoop
        mOrPlayer?.setLooping(isLoop)
    }

    override fun isMute(): Boolean {
        return mOrPlayer?.isMute() ?: false
    }

    override fun setMute(mute: Boolean) {
        this.mute = mute
        mOrPlayer?.setMute(mute)
    }

    override fun isLoading(): Boolean {
        return mOrPlayer?.isLoading() ?: super.isLoading()
    }

    override fun getVideoWidth(): Int {
        return mOrPlayer?.getVideoWidth() ?: super.getVideoWidth()
    }

    override fun getVideoHeight(): Int {
        return mOrPlayer?.getVideoHeight() ?: super.getVideoHeight()
    }

    override fun isPrepared(): Boolean {
        return mOrPlayer?.isPrepared() ?: super.isPrepared()
    }

    override fun setScaleMode(scaleMode: ScaleMode) {
        this.scaleMode = scaleMode
        mOrPlayer?.setScaleMode(scaleMode)
    }

    override fun setVolume(volume: Float) {
        this.volume = volume
        mOrPlayer?.setVolume(volume)
    }

    override fun getVolume(): Float? {
        return mOrPlayer?.getVolume()
    }

    override fun setAutoPlay(isAutoPlay: Boolean) {
        this.isAutoPlay = isAutoPlay
        mOrPlayer?.setAutoPlay(isAutoPlay)
    }

    override fun enableHardwareDecoder(enable: Boolean) {
        this.enableHardwareDecoder = enable
        mOrPlayer?.enableHardwareDecoder(enable)
    }

    override fun getDownloadBitrate(): Any? {
        return mOrPlayer?.getDownloadBitrate()
    }

    override fun setPlayerConfig(vodConfig: VodConfig) {
        this.vodConfig = vodConfig
        mOrPlayer?.setPlayerConfig(vodConfig)
    }


    override fun setSpeed(speed: Float) {
        this.speed = speed
        mOrPlayer?.setSpeed(speed)
    }

    override fun getCurrentTracks(): TnTracks? {
        return mOrPlayer?.getCurrentTracks()
    }

    override fun getCurrentPosition(): Long {
        return mOrPlayer?.getCurrentPosition() ?: 0L
    }

    override fun getCurrentVideoFormat(): TnFormat? {
        return mOrPlayer?.getCurrentVideoFormat()
    }

    override fun changeTrackSelection(mediaTrackGroup: TnTracksGroup, index: Int) {
        mOrPlayer?.changeTrackSelection(mediaTrackGroup, index)
    }

    fun onLossFocus(focusLoss: Boolean) {
        listener?.onFocusChange(focusLoss)
        listeners.forEach { listener ->
            listener.onFocusChange(focusLoss)
        }
    }


    private fun checkPlayer() {
        mediaSource?.also {
            resetErrorInterceptor()
            if (useExo(it)) {
                if (mOrPlayer !is ORExoPlayer) {
                    mOrPlayer?.release()
                    mOrPlayer = TnPlayerPool.newExoPlayer(Utils.getApp()).apply {
                        initNewPlayer(this)
                    }
                }
            } else {
                if (mOrPlayer !is TnAliPlayer) {
                    mOrPlayer?.release()
                    mOrPlayer = TnPlayerPool.newAliPlayer(Utils.getApp()).apply {
                        initNewPlayer(this)
                    }
                }
            }
        }
    }

    private fun useExo(mediaSource: MediaSource): Boolean {
        return getPlayerType() == 1 || mediaSource.isStream || mediaSource.forceExo
    }


    /**
     * simba配置开关，播放器类型，int 方便以后扩展
     * 0-阿里云，默认值
     * 1-exo
     */
    private fun getPlayerType(): Int {
        return TnPlayerManager.isUsePlayerType
    }

    private fun initNewPlayer(player: ORPlayer?) {
        if (null == player) return
        player.setMute(mute)
        vodConfig?.let {
            player.setPlayerConfig(it)
        }
        volume?.let {
            player.setVolume(it)
        }
        player.setLooping(isLoop)
        player.setAutoPlay(isAutoPlay)
        player.setSpeed(speed)
        player.enableHardwareDecoder(enableHardwareDecoder)
        if (null != surfaceView) {
            player.setSurfaceView(surfaceView)
        }
        if (null != textureView) {
            player.setTextureView(textureView)
        }
        listener?.let {
            player.setPlayerListener(it)
        }
        listeners.forEach { listener ->
            player.addPlayerListener(listener)
        }
//        setErrorInterceptor(player)
        player.setScaleMode(scaleMode)
    }

    override fun currentMediaSource(): MediaSource? {
        return mediaSource
    }

    private fun setErrorInterceptor(player: ORPlayer?) {
        ORPlayerLog.d(TAG, "setErrorInterceptor--- $player")
        player?.setErrorInterceptor(object : TnErrorInterceptorListener {
            override fun onPlayError(playerType: TnPlayerType, errorInfo: PlayError, mediaSource: MediaSource?): Boolean {
                if (isPlayerChanged) {
                    ORPlayerLog.i(TAG, "setErrorInterceptor is changed")
                    return false
                }
                if (playerType != TnPlayerType.ALIYUN && mediaSource?.isStream == true) {
                    //dash流媒体无法切到阿里云
                    ORPlayerLog.w(TAG, "setErrorInterceptor dash video--", true)
                    return false
                }
                mOrPlayer?.release()
                isPlayerChanged = true
                curProgress = player.getCurrentPosition()
                listener?.onPlayErrorChangePayer(playerType, this@TnPlayer.mediaSource)
                listeners.forEach { listener ->
                    listener.onPlayErrorChangePayer(playerType, this@TnPlayer.mediaSource)
                }
                when (playerType) {
                    TnPlayerType.EXO -> {
                        ORPlayerLog.d(TAG, "setErrorInterceptor change player, exo 2 aliyun", true)
                        mOrPlayer = TnPlayerPool.newAliPlayer(Utils.getApp()).apply {
                            enableHardwareDecoder = false
                            initNewPlayer(this)
                            this@TnPlayer.mediaSource?.let { source ->
                                setDataSource(source)
                                prepare()
                            }
                        }
                    }

                    TnPlayerType.ALIYUN -> {
                        ORPlayerLog.d(TAG, "setErrorInterceptor change player, aliyun 2 exo", true)
                        mOrPlayer = TnPlayerPool.newExoPlayer(Utils.getApp()).apply {
                            enableHardwareDecoder = false
                            initNewPlayer(this)
                            this@TnPlayer.mediaSource?.let { source ->
                                this.setDataSource(source)
                                prepare()
                            }
                        }
                    }
                }
                return true
            }

            override fun onPrepare(mediaSource: MediaSource?): Boolean {
                //视频准备好了，设置进度
                if (isPlayerChanged && curProgress > 0) {
                    mOrPlayer?.seekTo(curProgress)
                    mOrPlayer?.play()
                    return true
                }
                return false
            }
        })
    }

    private fun resetErrorInterceptor() {
        isPlayerChanged = false
        curProgress = 0L
    }
}