package com.transsion.player.mediasession

import android.app.Notification
import android.app.NotificationChannel
import android.app.NotificationManager
import android.app.PendingIntent
import android.content.Context
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.graphics.drawable.Drawable
import android.media.session.PlaybackState
import android.os.Build
import android.support.v4.media.session.MediaSessionCompat
import android.support.v4.media.session.PlaybackStateCompat
import android.text.TextUtils
import androidx.core.app.NotificationCompat
import androidx.core.app.ServiceCompat
import androidx.media.MediaBrowserServiceCompat
import androidx.media.session.MediaButtonReceiver
import com.blankj.utilcode.util.AppUtils
import com.blankj.utilcode.util.SizeUtils
import com.blankj.utilcode.util.Utils
import com.bumptech.glide.Glide
import com.bumptech.glide.load.engine.DiskCacheStrategy
import com.bumptech.glide.request.target.CustomTarget
import com.bumptech.glide.request.transition.Transition
import com.transsion.player.R
import com.transsion.player.utils.Resizer


/**
 * @author: zhangxinbing
 * @date : 2024/10/16 16:30
 * @description: 系统的媒体通知
 */
object MediaNotification {

    const val PAGE_NAME = "media_notification"
    private fun getClassTag(): String = javaClass.simpleName

    /**
     * Android 13及以上的手机无论是否设置按钮，只要setPlaybackState设置的时候包含了上下一首按钮的信息就会展示
     */
    private const val DEFAULT_OPEN_PREV_NEXT_BUTTON = false

    /**
     * Android 8.0之后需要处理的通知渠道
     */
    private const val CHANNEL_ID = "MUSIC_NOTIFICATION_ID"

    /**
     * 是否开启了前台服务标志
     */
    var isCreate: Boolean = false

    /**
     * 通知ID
     */
    private const val NOTIFICATION_ID = 89757

    /**
     * 处理系统媒体通知栏
     */
    private var mMediaSession: MediaSessionCompat? = null
    private var mediaService: MediaService? = null
    private var mMediaItem: MediaItem? = null
    private var mState: Int? = null


    // =============================================================================================


    /**
     * 初始化 通知的 渠道
     */
    fun initNotificationChannel() {
        SessionLogger.log("${getClassTag()} --> initNotificationChannel() --> 初始化 通知的 渠道")
        val manager = Utils.getApp()
            .getSystemService(MediaBrowserServiceCompat.NOTIFICATION_SERVICE) as NotificationManager
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            val channel = NotificationChannel(
                CHANNEL_ID,
                Utils.getApp().getString(R.string.player_notification_name),
                NotificationManager.IMPORTANCE_DEFAULT
            )
            channel.setSound(null, null)
            channel.description = "playing media"
            manager.createNotificationChannel(channel)
        }

        MusicNotificationBroadcastReceiver.register()
    }

    /**
     * 适配12.0及以上
     */
    fun getFlag(): Int = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
        PendingIntent.FLAG_IMMUTABLE
    } else {
        PendingIntent.FLAG_UPDATE_CURRENT
    }

    /**
     * 移除媒体通知
     */
    fun removeNotification() {
        mediaService?.let {
            ServiceCompat.stopForeground(it, ServiceCompat.STOP_FOREGROUND_REMOVE)
        }
        val manager = Utils.getApp()
            .getSystemService(MediaBrowserServiceCompat.NOTIFICATION_SERVICE) as NotificationManager
        manager.cancel(NOTIFICATION_ID)

        // 重置数据
        // 删除正在播放的视频，常驻通知自动消失，然后再播放任意一个视频，通知栏常驻消失可以手动滑动删除掉，正确应该不可以删除
        isCreate = false
    }

    /**
     * 展示通知
     */
    fun notifyNotification(
        service: MediaService?,
        mediaItem: MediaItem?,
        state: Int?,
        mediaSession: MediaSessionCompat?
    ) {

        // 目前只有 Music才有设置
        if (mediaItem == null) {
            SessionLogger.logE("${getClassTag()} --> notifyNotification() --> mediaItem == null --> return")
            removeNotification()
            return
        }

        innerNotify(service, mediaItem, state, mediaSession)
    }


    // =============================================================================================


    private fun innerNotify(
        service: MediaService?,
        mediaItem: MediaItem?,
        state: Int?,
        mediaSession: MediaSessionCompat?
    ) {

        // 本地存一份
        mediaService = service
        mMediaItem = mediaItem
        mState = state
        mMediaSession = mediaSession

        // 加载封面图片
        loadAlbum()

        val context: Context = Utils.getApp()
        val builder =
            NotificationCompat.Builder(context, CHANNEL_ID)
                .setContentTitle(mediaItem?.title ?: "VideoPlayer") // 标题
                .setContentText(mediaItem?.subTitle ?: "VideoPlayer") // 子标题
        //.setSubText(description.description) // 描述 暂时没有这个字段
        //.setLargeIcon(description.iconBitmap) // 封面大图设置

        // 封面大图设置
        if (mCover == null) {
            builder.setLargeIcon(
                BitmapFactory.decodeResource(
                    Utils.getApp().resources, R.mipmap.player_ic_push_small_logo
                )
            )
        } else {
            builder.setLargeIcon(
                mCover
            )
        }

        // 通知删除事件 -- 通知播放器暂停播放
        builder.setDeleteIntent(MusicNotificationBroadcastReceiver.getDeleteIntent())

        // 通知点击跳转到播放详情页
        //builder.setContentIntent(MusicNotificationBroadcastReceiver.getContentIntent(mMediaItem))
        if (mediaItem?.pendingIntent != null) {
            builder.setContentIntent(mediaItem.pendingIntent)
        }

        // 这个方法用于设置通知的可见性等级，它决定了在锁屏界面上通知的显示方式。可见性等级有三个：
        //      VISIBILITY_PUBLIC：通知的全部内容在锁屏界面上可见。
        //      VISIBILITY_PRIVATE：通知的基本信息在锁屏界面上可见，但隐藏敏感内容。
        //      VISIBILITY_SECRET：通知在锁屏界面上完全隐藏。
        builder.setVisibility(NotificationCompat.VISIBILITY_PUBLIC)

        if (DEFAULT_OPEN_PREV_NEXT_BUTTON) {
            // 上一首按钮
            builder.addAction(
                NotificationCompat.Action.Builder(
                    R.mipmap.prev, "prev", MediaButtonReceiver.buildMediaButtonPendingIntent(
                        context, PlaybackStateCompat.ACTION_SKIP_TO_PREVIOUS
                    )
                ).build()
            )
        }

        // 播放暂停按钮
        if (state == PlaybackState.STATE_PLAYING) {
            builder.addAction(
                NotificationCompat.Action.Builder(
                    R.mipmap.player_pause,
                    "pause",
                    MediaButtonReceiver.buildMediaButtonPendingIntent(
                        context, PlaybackStateCompat.ACTION_PAUSE
                    )
                ).build()
            )
        } else {
            builder.addAction(
                NotificationCompat.Action.Builder(
                    R.mipmap.player_play, "play", MediaButtonReceiver.buildMediaButtonPendingIntent(
                        context, PlaybackStateCompat.ACTION_PLAY
                    )
                ).build()
            )
        }

        if (DEFAULT_OPEN_PREV_NEXT_BUTTON) {
            // 下一首按钮
            builder.addAction(
                NotificationCompat.Action.Builder(
                    R.mipmap.player_next, "next", MediaButtonReceiver.buildMediaButtonPendingIntent(
                        context, PlaybackStateCompat.ACTION_SKIP_TO_NEXT
                    )
                ).build()
            )
        }

        // 样式设置
        if (DEFAULT_OPEN_PREV_NEXT_BUTTON) {
            builder.setStyle(
                androidx.media.app.NotificationCompat.MediaStyle()
                    .setMediaSession(mMediaSession?.sessionToken)
                    .setShowActionsInCompactView(0, 1, 2)
            )
        } else {
            builder.setStyle(
                androidx.media.app.NotificationCompat.MediaStyle()
                    .setMediaSession(mMediaSession?.sessionToken).setShowActionsInCompactView(0)
            )
        }

        // 小图标设置
        builder.setSmallIcon(R.mipmap.player_ic_push_small_logo)

        val notification = try {
            builder.build()
        } catch (e: Throwable) {
            return
        }
        notification.flags =
            Notification.FLAG_NO_CLEAR or Notification.FLAG_FOREGROUND_SERVICE or Notification.FLAG_HIGH_PRIORITY
        if (isCreate) {
            try {
                val manager =
                    service?.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager
                manager.notify(NOTIFICATION_ID, notification)
            } catch (e: Throwable) {
                e.printStackTrace()
            }
        } else {
            // App 不再前台就不发送通知
            if (AppUtils.isAppForeground()) {
                try {
                    service?.startForeground(NOTIFICATION_ID, notification)
                } catch (e: Throwable) {
                    e.printStackTrace()
                    try {
                        val manager =
                            service?.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager
                        manager.notify(NOTIFICATION_ID, notification)
                    } catch (e: Throwable) {
                        e.printStackTrace()
                    }
                }
                isCreate = true
            } else {
                SessionLogger.logE("Service is restricted, cannot start foreground.")
            }
        }
    }


    // ============================= 下面是封面加载 ===================================================


    /**
     * 封面图片 地址
     */
    private var mCover: Bitmap? = null
    private var lastCoverUrl: String? = ""

    /**
     * 加载封面图片
     */
    private fun loadAlbum() {

        // 通过地址判断是否是同一张图片
        val coverUrl = mMediaItem?.coverUrl
        if (TextUtils.isEmpty(coverUrl)) {
            return
        }
        if (TextUtils.equals(lastCoverUrl, coverUrl)) {
            // 如果是同一张图，那就不重复加载了
            return
        }
        lastCoverUrl = coverUrl

        // 处理加载本地磁盘图片
        if (lastCoverUrl?.contains("http") == true) {
            mCover = null
            // 开始加载图片
            Glide.with(Utils.getApp()).asBitmap().centerInside()
                .diskCacheStrategy(DiskCacheStrategy.DATA)
                .load(getResizeUrl(mMediaItem?.coverUrl ?: "")).into(target)
        } else {
            kotlin.runCatching {
                mCover = getBitmap(lastCoverUrl)
            }.getOrElse {
                mCover = BitmapFactory.decodeResource(
                    Utils.getApp().resources, R.mipmap.player_ic_push_small_logo
                )
            }
            innerNotify(mediaService, mMediaItem, mState, mMediaSession)
        }
    }

    private val target = object : CustomTarget<Bitmap?>() {

        override fun onResourceReady(
            resource: Bitmap, transition: Transition<in Bitmap?>?
        ) {
            if (TextUtils.equals(lastCoverUrl, mMediaItem?.coverUrl)) {
                mCover = resource
                notifyNotification(mediaService, mMediaItem, mState, mMediaSession)
            }
        }

        override fun onLoadCleared(placeholder: Drawable?) {
            SessionLogger.log("${getClassTag()} --> loadAlbum --> onLoadCleared() ---> 加载失败")
        }

        override fun onLoadFailed(errorDrawable: Drawable?) {
            super.onLoadFailed(errorDrawable)
            SessionLogger.log("${getClassTag()} --> loadAlbum --> onLoadFailed() ---> 加载失败")
            mCover = BitmapFactory.decodeResource(
                Utils.getApp().resources, R.mipmap.player_ic_notification_icon
            )
            notifyNotification(mediaService, mMediaItem, mState, mMediaSession)
        }
    }

    private fun getResizeUrl(url: String): String {
        val width = SizeUtils.dp2px(48f)
        return Resizer.getReSizeUrl(url, width, true)
    }

    /**
     * 通过本地路径加载 bitmap
     */
    private fun getBitmap(path: String?): Bitmap? {
        if (TextUtils.isEmpty(path)) {
            return null
        }
        return resizedBitmap(path)
    }

    private fun resizedBitmap(imagePath: String?): Bitmap? {
        if (TextUtils.isEmpty(imagePath)) {
            return null
        }
        val targetWidth = SizeUtils.dp2px(48f)
        try {
            // Step 1: 获取图片的原始宽高
            val options = BitmapFactory.Options()
            options.inJustDecodeBounds = true // 只加载图片的宽高信息
            BitmapFactory.decodeFile(imagePath, options)

            val originalWidth = options.outWidth
            val originalHeight = options.outHeight

            // Step 2: 判断是否需要处理图片
            if (originalWidth <= targetWidth && originalHeight <= targetWidth) {
                // 如果图片分辨率已经低于目标分辨率，直接加载原图并返回
                options.inJustDecodeBounds = false
                return BitmapFactory.decodeFile(imagePath, options)
            }

            // Step 3: 如果图片分辨率高于目标分辨率，按比例缩小
            var inSampleSize = 1
            if (originalHeight > targetWidth || originalWidth > targetWidth) {
                val heightRatio = originalHeight.toFloat() / targetWidth
                val widthRatio = originalWidth.toFloat() / targetWidth
                inSampleSize = Math.round(heightRatio.coerceAtLeast(widthRatio)) // 取最大缩放比例
            }

            // Step 4: 按缩放比例加载图片
            options.inSampleSize = inSampleSize
            options.inJustDecodeBounds = false
            return BitmapFactory.decodeFile(imagePath, options)
        } catch (e: Throwable) {
            e.printStackTrace()
            return null // 如果加载失败，返回 null
        }
    }


}