package com.transsion.player.orplayer

import android.content.Context
import android.media.AudioAttributes
import android.media.AudioFocusRequest
import android.media.AudioManager
import android.os.Build
import com.blankj.utilcode.util.ThreadUtils
import com.blankj.utilcode.util.Utils
import com.transsion.player.utils.ORPlayerLog

/**
 * @FileName:    音频焦点管理类
 * @Author:      kun.xu
 * @Date:        2022/11/17 10:36
 * @Description:
 */
class AudioFocusHelper(private val player: AudioFocusPlay, private val callback: (focusLoss: Boolean) -> Unit) : AudioManager.OnAudioFocusChangeListener {
    private var TAG = "AudioFocusHelper"
    private val mAudioManager: AudioManager = Utils.getApp().applicationContext.getSystemService(Context.AUDIO_SERVICE) as AudioManager

    private var mStartRequested = false
    var isUserPause = false
    var isPlaying = false
    private var focusRequest: AudioFocusRequest? = null

    init {
        ORPlayerLog.d(TAG, "AudioFocusHelper orPlayer $this")
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            focusRequest = AudioFocusRequest.Builder(AudioManager.AUDIOFOCUS_GAIN).run {
                setAudioAttributes(AudioAttributes.Builder().run {
                    setUsage(AudioAttributes.USAGE_GAME)
                    setContentType(AudioAttributes.CONTENT_TYPE_MUSIC)
                    build()
                })
                setAcceptsDelayedFocusGain(true)
                setOnAudioFocusChangeListener(this@AudioFocusHelper)
                build()
            }
        }
    }

    override fun onAudioFocusChange(focusChange: Int) {
        //由于onAudioFocusChange有可能在子线程调用，故通过此方式切换到主线程去执行
        ThreadUtils.runOnUiThread { handleAudioFocusChange(focusChange) }
    }

    private fun handleAudioFocusChange(focusChange: Int) {
        ORPlayerLog.d(TAG, "handleAudioFocusChange ------------ $this, focusChange = $focusChange")
        when (focusChange) {
            AudioManager.AUDIOFOCUS_GAIN, AudioManager.AUDIOFOCUS_GAIN_TRANSIENT -> {
                ORPlayerLog.d(TAG, "handleAudioFocusChange GAIN, focusChange = $focusChange， mStartRequested = $mStartRequested, isUserPause = $isUserPause")
                callback(false)
                if (mStartRequested && isUserPause.not()) {
                    player.play()
                }
                if (!player.isMute()) {//恢复音量
                    player.setVolume(1.0f)
                }
            }

            AudioManager.AUDIOFOCUS_LOSS, AudioManager.AUDIOFOCUS_LOSS_TRANSIENT -> {
                ORPlayerLog.v(TAG, "handleAudioFocusChange LOSS, focusChange = $focusChange, isUserPause = $isUserPause")
                callback(true)
                if (isPlaying) {
                    player.pause()
                }
            }
            //暂时失去AudioFocus，但是可以继续播放，不过要在降低音量
            AudioManager.AUDIOFOCUS_LOSS_TRANSIENT_CAN_DUCK -> if (isPlaying && !player.isMute()) {
                ORPlayerLog.d(TAG, "handleAudioFocusChange AUDIOFOCUS_LOSS_TRANSIENT_CAN_DUCK, focusChange = $focusChange")
                player.setVolume(1.0f)
            }
        }
    }

    fun requestFocus() {
        ORPlayerLog.d(TAG, "requestFocus $this")
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            focusRequest?.let {
                //0-失败，1-获取，2-延迟
                val result = mAudioManager.requestAudioFocus(it)
                ORPlayerLog.d(TAG, "requestFocus, result:$result")
            }
        } else {
            //0-失败，1-获取，2-延迟
            val result = mAudioManager.requestAudioFocus(this, AudioManager.STREAM_MUSIC, AudioManager.AUDIOFOCUS_GAIN)
            ORPlayerLog.d(TAG, "requestFocus, result:$result")
        }
        mStartRequested = true
    }

    fun abandonFocus() {
        ORPlayerLog.d(TAG, "abandonFocus $this")
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            focusRequest?.let { mAudioManager.abandonAudioFocusRequest(it) }
        } else {
            mAudioManager.abandonAudioFocus(this)
        }
        mStartRequested = false
    }
}