package com.transsion.player.shorttv.preload

import android.content.Context
import android.util.Log
import androidx.media3.common.util.UnstableApi
import androidx.media3.exoplayer.offline.Download
import androidx.media3.exoplayer.offline.DownloadManager
import androidx.media3.exoplayer.scheduler.Requirements
//import com.tn.lib.util.networkinfo.NetworkUtil
import com.transsion.player.MediaSource
import com.transsion.player.exo.DemoUtil
import java.util.concurrent.ConcurrentHashMap
import java.util.concurrent.CopyOnWriteArrayList

@UnstableApi
class VideoPreloadHelper(private val context: Context) {
    companion object {
        const val PRELOAD_COUNT = 5
        const val PRELOAD_FIRST_SIZE = 300 * 1024L
        const val PAUSE_REASON = 10086
        const val PRELOAD_ALL = false

        const val TAG = "VideoPreloadHelper"
    }

    private val shortTVItemList: MutableList<MediaSource?> = CopyOnWriteArrayList()
    private var currentPosition = 0
    private var currentDownloadingUuid: String? = null

    private val listener = object : DownloadManager.Listener {
        override fun onInitialized(downloadManager: DownloadManager) {
        }

        override fun onDownloadsPausedChanged(downloadManager: DownloadManager, downloadsPaused: Boolean) {
        }
        override fun onDownloadChanged(
            downloadManager: DownloadManager,
            download: Download,
            finalException: Exception?
        ) {
            Log.e(
                TAG, "onDownloadChanged:" +
                        "  download.requestId:${download.request.id}   " +
                        "  bytesDownloaded:${download.bytesDownloaded}" +
                        "  contentLength:${download.contentLength}" +
                        "  percentDownloaded:${download.percentDownloaded}"
            )
            activeTasks[download.request.id]?.apply {
                downloadLength = download.bytesDownloaded
                contentLength = download.contentLength
            }
            if (download.request.id == currentDownloadingUuid
                && (download.state == Download.STATE_STOPPED || download.state == Download.STATE_COMPLETED || download.state == Download.STATE_FAILED)
            ) {
                currentDownloadingUuid = null
                loadNext()
            }
        }

        override fun onDownloadRemoved(downloadManager: DownloadManager, download: Download) {
        }
        override fun onIdle(downloadManager: DownloadManager) {
            Log.e(TAG, "onIdle:")
            loadNext()
        }

        override fun onRequirementsStateChanged(downloadManager: DownloadManager, requirements: Requirements, notMetRequirements: Int) {
        }

        override fun onWaitingForRequirementsChanged(downloadManager: DownloadManager, waitingForRequirements: Boolean) {
        }
    }


    private val activeTasks: ConcurrentHashMap<String, VideoDownloadBean> = ConcurrentHashMap()


    init {
        DemoUtil.getDownloadManager(context).addListener(listener)
    }

    fun changeSelect(position: Int) {
        Log.e(TAG, "changeSelect  position:$position")
        if (currentPosition == position) return
        currentDownloadingUuid = null
        currentPosition = position
        checkToPause(currentPosition)
        checkToStart(currentPosition + 1)
    }

    fun pause() {
        checkToPause(currentPosition)
        currentPosition = -1
    }

    fun resume(position: Int) {
        checkToStart(position)
    }

    fun remove(key: String) {
        activeTasks.remove(key)
        VideoPreloadManager.removePreload(key)
    }

    fun loadNext() {
        val position = if (currentPosition < 0) return else currentPosition
        if (null != currentDownloadingUuid) return
        VideoPreloadManager.runOnPreloadThread {
            val size = shortTVItemList.size
            var skipCount = 1
            var isCurrentComplete = false
            Log.e(TAG, "loadNext， position:$position   size:$size")
            //从下一个视频开始预加载，当前视频由播放器处理，避免同时下载
            val start = position + 1
            for (index in start until (start + PRELOAD_COUNT).coerceAtMost(size)) {
                val shortTVItem = shortTVItemList[index] ?: continue
                createDownloadRequest(index, shortTVItem).let {
                    it.maxLength = PRELOAD_FIRST_SIZE
//                if (index == position) {
//                    if (it.downloadLength == it.contentLength) {
//                        Log.e(TAG, "当前播放的已经下载完成， index:$index")
//                        isCurrentComplete = true
//                        skipCount++
//                    } else {
//                        Log.e(TAG, "当前播放的未下载完成，优先下载完整视频 index:$index")
//                        it.maxLength = -1
//                        checkToStart(index)
//                        //todo 这里可以优先保证当前的视频下载完才预计加载下来的，
////                        return
//                    }
//                } else {
//                if (it.contentLength > 0) {
//                    it.maxLength = it.contentLength.coerceAtMost(it.maxLength)
//                }
                    if (it.downloadLength < it.maxLength) {
                        Log.e(
                            TAG,
                            "非当前选中  下载大小 downloadLength:${it.downloadLength}  index:$index"
                        )
//                        if (PRELOAD_ALL && skipCount >= PRELOAD_COUNT) {
//                            Log.e(
//                                TAG,
//                                "非当前选中  下载大小 downloadLength:${it.downloadLength}  index:$index  *** 是最后一个要预加载的，直接全量下载"
//                            )
//                            it.maxLength = -1
//                        }
                        currentDownloadingUuid = it.id
                        VideoPreloadManager.preloadVideo(it)
                        return@runOnPreloadThread
                    } else {
                        if (it.downloadLength == it.contentLength) {
                            skipCount++
                        }
                        Log.e(
                            TAG,
                            "非当前选中  下载大小 downloadLength:${it.downloadLength} 大于最小下载，跳过 index:$index"
                        )
                    }
                }
//            }
            }

            if (!PRELOAD_ALL) {
                Log.e(TAG, "不用全量下载")
                return@runOnPreloadThread
            }
            if (!isCurrentComplete) {
                Log.e(
                    TAG,
                    "检查一圈下来，没有需求最小下载视频，开始全量下载? 当前的还没有下载完，优先下载当前的视频再考虑全量预加载"
                )
                return@runOnPreloadThread
            }

            Log.e(TAG, "检查一圈下来，没有需求最小下载视频，开始全量下载")
            for (index in position + 1 until (position + PRELOAD_COUNT).coerceAtMost(size)) {
                val shortTVItem = shortTVItemList[index] ?: continue
                createDownloadRequest(index, shortTVItem).let {
                    if (it.contentLength <= 0 || it.downloadLength < it.contentLength) {
                        Log.e(
                            TAG,
                            "非当前选中  下载大小 downloadLength:${it.downloadLength} 开始全量下载 index:$index"
                        )
                        it.maxLength = -1
                        VideoPreloadManager.preloadVideo(it)
                        return@runOnPreloadThread
                    } else {
                        Log.e(TAG, "非当前选中，已完整下载，跳过 index:$index")
                    }
                }
            }
            Log.e(TAG, "检查一圈下来，没有需求全量下载视频，预加载完成")
        }
    }

    private fun createDownloadRequest(position: Int, shortTVItem: MediaSource): VideoDownloadBean {
        val data = activeTasks[shortTVItem.key]
        if (null != data) {
            Log.e(
                TAG,
                "createDownloadRequest [from map] position:$position  shortTVItem:$shortTVItem"
            )
            return data
        }
        val cacheKey = shortTVItem.key
        val shortTVDownloadBean = VideoPreloadManager.createDownloadRequest(shortTVItem)
        activeTasks[cacheKey] = shortTVDownloadBean
        Log.e(TAG, "createDownloadRequest [from new] position:$position  shortTVItem:$shortTVItem")
        return shortTVDownloadBean

    }

    private fun checkToPause(position: Int) {
        if (position in 0 until shortTVItemList.size) {
            val shortTVItem = shortTVItemList[position] ?: return
            Log.e(TAG, "checkToPause   position:$position  shortTVItem:$shortTVItem")
            VideoPreloadManager.stopPreload(shortTVItem.key)
        }
    }


    private fun checkToStart(position: Int) {
//        if (!NetworkUtil.isNetworkConnected(context)) {
//            Log.e(TAG, "checkToStart  position:$position  没有网")
//            return
//        }
        if (position in 0 until shortTVItemList.size) {
            val shortTVItem = shortTVItemList[position] ?: return
            VideoPreloadManager.runOnPreloadThread {
                VideoPreloadManager.preloadVideoAsync(shortTVItem).apply {
                    activeTasks[shortTVItem.key] = this
                }
            }
        }
    }

    fun release() {
        DemoUtil.getDownloadManager(context).removeListener(listener)
        pause()
    }

    fun add(mediaSource: MediaSource) {
        shortTVItemList.add(mediaSource)
        loadNext()
    }

    fun add(index: Int, mediaSource: MediaSource) {
        if (index <= shortTVItemList.size) {
            shortTVItemList.add(index, mediaSource)
        }
        loadNext()
    }
}