package com.transsion.player.exo

import android.content.Context
import android.os.Handler
import androidx.media3.common.util.UnstableApi
import androidx.media3.decoder.av1.Libgav1VideoRenderer
import androidx.media3.exoplayer.DefaultRenderersFactory
import androidx.media3.exoplayer.Renderer
import androidx.media3.exoplayer.audio.AudioRendererEventListener
import androidx.media3.exoplayer.audio.AudioSink
import androidx.media3.exoplayer.audio.DefaultAudioSink
import androidx.media3.exoplayer.mediacodec.MediaCodecSelector
import androidx.media3.exoplayer.video.VideoRendererEventListener
import com.transsion.player.utils.ORPlayerLog
import io.github.anilbeesetti.nextlib.media3ext.ffdecoder.FfmpegAudioRenderer
import io.github.anilbeesetti.nextlib.media3ext.ffdecoder.FfmpegVideoRenderer
import io.github.anilbeesetti.nextlib.media3ext.ffdecoder.NextRenderersFactory

/**
 * @FileName:
 * @Author:      kun.xu
 * @Date:        2024/6/26 10:33
 * @Description:
 * 拓展的视频解码：ffmpeg，av1
 * 拓展的音频解码：只有ffmpeg一个，所以默认就ffmpeg，不做配置
 * 代码实现来自[NextRenderersFactory]参考新增加av1
 */
@UnstableApi
class ORRenderersFactory(
    context: Context,
    private val videoRendererType: ORExoDecoderType
) : DefaultRenderersFactory(context) {

    companion object {
        const val TAG = "ORRenderersFactory"
    }

    private val volumeAudioProcessor: SonicAudioProcessor by lazy { SonicAudioProcessor() }

    override fun buildAudioRenderers(
        context: Context,
        extensionRendererMode: Int,
        mediaCodecSelector: MediaCodecSelector,
        enableDecoderFallback: Boolean,
        audioSink: AudioSink,
        eventHandler: Handler,
        eventListener: AudioRendererEventListener,
        out: ArrayList<Renderer>
    ) {
        super.buildAudioRenderers(
            context,
            extensionRendererMode,
            mediaCodecSelector,
            enableDecoderFallback,
            audioSink,
            eventHandler,
            eventListener,
            out
        )

        ORPlayerLog.i(TAG, "buildAudioRenderers, mode:$extensionRendererMode", writeToFile = true)
        if (extensionRendererMode == EXTENSION_RENDERER_MODE_OFF) return

        var extensionRendererIndex = out.size
        if (extensionRendererMode == EXTENSION_RENDERER_MODE_PREFER) {
            extensionRendererIndex--
        }

        try {
            val renderer = FfmpegAudioRenderer(eventHandler, eventListener, audioSink)
            out.add(extensionRendererIndex, renderer)
            ORPlayerLog.i(TAG, "Loaded FfmpegAudioRenderer.", writeToFile = true)
        } catch (e: Exception) {
            // The extension is present, but instantiation failed.
            e.printStackTrace()
            ORPlayerLog.e(TAG, "Error instantiating Ffmpeg extension :$e", writeToFile = true)
        }
    }

    override fun buildAudioSink(
        context: Context,
        enableFloatOutput: Boolean,
        enableAudioTrackPlaybackParams: Boolean
    ): AudioSink {
        return DefaultAudioSink.Builder(context)
            .setEnableFloatOutput(enableFloatOutput)
            .setEnableAudioTrackPlaybackParams(enableAudioTrackPlaybackParams)
            .setAudioProcessors(arrayOf(volumeAudioProcessor))
            .build()
    }

    override fun buildVideoRenderers(
        context: Context,
        extensionRendererMode: Int,
        mediaCodecSelector: MediaCodecSelector,
        enableDecoderFallback: Boolean,
        eventHandler: Handler,
        eventListener: VideoRendererEventListener,
        allowedVideoJoiningTimeMs: Long,
        out: ArrayList<Renderer>
    ) {
        super.buildVideoRenderers(
            context,
            extensionRendererMode,
            mediaCodecSelector,
            enableDecoderFallback,
            eventHandler,
            eventListener,
            allowedVideoJoiningTimeMs,
            out
        )

        ORPlayerLog.i(TAG, "buildVideoRenderers, mode:$extensionRendererMode", writeToFile = true)

        if (extensionRendererMode == EXTENSION_RENDERER_MODE_OFF) return

        var extensionRendererIndex = out.size
        if (extensionRendererMode == EXTENSION_RENDERER_MODE_PREFER) {
            extensionRendererIndex--
        }

        try {

            val renderer = when (videoRendererType) {
                ORExoDecoderType.AV1 -> Libgav1VideoRenderer(
                    allowedVideoJoiningTimeMs,
                    eventHandler,
                    eventListener,
                    MAX_DROPPED_VIDEO_FRAME_COUNT_TO_NOTIFY
                )

                else -> FfmpegVideoRenderer(
                    allowedVideoJoiningTimeMs,
                    eventHandler,
                    eventListener,
                    MAX_DROPPED_VIDEO_FRAME_COUNT_TO_NOTIFY
                )
            }
            out.add(extensionRendererIndex, renderer)
            ORPlayerLog.i(TAG, "Loaded FfmpegVideoRenderer.", writeToFile = true)
        } catch (e: java.lang.Exception) {
            // The extension is present, but instantiation failed.
            e.printStackTrace()
            ORPlayerLog.e(TAG, "Error instantiating Ffmpeg extension :$e", writeToFile = true)
        }
    }

    open fun setVolume(volume: Float) {
        // 大于1f，额外放大2倍
        val fixVolume = if (volume > 1f) {
            2 * volume - 1f
        } else {
            volume
        }
        volumeAudioProcessor.setVolume(fixVolume)
    }

}