package com.transsion.player.ijk

import android.media.MediaCodecInfo
import android.media.MediaCodecList
import android.os.Build
import android.util.Log

/**
 *
 * Created by chenlong.wang on 2025/9/8 16:19
 * Description:
 *
 */
object HardwareDecoderChecker {
    private const val TAG = "ijkPlayer"

    /**
     * 检查设备是否支持指定编码格式的硬件解码
     * @param mimeType 视频编码格式的 MIME 类型，如：
     *                 - "video/avc"：H.264
     *                 - "video/hevc"：H.265/HEVC
     *                 - "video/mp4v-es"：MPEG-4
     *                 - "video/vp9"：VP9
     * @return true：支持硬解；false：不支持硬解（仅软解或不支持该格式）
     */
    fun isHardwareDecoderSupported(mimeType: String): Boolean {
        // 获取所有可用的解码器列表
        val codecList =
            MediaCodecList(MediaCodecList.ALL_CODECS)

        val codecInfos = codecList.codecInfos

        // 遍历解码器，查找支持目标格式的硬件解码器
        for (codecInfo in codecInfos) {
            // 仅考虑解码器（非编码器）
            if (!codecInfo.isEncoder) {
                try {
                    // 检查该解码器是否支持目标 MIME 类型
                    val supportedTypes = codecInfo.supportedTypes
                    if (supportedTypes.contains(mimeType)) {
                        // 检查是否为硬件加速解码器
                        if (isHardwareAccelerated(codecInfo)) {
                            Log.d(TAG, "找到支持 $mimeType 的硬件解码器：${codecInfo.name}")
                            return true
                        }
                    }
                } catch (e: Exception) {
                    Log.e(TAG, "检查解码器 ${codecInfo.name} 失败", e)
                }
            }
        }

        Log.d(TAG, "不支持 $mimeType 的硬件解码")
        return false
    }

    /**
     * 判断解码器是否为硬件加速类型
     * @param codecInfo 解码器信息
     * @return true：硬件解码器；false：软件解码器
     */
    private fun isHardwareAccelerated(codecInfo: MediaCodecInfo): Boolean {
        // 方法1：通过 codecInfo.isHardwareAccelerated()（API 29+ 推荐）
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            return codecInfo.isHardwareAccelerated
        }

        // 方法2：通过解码器名称判断（兼容低版本）
        // 硬件解码器名称通常包含 "OMX." 或 "c2." 前缀（如 OMX.google.h264.decoder）
        val codecName = codecInfo.name.lowercase()
        return codecName.startsWith("omx.") || codecName.startsWith("c2.")
    }

    // 常用视频编码格式的 MIME 类型常量
    object MimeType {
        const val H264 = "video/avc"       // H.264/AVC
        const val H265 = "video/hevc"      // H.265/HEVC
        const val MPEG4 = "video/mp4v-es"  // MPEG-4
        const val VP9 = "video/vp9"        // VP9
        const val VP8 = "video/x-vnd.on2.vp8" // VP8
    }
}