package com.transsion.player.orplayer.global

import android.os.Build
import android.os.Handler
import android.os.HandlerThread
import android.os.Looper
import android.os.Process
import android.view.SurfaceView
import android.view.TextureView
import com.blankj.utilcode.util.Utils
import com.transsion.player.MediaSource
import com.transsion.player.config.VodConfig
import com.transsion.player.enum.ScaleMode
import com.transsion.player.orplayer.IPlayerListener
import com.transsion.player.orplayer.ORPlayer
import com.transsion.player.orplayer.PlayError
import com.transsion.player.tracks.TnFormat
import com.transsion.player.tracks.TnTracks
import com.transsion.player.tracks.TnTracksGroup
import com.transsion.player.utils.ORPlayerLog
import java.util.concurrent.CopyOnWriteArrayList
import java.util.concurrent.atomic.AtomicBoolean

class TnAliPlayer private constructor() : ORPlayer, IPlayerListener {
    companion object {
        fun newInstance(): TnAliPlayer {
            return TnAliPlayer()
        }

        val mUncaughtExceptionHandler: Thread.UncaughtExceptionHandler =
            Thread.UncaughtExceptionHandler { t, e -> ORPlayerLog.e(TAG, "uncaughtException:$e", true) }
        const val TAG = "TnPlayer"
    }

    private val handlerThread: HandlerThread?
    private val playerHandler: Handler
    private var mOrPlayer: ORPlayer? = null

    @Volatile
    private var mDuration: Long = 0

    @Volatile
    private var listener: IPlayerListener? = null

    private var listeners: CopyOnWriteArrayList<IPlayerListener> = CopyOnWriteArrayList()

    @Volatile
    var mediaSource: MediaSource? = null


    init {

        if (TnPlayerManager.isASyncPlayer) {
            val priority =
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) Process.THREAD_PRIORITY_VIDEO else Process.THREAD_PRIORITY_AUDIO

            handlerThread = HandlerThread(
                "TnPlayerThread",
                priority
            ).apply {
                start()
                uncaughtExceptionHandler = mUncaughtExceptionHandler
                playerHandler = Handler(this.looper)
            }
        } else {
            handlerThread = null
            playerHandler = Handler(Looper.getMainLooper())
        }


        runOnPlayerThread {
            mOrPlayer = ORPlayer.Builder(Utils.getApp())
                .vodConfig(VodConfig(clearFrameWhenStop = false))
                .builder().apply {
                    setLooping(false)
                    setAutoPlay(false)
                    setPlayerListener(this@TnAliPlayer)
                }
            log("init player:$mOrPlayer")
        }
    }


    override fun setSurfaceView(surfaceView: SurfaceView?) {
        runOnPlayerThread {
            getPlayer()?.setSurfaceView(surfaceView)
            log("setSurfaceView:$surfaceView")
        }
    }

    override fun setTextureView(textureView: TextureView?) {
        runOnPlayerThread {
            getPlayer()?.setTextureView(textureView)
            log("setTextureView:$textureView")
        }
    }

    override fun setPlayerListener(listener: IPlayerListener) {
        runOnPlayerThread {
            this.listener = listener
            log("setPlayerListener:$listener")
        }
    }

    override fun addPlayerListener(listener: IPlayerListener) {
        if (listeners.contains(listener).not()){
            runOnPlayerThread {
                listeners.add(listener)
                log("addPlayerListener:$listener")
            }
        }
    }

    override fun removePlayerListener(listener: IPlayerListener) {
        runOnPlayerThread {
            listeners.remove(listener)
            log("removePlayerListener:$listener")
        }
    }

    override fun setDataSource(mediaSource: MediaSource) {
        runOnPlayerThread {
            //TnPlayerManager.requestFocus(this, "setDataSource")
            this.mediaSource = mediaSource
            mDuration = 0
            nextSeekTo = 0L
            isSeekToFinish.set(true)
            getPlayer()?.setDataSource(mediaSource)
            log("setDataSource:$mediaSource")
        }
    }

    override fun addDataSource(mediaSource: MediaSource): Boolean {
        setDataSource(mediaSource)
        return true
    }

    override fun prepare() {
        runOnPlayerThread {
            getPlayer()?.prepare()
            log("prepare:$mediaSource")
        }
    }

    override fun play() {
        runOnPlayerThread {
            getPlayer()?.play()
            log("play:$mediaSource")
        }
    }

    override fun pause() {
        runOnPlayerThread {
            getPlayer()?.pause()
            log("pause:$mediaSource")
        }
    }

    override fun stop() {
        runOnPlayerThread {
            getPlayer()?.stop()
            log("stop:$mediaSource")
        }
    }

    override fun release() {
        mDuration = 0
        runOnPlayerThread {
            nextSeekTo = 0L
            isSeekToFinish.set(true)
            getPlayer()?.stop()
            if (TnPlayerManager.isMultiplePlayer) {
                if (TnPlayerPool.needReleasePlayer(this)) {
                    getPlayer()?.release()
                    handlerThread?.quitSafely()
                    handlerThread?.uncaughtExceptionHandler = null
                    playerHandler.removeCallbacksAndMessages(null)
                } else {
                    getPlayer()?.reset()
                    getPlayer()?.clearScreen()
                }
                log(" release:$mediaSource")
            } else {
                getPlayer()?.reset()
                log(" reset:$mediaSource")
            }
            listener = null
        }
    }

    override fun reset() {
        runOnPlayerThread {
            log("reset:$mediaSource")
            nextSeekTo = 0L
            isSeekToFinish.set(true)
            getPlayer()?.clearScreen()
            getPlayer()?.reset()
        }
    }

    private var nextSeekTo = 0L
    override fun seekTo(mills: Long) {
        if (isSeekToFinish.compareAndSet(true, false)) {
            runOnPlayerThread {
                getPlayer()?.seekTo(mills)
                log(" seekTo:$mills  mediaSource:$mediaSource")
            }
            nextSeekTo = 0L
        } else {
            nextSeekTo = mills
            log("seekTo:$mills  waiting.... mediaSource:$mediaSource")
        }
    }

    override fun getDuration(): Long {
        return mDuration
    }

    override fun isPlaying(): Boolean {
        return getPlayer()?.isPlaying() ?: false
    }

    override fun isComplete(): Boolean {
        return getPlayer()?.isComplete() ?: false
    }

    override fun setLooping(isLoop: Boolean) {
        runOnPlayerThread {
            getPlayer()?.setLooping(isLoop)
            log("setLooping:$isLoop  mediaSource:$mediaSource")
        }
    }

    override fun isMute(): Boolean {
        return getPlayer()?.isMute() ?: false
    }

    override fun setMute(mute: Boolean) {
        getPlayer()?.setMute(mute)
    }

    override fun setScaleMode(scaleMode: ScaleMode) {
        getPlayer()?.setScaleMode(scaleMode)
    }

    override fun setVolume(volume: Float) {
        runOnPlayerThread {
            getPlayer()?.setVolume(volume)
            log("setVolume:$volume  mediaSource:$mediaSource")
        }
    }

    override fun getVolume(): Float? {
        return getPlayer()?.getVolume()
    }

    override fun isLoading(): Boolean {
        return getPlayer()?.isLoading() ?: super.isLoading()
    }

    override fun getVideoWidth(): Int {
        return getPlayer()?.getVideoWidth() ?: super.getVideoWidth()
    }

    override fun getVideoHeight(): Int {
        return getPlayer()?.getVideoHeight() ?: super.getVideoHeight()
    }

    override fun isPrepared(): Boolean {
        return getPlayer()?.isPrepared() ?: super.isPrepared()
    }

    override fun getBitrate() = getPlayer()?.getBitrate() ?: Pair(0, 0)

    override fun setAutoPlay(isAutoPlay: Boolean) {
        runOnPlayerThread {
            getPlayer()?.setAutoPlay(isAutoPlay)
            log("setAutoPlay:$isAutoPlay  mediaSource:$mediaSource")
        }
    }

    override fun enableHardwareDecoder(enable: Boolean) {
        runOnPlayerThread {
            getPlayer()?.enableHardwareDecoder(enable)
        }
    }

    override fun setPlayerConfig(vodConfig: VodConfig) {
        runOnPlayerThread { getPlayer()?.setPlayerConfig(vodConfig) }
    }

    override fun clearScreen() {
        runOnPlayerThread {
            getPlayer()?.clearScreen()
            log("clearScreen  mediaSource:$mediaSource")
        }
    }


    override fun onSetDataSource() {
        listener?.onSetDataSource()
        listeners.forEach { listener ->
            listener.onSetDataSource()
        }
        log("onSetDataSource  mediaSource:$mediaSource")
    }

    /**
     * 播放器初始化
     * play/prepare之前调用
     */
    override fun initPlayer() {
        listener?.initPlayer()
        listeners.forEach { listener ->
            listener.initPlayer()
        }
        log("initPlayer  mediaSource:$mediaSource")
    }

    /**
     * 首帧渲染
     */
    override fun onRenderFirstFrame() {
        log("onRenderFirstFrame  mediaSource:$mediaSource")
        listener?.onRenderFirstFrame()
        listeners.forEach { listener ->
            listener.onRenderFirstFrame()
        }
    }

    /**
     * 播放完成
     * 开启循环播放时，不会调用该方法
     */
    override fun onCompletion(mediaSource: MediaSource?) {
        listener?.onCompletion(mediaSource)
        listeners.forEach { listener ->
            listener.onCompletion(mediaSource)
        }
        log("onCompletion  mediaSource:$mediaSource")
    }

    override fun onPlayerRelease(mediaSource: MediaSource?) {
        listener?.onPlayerRelease(mediaSource)
        listeners.forEach { listener ->
            listener.onPlayerRelease(mediaSource)
        }
        log("onPlayerRelease  mediaSource:$mediaSource")
    }

    /***准备播放新的音频 将唯一标识url回调*/
    override fun onPrepare(mediaSource: MediaSource?) {
        mDuration = getPlayer()?.getDuration() ?: 0
        listener?.onPrepare(mediaSource)
        listeners.forEach { listener ->
            listener.onPrepare(mediaSource)
        }
        log("onPrepare  mediaSource:$mediaSource")
    }

    /**
     * 暂停
     */
    override fun onVideoPause(mediaSource: MediaSource?) {
        listener?.onVideoPause(mediaSource)
        listeners.forEach { listener ->
            listener.onVideoPause(mediaSource)
        }
        log("onVideoPause  mediaSource:$mediaSource")
    }

    /**
     * 播放开始
     */
    override fun onVideoStart(mediaSource: MediaSource?) {
        listener?.onVideoStart(mediaSource)
        listeners.forEach { listener ->
            listener.onVideoStart(mediaSource)
        }
        log("onVideoStart  mediaSource:$mediaSource")
    }

    /**
     * 自动开始播放开始
     */
    override fun onLoopingStart() {
        listener?.onLoopingStart()
        listeners.forEach { listener ->
            listener.onLoopingStart()
        }
        log("onLoopingStart  mediaSource:$mediaSource")
    }

    /**
     * 播放出错
     */
    override fun onPlayError(errorInfo: PlayError, mediaSource: MediaSource?) {
        listener?.onPlayError(errorInfo, mediaSource)
        listeners.forEach { listener ->
            listener.onPlayError(errorInfo, mediaSource)
        }
        isSeekToFinish.set(true)
        log("onPlayError  errorInfo:$errorInfo mediaSource:$mediaSource")
    }

    /**
     * 视频大小改变
     */
    override fun onVideoSizeChanged(width: Int, height: Int) {
        listener?.onVideoSizeChanged(width, height)
        listeners.forEach { listener ->
            listener.onVideoSizeChanged(width, height)
        }
        log("onVideoSizeChanged width:$width height:$height mediaSource:$mediaSource")
    }

    /**
     * 缓冲开始
     */
    override fun onLoadingBegin(mediaSource: MediaSource?) {
        listener?.onLoadingBegin(mediaSource)
        listeners.forEach { listener ->
            listener.onLoadingBegin(mediaSource)
        }
        log("onLoadingBegin  mediaSource:$mediaSource")
    }

    override fun setOnSeekCompleteListener() {
        super.setOnSeekCompleteListener()
        isSeekToFinish.set(true)
        listener?.setOnSeekCompleteListener()
        listeners.forEach { listener ->
            listener.setOnSeekCompleteListener()
        }
        log("setOnSeekCompleteListener nextSeekTo:$nextSeekTo mediaSource:$mediaSource")

        if (nextSeekTo > 0) {
            seekTo(nextSeekTo)
        }
    }


    /**
     * 缓冲进度
     */
    override fun onBufferedPosition(progress: Long, mediaSource: MediaSource?) {
        listener?.onBufferedPosition(progress, mediaSource)
        listeners.forEach { listener ->
            listener.onBufferedPosition(progress, mediaSource)
        }
//        log( "onBufferedPosition progress:$progress mediaSource:$mediaSource")
    }

    override fun onLoadingProgress(percent: Int, netSpeed: Float, mediaSource: MediaSource?) {
        listener?.onLoadingProgress(percent, netSpeed, mediaSource)
        listeners.forEach { listener ->
            listener.onLoadingProgress(percent, netSpeed, mediaSource)
        }
        log("onLoadingProgress percent:$percent netSpeed:$netSpeed mediaSource:$mediaSource")
    }

    private val isSeekToFinish = AtomicBoolean(true)
    override fun onProgress(progress: Long, mediaSource: MediaSource?) {
        if (isSeekToFinish.get()) {
            listener?.onProgress(progress, mediaSource)
            listeners.forEach { listener ->
                listener.onProgress(progress, mediaSource)
            }
        }
//        log( "onProgress progress:$progress mediaSource:$mediaSource")
    }

    /**
     * 缓冲结束
     */
    override fun onLoadingEnd(mediaSource: MediaSource?) {
        listener?.onLoadingEnd(mediaSource)
        listeners.forEach { listener ->
            listener.onLoadingEnd(mediaSource)
        }
        log("onLoadingEnd mediaSource:$mediaSource")
    }

    override fun onPlayerReset() {
        listener?.onPlayerReset()
        listeners.forEach { listener ->
            listener.onPlayerReset()
        }
        log("onPlayerReset mediaSource:$mediaSource")
    }

    override fun onIsPlayingChanged(isPlaying: Boolean) {
        listener?.onIsPlayingChanged(isPlaying)
        listeners.forEach { listener ->
            listener.onIsPlayingChanged(isPlaying)
        }
    }

    //focusLoss true：音频焦点丢失，false：音频焦点获取
    override fun onFocusChange(focusLoss: Boolean) {
        listener?.onFocusChange(focusLoss)
        listeners.forEach { listener ->
            listener.onFocusChange(focusLoss)
        }
        log("onFocusChange mediaSource:$mediaSource")
    }

    override fun onAliyunDecodeErrorChangeSoftwareDecoder(mediaSource: MediaSource?) {
        listener?.onAliyunDecodeErrorChangeSoftwareDecoder(mediaSource)
        listeners.forEach { listener ->
            listener.onAliyunDecodeErrorChangeSoftwareDecoder(mediaSource)
        }
    }

    override fun onTracksVideoBitrateChange(videoBitrate: Int) {
        listener?.onTracksVideoBitrateChange(videoBitrate)
        listeners.forEach { listener ->
            listener.onTracksVideoBitrateChange(videoBitrate)
        }
    }

    override fun onTracksAudioBitrateChange(audioBitrate: Int) {
        listener?.onTracksAudioBitrateChange(audioBitrate)
        listeners.forEach { listener ->
            listener.onTracksAudioBitrateChange(audioBitrate)
        }
    }

    fun runOnPlayerThread(action: () -> Unit) {
        if (Thread.currentThread() == playerHandler.looper.thread) {
            action.invoke()
        } else {
            if (handlerThread?.isAlive != true) {
                log("runOnPlayerThread handlerThread.isAlive:false")
                return
            }
            playerHandler.post {
                action.invoke()
            }

        }
    }

    private fun log(msg: String) {
        ORPlayerLog.d(TAG, "this@${this.hashCode()}  player@${mOrPlayer.hashCode()}  $msg", true)
    }

    override fun getDownloadBitrate(): Any? {
        return getPlayer()?.getDownloadBitrate()
    }

    override fun getCurrentTracks(): TnTracks? {
        return getPlayer()?.getCurrentTracks()
    }

    override fun getCurrentPosition(): Long {
        return getPlayer()?.getCurrentPosition() ?: 0L
    }

    override fun getCurrentVideoFormat(): TnFormat? {
        return getPlayer()?.getCurrentVideoFormat()
    }

    override fun changeTrackSelection(mediaTrackGroup: TnTracksGroup, index: Int) {
        getPlayer()?.changeTrackSelection(mediaTrackGroup, index)
    }

    override fun setSpeed(speed: Float) {
        runOnPlayerThread { getPlayer()?.setSpeed(speed) }
    }

    override fun setErrorInterceptor(listener: TnErrorInterceptorListener) {
        runOnPlayerThread { getPlayer()?.setErrorInterceptor(listener) }
    }

    override fun clearSurfaceOnly() {
        getPlayer()?.clearSurfaceOnly()
    }

    private fun getPlayer(): ORPlayer? {
        return mOrPlayer
    }

}