package com.transsion.player.ui

import android.content.Context
import android.graphics.Color
import android.util.AttributeSet
import android.view.Gravity
import android.view.View
import android.view.ViewGroup
import android.widget.FrameLayout
import android.widget.LinearLayout
import com.transsion.player.MediaSource
import com.transsion.player.control.BaseVideoController
import com.transsion.player.enum.PipStatus
import com.transsion.player.enum.ScaleMode
import com.transsion.player.orplayer.IPlayerListener
import com.transsion.player.orplayer.ORPlayer
import com.transsion.player.orplayer.global.TnPlayerManager
import com.transsion.player.orplayer.global.TnPlayerType
import com.transsion.player.tracks.TnTracksGroup
import com.transsion.player.utils.LandSpaceHelper
import com.transsion.player.utils.ORPlayerLog
import java.util.concurrent.CopyOnWriteArrayList

/**
 *
 * Created by kun.xu on 2024/11/14 10:13
 * Description:
 *
 * 视频view基类
 * context必须是 FragmentActivity
 * 全屏切换就是这里切换到activity的content上
 *
 */
class ORVideoView : FrameLayout, IORVideoView {

    private val TAG = "ORVideoView-player"
    private var mPlayer: ORPlayer? = null

    private var playerRoot = FrameLayout(context)//播放器+控制层 容器
    private val startSpaceView = View(context).apply {
        setBackgroundColor(Color.BLACK)
    }
    private val endSpaceView = View(context).apply {
        setBackgroundColor(Color.BLACK)
    }
    private var playerContainer = LinearLayout(context)//跟布局，添加：左空白+playerRoot播放器root+右空白

    private var videoController: BaseVideoController<*, *>? = null//播放器控制层
    private val spaceHelper: LandSpaceHelper

    private var isFullscreen = false//是否全屏
    private var isLand = true//全屏下-当前是否是横屏

    private var curPipStatus = PipStatus.DEFAULT//全屏下-当前是否是横屏

    //
    private var listeners = CopyOnWriteArrayList<IPlayerListener>()
    private var speed: Float = 1f
    private var isLoop = false
    private var volume = 1f
    private var mute = false
    private var scaleMode = ScaleMode.SCALE_ASPECT_FIT

    //activity页面的全屏容器
    private var activityContainer: (() -> ViewGroup?)? = null

    constructor(context: Context) : this(context, null)
    constructor(context: Context, attrs: AttributeSet?) : this(context, attrs, 0)
    constructor(context: Context, attrs: AttributeSet?, defStyleAttr: Int) : super(context, attrs, defStyleAttr) {

    }

    init {
        spaceHelper = LandSpaceHelper(startSpaceView, endSpaceView)
    }

    /**
     * 初始化创建播放器
     *
     */
    fun initCreatePlayer(context: Context, player: ORPlayer) {
        mPlayer = player
        val orPlayerView = ORPlayerView(context)//默认使用surfaceView
        addPlayerView(orPlayerView)
//        mPlayer?.setSurfaceView(orPlayerView.getSurface())
        var playerTye = TnPlayerManager.isUsePlayerType
        if (playerTye == TnPlayerType.IJK.value) {
            mPlayer?.setTextureView(orPlayerView.getTextureView())
        } else {
            mPlayer?.setSurfaceView(orPlayerView.getSurface())
        }

    }

    fun setActivityContainerCallback(callback: () -> ViewGroup?) {
        activityContainer = callback
    }

    private fun addPlayerView(orPlayerView: ORPlayerView) {
        addView(playerContainer, LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT))

        //添加左右SpaceView
        playerContainer.orientation = LinearLayout.HORIZONTAL
        playerContainer.addView(startSpaceView, LinearLayout.LayoutParams(0, LinearLayout.LayoutParams.MATCH_PARENT))
        val rootLp = LinearLayout.LayoutParams(0, LinearLayout.LayoutParams.MATCH_PARENT).apply { weight = 1f }
        playerContainer.addView(playerRoot, rootLp)
        playerContainer.addView(endSpaceView, LinearLayout.LayoutParams(0, LinearLayout.LayoutParams.MATCH_PARENT))

        val renderViewLp = LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT).apply {
            gravity = Gravity.CENTER
        }
        playerRoot.addView(orPlayerView, 0, renderViewLp)
    }

    /**
     * 设置控制器，传null表示移除控制器
     */
    fun setVideoController(mediaController: BaseVideoController<*, *>?) {
        playerRoot.removeView(videoController)
        videoController = mediaController
        if (mediaController != null) {
            mediaController.setPayerView(this)
            playerRoot.addView(mediaController, LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT))
        }
    }

    override fun setMediaData(data: MediaSource) {
        ORPlayerLog.d(TAG, "setMediaData, url:${data.url} ",true)
        mPlayer?.setDataSource(data)
        spaceHelper.enable()
        keepScreenOn = true
    }


    override fun prepare() {
        mPlayer?.prepare()
    }

    override fun play() {
        mPlayer?.play()
        spaceHelper.enable()
        keepScreenOn = true
    }

    override fun pause() {
        mPlayer?.pause()
        spaceHelper.disable()
        keepScreenOn = false
    }

    override fun addPlayerListener(listener: IPlayerListener) {
        if (listeners.contains(listener).not()) {
            listeners.add(listener)
        }
        mPlayer?.addPlayerListener(listener)
    }

    override fun removePayerListener(listener: IPlayerListener) {
        listeners.remove(listener)
        mPlayer?.removePlayerListener(listener)
    }

    override fun seekTo(mills: Long) {
        mPlayer?.seekTo(mills)
    }

    override fun setSpeed(speed: Float) {
        this.speed = speed
        mPlayer?.setSpeed(speed)
    }

    override fun reset() {
        mPlayer?.reset()
        spaceHelper.disable()
        keepScreenOn = false
    }

    override fun release() {
        mPlayer?.release()
        videoController?.getControlComponentMap()?.forEach {
            it.key.release()
        }
        spaceHelper.disable()
        keepScreenOn = false
    }

    override fun clearScreen() {
        mPlayer?.clearScreen()
    }

    override fun setLooping(isLoop: Boolean) {
        this.isLoop = isLoop
        mPlayer?.setLooping(isLoop)
    }

    override fun setScaleMode(scaleMode: ScaleMode) {
        this.scaleMode = scaleMode
        mPlayer?.setScaleMode(scaleMode)
    }

    override fun setVolume(volume: Float) {
        this.volume = volume
        mPlayer?.setVolume(volume)
    }

    override fun setMute(mute: Boolean) {
        this.mute = mute
        mPlayer?.setMute(mute)
    }

    override fun enableHardwareDecoder(enable: Boolean) {
        mPlayer?.enableHardwareDecoder(enable)
    }

    override fun changeTrackSelection(mediaTrackGroup: TnTracksGroup, index: Int) {
        mPlayer?.changeTrackSelection(mediaTrackGroup, index)
    }

    override fun getMediaData(): MediaSource? = mPlayer?.currentMediaSource()

    override fun getDuration(): Long = mPlayer?.getDuration() ?: 0L

    override fun getPosition(): Long = mPlayer?.getCurrentPosition() ?: 0L

    override fun isPlaying(): Boolean = mPlayer?.isPlaying() ?: false

    override fun isComplete(): Boolean = mPlayer?.isComplete() ?: false

    override fun isMute(): Boolean = mPlayer?.isMute() ?: false

    override fun isBuffering(): Boolean = mPlayer?.isLoading() ?: false

    override fun getVideoWidth(): Int = mPlayer?.getVideoWidth() ?: 0

    override fun getVideoHeight(): Int = mPlayer?.getVideoHeight() ?: 0

    override fun isPrepared(): Boolean = mPlayer?.isPrepared() ?: false

    override fun getVideoFormat() = mPlayer?.getCurrentVideoFormat()

    override fun isLand(): Boolean = isLand
    override fun isFullscreen(): Boolean = isFullscreen

    override fun changeFullscreen(fullscreen: Boolean, land: Boolean) {
        val activityContent = activityContainer?.invoke() ?: return
        isFullscreen = fullscreen
        isLand = if (fullscreen.not()) false else land
        ORPlayerLog.d(TAG, "changeFullscreen, isFullscreen:$isFullscreen, isLand:$isLand, curPipStatus:$curPipStatus")
        (playerContainer.parent as? ViewGroup)?.removeView(playerContainer)//切换之前先从父类移除
        if (fullscreen) {//全屏-添加到activityContent上
            activityContent.addView(playerContainer, LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT))
        } else {//退出全屏-添加恢复
            addView(playerContainer, 0, LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT))
        }

        //回调切换activity横竖屏，状态栏
        videoController?.getControlComponentMap()?.forEach { (component, _) ->
            component.onFullscreenChanged(fullscreen, land)
        }

        spaceHelper.onScreenChange(isLand)


    }

    override fun pipChangeFullscreen(pipStatus: PipStatus) {
        curPipStatus = pipStatus
        videoController?.getControlComponentMap()?.forEach { (component, _) ->
            component.onPipStatusChange(pipStatus)
        }
        when (pipStatus) {
            PipStatus.DEFAULT -> spaceHelper.enable()
            PipStatus.PIP, PipStatus.FLOAT -> {
                spaceHelper.disable()
                spaceHelper.onScreenChange(false)
            }
        }
        val activityContent = activityContainer?.invoke() ?: return
        ORPlayerLog.d(TAG, "pipChangeFullscreen, isLand:$isLand, curPipStatus:$curPipStatus")
        (playerContainer.parent as? ViewGroup)?.removeView(playerContainer)//切换之前先从父类移除
        if (pipStatus == PipStatus.PIP) {//全屏-添加到activityContent上
            activityContent.addView(playerContainer, LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT))
        } else {//退出全屏-添加恢复
            addView(playerContainer, 0, LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT))
        }

        if (isFullscreen && pipStatus == PipStatus.PIP) {
            //当前是全屏，又进入了画中画，activity需要退出全屏
            videoController?.getControlComponentMap()?.forEach { (component, _) ->
                component.onFullscreenChanged(fullscreen = false, land = false)
            }
        }
    }

    override fun setPipStatus(pipStatus: PipStatus) {
        curPipStatus = pipStatus
        videoController?.onPipStatusChange(pipStatus)
        videoController?.getControlComponentMap()?.forEach { (component, _) ->
            component.onPipStatusChange(pipStatus)
        }
        when (pipStatus) {
            PipStatus.DEFAULT -> spaceHelper.enable()
            PipStatus.PIP -> spaceHelper.disable()
            PipStatus.FLOAT -> spaceHelper.disable()
        }
    }

    override fun getCurPipStatus(): PipStatus = curPipStatus
}