package com.tn.lib.logger

import android.util.Log
import com.elvishew.xlog.BuildConfig
import com.elvishew.xlog.XLog
import com.tn.lib.logger.xlog.ILogger
import com.tn.lib.logger.xlog.LogType
import com.tn.lib.logger.xlog.LoggerApplication
import com.tn.lib.logger.xlog.impl.LogcatLoggerImpl
import com.tn.lib.logger.xlog.impl.XLogImpl


class Logger {
    companion object {
        private var logger: ILogger? = null // getLoggerImp()

        const val TAG = "MG_"

        private fun isLogcat(tag: String, writeToFile: Boolean): Boolean {
            return writeToFile.not() || Log.isLoggable(contact(tag), Log.DEBUG)
        }

        // log.d log.v log.i 在release版本不会输出到logcat 使用log.w代替
        private fun releaseNeedToLog(tag: String): Boolean {
            return BuildConfig.DEBUG.not() && Log.isLoggable(contact(tag), Log.DEBUG)
        }

        private fun contact(tag: String):String {
            (TAG + tag).apply {
                return if (length <= 23) {
                    this
                } else {
                    substring(0, 23)
                }
            }
        }

        private fun getLoggerImp(): ILogger {
            return when (LoggerApplication.getLogType()) {
                LogType.TYPE_LOGCAT -> LogcatLoggerImpl.instance
                LogType.TYPE_XLOG -> XLogImpl.instance
            }
        }

        fun i(msg: String, writeToFile: Boolean = false) {
            if (writeToFile.not()) {
                LogcatLoggerImpl.instance.i(msg)
                return
            }
            if (logger == null) {
                logger = getLoggerImp()
            }
            logger?.i(msg)
        }

        /**
         * @param tag
         * @param msg 日志内容
         * @param writeToFile false 默认只输出到logcat，不输出到文件
         */
        @JvmStatic
        fun i(tag: String, vararg msg: String, writeToFile: Boolean = false) {
            if (releaseNeedToLog(tag)) {
                LogcatLoggerImpl.instance.w(contact(tag), *msg)
                return
            }
            if (writeToFile.not()) {
                LogcatLoggerImpl.instance.i(contact(tag), *msg)
                return
            }
            if (logger == null) {
                logger = getLoggerImp()
            }
            logger?.i(contact(tag), *msg)
        }

        fun i(tag: String, msg: String, throwable: Throwable, writeToFile: Boolean = false) {
            if (releaseNeedToLog(tag)) {
                LogcatLoggerImpl.instance.w(contact(tag), msg, throwable.stackTrace.toString())
                return
            }
            if (writeToFile.not()) {
                LogcatLoggerImpl.instance.i(contact(tag), msg, throwable.stackTrace.toString())
                return
            }
            if (logger == null) {
                logger = getLoggerImp()
            }
            logger?.i(contact(tag), msg, throwable.stackTrace.toString())
        }

        fun v(msg: String, writeToFile: Boolean = false) {
            if (writeToFile.not()) {
                LogcatLoggerImpl.instance.v(msg)
                return
            }
            if (logger == null) {
                logger = getLoggerImp()
            }
            logger?.v(msg)
        }

        @JvmStatic
        fun v(tag: String, msg: String, writeToFile: Boolean = false) {
            if (releaseNeedToLog(tag)) {
                LogcatLoggerImpl.instance.w(contact(tag), msg)
                return
            }
            if (writeToFile.not()) {
                LogcatLoggerImpl.instance.v(contact(tag), msg)
                return
            }
            if (logger == null) {
                logger = getLoggerImp()
            }
            logger?.v(contact(tag), msg)
        }

        fun v(tag: String, msg: String, throwable: Throwable, writeToFile: Boolean = false) {
            if (releaseNeedToLog(tag)) {
                LogcatLoggerImpl.instance.w(contact(tag), msg, throwable.stackTrace.toString())
                return
            }
            if (writeToFile.not()) {
                LogcatLoggerImpl.instance.v(contact(tag), msg, throwable.stackTrace.toString())
                return
            }
            if (logger == null) {
                logger = getLoggerImp()
            }
            logger?.v(contact(tag), msg, throwable.stackTrace.toString())
        }

        fun d(msg: String, writeToFile: Boolean = false) {
            if (writeToFile.not()) {
                LogcatLoggerImpl.instance.d(msg)
                return
            }
            if (logger == null) {
                logger = getLoggerImp()
            }
            logger?.d(msg)
        }

        @JvmStatic
        fun d(tag: String, msg: String, writeToFile: Boolean = false) {
            if (releaseNeedToLog(tag)) {
                LogcatLoggerImpl.instance.w(contact(tag), msg)
                return
            }
            if (writeToFile.not()) {
                LogcatLoggerImpl.instance.d(contact(tag), msg)
                return
            }
            if (logger == null) {
                logger = getLoggerImp()
            }
            logger?.d(contact(tag), msg)
        }

        fun d(tag: String, msg: String, throwable: Throwable, writeToFile: Boolean = false) {
            if (releaseNeedToLog(tag)) {
                LogcatLoggerImpl.instance.w(contact(tag), msg, throwable.stackTrace.toString())
                return
            }
            if (writeToFile.not()) {
                LogcatLoggerImpl.instance.i(contact(tag), msg, throwable.stackTrace.toString())
                return
            }
            if (logger == null) {
                logger = getLoggerImp()
            }
            logger?.d(contact(tag), msg, throwable.stackTrace.toString())
        }

        fun w(msg: String, writeToFile: Boolean = false) {
            if (writeToFile.not()) {
                LogcatLoggerImpl.instance.w(msg)
                return
            }
            if (logger == null) {
                logger = getLoggerImp()
            }
            logger?.w(msg)
        }

        @JvmStatic
        fun w(tag: String, msg: String, writeToFile: Boolean = false) {
            if (isLogcat(tag, writeToFile)) {
                LogcatLoggerImpl.instance.w(contact(tag), msg)
                return
            }
            if (logger == null) {
                logger = getLoggerImp()
            }
            logger?.w(contact(tag), msg)
        }

        fun w(tag: String, msg: String, throwable: Throwable, writeToFile: Boolean = false) {
            if (isLogcat(tag, writeToFile)) {
                LogcatLoggerImpl.instance.w(contact(tag), msg, throwable.stackTrace.toString())
                return
            }
            if (logger == null) {
                logger = getLoggerImp()
            }
            logger?.w(contact(tag), msg, throwable.stackTrace.toString())
        }

        fun e(msg: String, writeToFile: Boolean = false) {
            if (writeToFile.not()) {
                LogcatLoggerImpl.instance.e(msg)
                return
            }
            if (logger == null) {
                logger = getLoggerImp()
            }
            logger?.e(msg)
        }

        @JvmStatic
        fun e(tag: String, msg: String, writeToFile: Boolean = false) {
            if (isLogcat(tag, writeToFile)) {
                LogcatLoggerImpl.instance.e(contact(tag), msg)
                return
            }
            if (logger == null) {
                logger = getLoggerImp()
            }
            logger?.e(contact(tag), msg)
        }

        fun e(tag: String, msg: String, throwable: Throwable, writeToFile: Boolean = false) {
            if (isLogcat(tag, writeToFile)) {
                LogcatLoggerImpl.instance.e(contact(tag), msg, throwable.stackTrace.toString())
                return
            }
            if (logger == null) {
                logger = getLoggerImp()
            }
            logger?.e(contact(tag), msg, throwable.stackTrace.toString())
        }

        fun json(obj: Any?) {
            if (logger == null) {
                logger = getLoggerImp()
            }
            logger?.json(obj)
        }

        fun json(tag: String, obj: Any?) {
            if (logger == null) {
                logger = getLoggerImp()
            }
            logger?.json(contact(tag), obj)
        }

        fun xml(msg: String) {
            if (logger == null) {
                logger = getLoggerImp()
            }
            logger?.xml(msg)
        }

        fun xml(tag: String, msg: String) {
            if (logger == null) {
                logger = getLoggerImp()
            }
            logger?.xml(contact(tag), msg)
        }

        /**
         * 文件输出，默认使用XLOG来实现
         */
        fun file(tag: String, msg: String) {
            XLog.tag(contact(tag)).i(msg)
        }
    }
}