package com.tn.lib.logger.xlog

import android.app.Application
import android.os.Environment
import android.util.Log
import com.elvishew.xlog.LogConfiguration
import com.elvishew.xlog.LogLevel
import com.elvishew.xlog.XLog
import com.elvishew.xlog.flattener.ClassicFlattener
import com.elvishew.xlog.formatter.border.DefaultBorderFormatter
import com.elvishew.xlog.formatter.message.json.DefaultJsonFormatter
import com.elvishew.xlog.formatter.message.throwable.DefaultThrowableFormatter
import com.elvishew.xlog.formatter.message.xml.DefaultXmlFormatter
import com.elvishew.xlog.formatter.stacktrace.DefaultStackTraceFormatter
import com.elvishew.xlog.printer.AndroidPrinter
import com.elvishew.xlog.printer.file.FilePrinter
import com.elvishew.xlog.printer.file.backup.NeverBackupStrategy
import com.tn.lib.logger.xlog.conf.CusDateFileNameGenerator
import com.tn.lib.logger.xlog.conf.CusThreadFormatter
import com.tn.lib.logger.xlog.conf.FileSizeCleanStrategy
import com.tn.lib.logger.xlog.conf.LogConf
import java.io.File

class LoggerApplication {
    companion object {
        /**
         * adb shell setprop log.tag.oneroom_logger D
         */
        private const val LOGGER_ABLE = "oneroom_logger"

        var application: Application? = null

        /**
         * 是否是调试模式
         */
        private var isDebug: Boolean = false
        private var logType: LogType = LogType.TYPE_LOGCAT

        fun iSDebug(): Boolean {
            return isDebug
        }

        fun getLogType(): LogType {
            return logType
        }

        fun getDefaultLogPath(application: Application?): String {
            if (application == null) {
                throw RuntimeException("日志库传入Application为空")
            } else {
                return if (isExternalStorageWritable()) {
                    application.getExternalFilesDir(null)?.parent + File.separator + "log"
                } else {
                    application.filesDir?.parent + File.separator + "log"
                }
            }
        }

        private fun isExternalStorageWritable(): Boolean {
            return Environment.getExternalStorageState() == Environment.MEDIA_MOUNTED
        }
        /**
         * @param loggerPath: 日志文件存储路径信息
         * @param autoCleanTime: 日志文件清除时间，时间单位为毫秒
         * @param isDebug: 是否是调试模式，调试模式是全部日志输出，Release输出模式为INFO以上
         */
        fun initLog(
            application: Application,
            logType: LogType,
            isDebug: Boolean,
            loggerPath: String = "",
            autoCleanTime: Long = 0
        ) {
            if (Companion.application != null) {
                return
            }

            Companion.logType = logType
            Companion.isDebug = isDebug || Log.isLoggable(LOGGER_ABLE, Log.DEBUG)
            Companion.application = application

            when (logType) {
                LogType.TYPE_LOGCAT -> {
                    initLogCat(loggerPath, autoCleanTime, isDebug)
                }
                LogType.TYPE_XLOG -> {
                    initXLog(loggerPath, autoCleanTime, isDebug)
                }
            }
        }

        private fun initLogCat(loggerPath: String, autoCleanTime: Long, isDebug: Boolean) {
            initXLog(loggerPath, autoCleanTime, isDebug)
        }

        private fun initXLog(loggerPath: String, autoCleanTime: Long, isDebug: Boolean) {
            //自动清理时间
            var realCleanTime = LogConf.DEFAULT_AUTO_CLEAN_TIME
            if (autoCleanTime > 0L) {
                realCleanTime = autoCleanTime
            }

            //日志存储路径信息
            var realLoggerPath = getDefaultLogPath(application)
            if (loggerPath.isNotEmpty()) {
                realLoggerPath = loggerPath
            }

            val config = LogConfiguration.Builder()
                .logLevel(/*if (isDebug) LogLevel.ALL else LogLevel.INFO*/LogLevel.ALL)      // 指定日志级别，低于该级别的日志将不会被打印，默认为 LogLevel.ALL
                .tag("X-LOG")                                         // 指定 TAG，默认为 "X-LOG"
//                    .t()                                                   // 允许打印线程信息，默认禁止
//                    .st(0)                                                 // 允许打印深度为2的调用栈信息，默认禁止
                //.b()                                                   // 允许打印日志边框，默认禁止
                .jsonFormatter(DefaultJsonFormatter())
                .xmlFormatter(DefaultXmlFormatter())
                .throwableFormatter(DefaultThrowableFormatter())
                .threadFormatter(CusThreadFormatter())
                .stackTraceFormatter(DefaultStackTraceFormatter())
                .borderFormatter(DefaultBorderFormatter())
                .addInterceptor { return@addInterceptor it }
                //.enableThreadInfo()
                .build()

            val androidPrinter = AndroidPrinter()             // 通过 android.util.Log 打印日志的打印器
//            val consolePrinter = ConsolePrinter()             // 通过 System.out 打印日志到控制台的打印器

            var filePrinter: FilePrinter? = null
            if (realLoggerPath.isNotEmpty()) {
                filePrinter = FilePrinter.Builder(realLoggerPath)
                    .fileNameGenerator(CusDateFileNameGenerator())
                    .backupStrategy(NeverBackupStrategy())
                    .cleanStrategy(FileSizeCleanStrategy())
                    .flattener(ClassicFlattener())
                    .build()
            }

            if (isDebug) {
                //Debug模式下输出配置
                if (filePrinter == null) {
                    XLog.init(config, androidPrinter)
                } else {
                    XLog.init(config, androidPrinter, filePrinter)
                }
            } else {
                //Release输出配置
                if (filePrinter == null) {
                    XLog.init(config)
                } else {
                    XLog.init(config, filePrinter)
                }
            }
        }
    }
}