package com.transsion.base.report.athena

import android.app.Application
import android.os.Bundle
import android.os.SystemClock
import android.text.TextUtils
import com.blankj.utilcode.util.Utils
import com.transsion.athenacust.AthenaCust
import com.tn.lib.logger.Logger
import com.transsion.base.report.BuildConfig
import com.transsion.ga.AthenaAnalytics
import org.json.JSONObject
import java.util.*
import java.util.concurrent.Executors


/**
 *   Transsion MI
 *   Created By Liupeng
 *   On 2022/8/11 13:51
 *
 *   Desc 上报工具类
 *   业务埋点文档
 *   埋点sdk文档 https://transsioner.feishu.cn/wiki/wikcnOYd5HX7w30uhwNSk71Q6wd#
 *
 */
object Report {
    const val TAG = "Report"

    private const val EXT = "ext"

    /**
     * 上一个页面的名称
     */
    private var mLastPageName: String = ""
    private var api: ReportApi? = null
    private val singleExecutor =  Executors.newSingleThreadExecutor()

    fun initSDK(context: Application, debug: Boolean, api: ReportApi, channelId: String, appId: Int) {
        AthenaAnalytics.setWorkMode(AthenaAnalytics.MODE_NORMAL)
        /**
         * @param context
         * @param channel
         * @param appId
         * @param isDebug 是否打印日志
         * @param isHost 是否宿主
         */
        AthenaAnalytics.init(context, channelId, appId, true, true)
        // 数据上报到正式 or 测试环境
        AthenaAnalytics.setTest(debug)
        // 是否启用Athena
        AthenaAnalytics.enable(true)
        AthenaAnalytics.setDelayTime(2000)
        AthenaAnalytics.submitAll()
        Report.api = api
        Logger.d(TAG, "initSDK debug $debug")
    }

    /**
     * 页面曝光事件
     * @param logConfig ILogViewConfig 页面信息
     */
    fun reportPV(logConfig: LogViewConfig) {
        logConfig.paramsMap["page_from"] = logConfig.fromPageName ?: mLastPageName
        logConfig.ops?.let {
            logConfig.paramsMap["ops"] = it
        }

        reportUIEvent(
            logConfig.pageName,
            ReportConstants.PV,
            logConfig.paramsMap
        )
    }

    /**
     * 页面曝光事件
     * @param logConfig ILogViewConfig 页面信息
     */
    fun reportPT(logConfig: LogViewConfig) {
        logConfig.paramsMap["page_from"] = logConfig.fromPageName ?: mLastPageName
        logConfig.paramsMap["duration"] =
            (SystemClock.elapsedRealtime() - logConfig.resumeTime).toString()
        logConfig.paramsMap["is_load_success"] = logConfig.isLoadSuccess.toString()
        logConfig.ops?.let {
            logConfig.paramsMap["ops"] = it
        }

        reportUIEvent(
            logConfig.pageName,
            ReportConstants.PT,
            logConfig.paramsMap
        )
        if (!TextUtils.equals(mLastPageName, logConfig.pageName) && logConfig.needAddToPageFrom) {
            mLastPageName = logConfig.pageName
        }
    }

    /**
     * Dialog弹框
     * @param pageName String 页面名称
     * @param map Map<String, String> 其他参数
     */
    fun reportDialog(pageName: String, map: Map<String, String?>) {
        reportUIEvent(pageName, ReportConstants.DIALOG_SHOW, map)
    }


    /**
     * 曝光/展现类事件
     * @param pageName String 页面名称
     * @param event String  事件名称
     * @param map Map<String, String> 其他参数
     */
    fun reportShow(pageName: String, map: Map<String, String?>) {
        reportUIEvent(pageName, ReportConstants.BROWSE, map)
    }


    /**
     * 点击/交互 类事件
     * @param pageName String 页面名称
     * @param event String  事件名称
     * @param map Map<String, String> 其他参数
     */
    fun reportClick(pageName: String, map: Map<String, String?>) {
        reportUIEvent(pageName, ReportConstants.CLICK, map)
    }

    fun reportDelete(pageName: String, map: Map<String, String?>) {
        reportUIEvent(pageName, ReportConstants.DELETE, map)
    }

    /**
     * 时长 类事件
     * @param pageName String 页面名称
     * @param event String  事件名称
     * @param map Map<String, String> 其他参数
     */
    fun reportUseTime(
        pageName: String,
        event: String,
        useTime: Long,
        map: MutableMap<String, String>,
    ) {
        map["duration"] = useTime.toString()
        reportUIEvent(pageName, event, map)
    }

    fun reportEvent(
        pageName: String,
        event: String,
        map: Map<String, String?>,
    ) {
        reportUIEvent(pageName, event, map)
    }

    fun reportUIEvent(
        pageName: String,
        event: String,
        map: Map<String, String?>,
    ) {
        singleExecutor.execute {
            Logger.d(TAG, "event=$event  pageName=$pageName map=$map")
            val bundle = Bundle()
            bundle.putString("page_name", pageName)
            //bundle.putString("event", event)
            val map2Jsonstr = map2Jsonstr(map)
            bundle.putString(EXT, map2Jsonstr)
            track(event, bundle)
            addToRecentEvent(pageName,event,map)
        }
    }


    private fun addToRecentEvent(pageName: String,event: String, map: Map<String, String?>) {
        api?.addToRecentEvent(pageName, event, map)
    }

    /**
     *  启动等无页面相关的事件
     * @param event String  事件名称
     * @param map Map<String, String> 其他参数
     */
    fun reportEvent(
        event: String,
        map: Map<String, String?>,
    ) {
        singleExecutor.execute {
            Logger.d(TAG, "reportEvent event=$event map=$map ")
            val bundle = Bundle()
            //bundle.putString("event", event)
            val map2Jsonstr = map2Jsonstr(map)
            bundle.putString(EXT,  map2Jsonstr)

            track(event, bundle)
        }
    }

    /**
     * 登录激活
     */
    fun loginActive(userType: Int, userid: String?) {
        singleExecutor.execute {
            Logger.d(TAG, "loginActive:userType$userType,userid:$userid")
            val event = ReportConstants.LOGIN_ACTIVE
            val map = HashMap<String, String>()
            map["userType"] = userType.toString()
            map["userId"] = userid ?: "0"
            val json = map2Jsonstr(map)
            val bundle = Bundle()
            //bundle.putString("pageName", pageName)
            bundle.putString("event", event)
            bundle.putString(EXT, json)

            track(event, bundle)
        }
    }

    /**
     * 最底层都会调用这个方法.
     */
    private fun track(event: String, bundle: Bundle) {
        if (BuildConfig.DEBUG) {
            val deviceId = api?.addCommonParams(event, bundle)?.getString("device_id")
            if (deviceId.isNullOrEmpty()) {
                throw Exception("no athena common params")
            }

            if (api == null) {
                throw Exception("athena api is null")
            }
        }

        try {
            bundle.putString("vaid", AthenaAnalytics.getAppVAID(Utils.getApp()))
            val commonBundle = api?.addCommonParams(event, bundle)
            // 更新 Athena 用户数据
            api?.getAccount()?.apply {
                AthenaAnalytics.setAccount(this.first, this.second)
            }
            AthenaCust(event).trackCommon(commonBundle, null).submit()
        } catch (ignore: Exception) {
            Logger.e("report", "event $event exception $ignore", true)
        }
    }

    fun getLastPageName(): String {
        return mLastPageName
    }

    fun map2Jsonstr(map: Map<String, String?>): String {
        return JSONObject(map).toString()
    }
}
